/*
 * sys_panic.c - kill calling process and display a debug message on kernel terminal.
 * 
 * Author    Alain Greiner (2016,2017)
 *  
 * Copyright (c) UPMC Sorbonne Universites
 *
 * This file is part of ALMOS-MKH.
 *
 * ALMOS-MKH is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2.0 of the License.
 *
 * ALMOS-MKH is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with ALMOS-MKH; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */

#include <hal_types.h>
#include <hal_uspace.h>
#include <core.h>
#include <thread.h>
#include <process.h>
#include <vmm.h>
#include <printk.h>

//////////////////////////////
int sys_panic( char * string )
{
	error_t   error;
    paddr_t   paddr;

    char      kbuf[256];

    thread_t  * this    = CURRENT_THREAD;
    process_t * process = this->process;
    core_t    * core    = this->core;

    // check string in user space
    error = vmm_v2p_translate( false , string , &paddr );

	if( error )
	{
        printk("\n[USER DBG] thread %x / process %x/ core[%x,%d] / cycle %d\n"
        "cause unknown, because string not in user space\n",
        this->trdid , process->pid , local_cxy , core->lid ,
        (uint32_t)hal_get_cycles() );
	}

    // ckeck string length
    if( hal_strlen_from_uspace( string ) >= 256 )
	{
        printk("\n[USER DBG] thread %x / process %x/ core[%x,%d] / cycle %d\n"
        "cause unknown, because string larger than 256 characters\n",
        this->trdid , process->pid , local_cxy , core->lid ,
        (uint32_t)hal_get_cycles() );
	}

    // copy string in kernel space
    hal_strcpy_from_uspace( kbuf , string , 256 );

    // print user debug message on kernel terminal
    printk("\n[USER DBG] thread %x / process %x / core[%x,%d] / cycle %d\n %s\n",
    this->trdid , process->pid , local_cxy, core->lid ,
    (uint32_t)hal_get_cycles() , kbuf );

	return 0; 

}  // end sys_get_cycle()
