/*
 * stdio.h - User side syscalls definition.
 * 
 * Author     Alain Greiner (2016,2017)
 *
 * Copyright (c) UPMC Sorbonne Universites
 *
 * This file is part of ALMOS-MKH.
 *
 * ALMOS-MKH is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2.0 of the License.
 *
 * ALMOS-MKH is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with ALMOS-MKH; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */

#ifndef _STDIO_H_
#define _STDIO_H_

#include <shared_syscalls.h>

#define  NULL  (void *)0

/***************************************************************************************** 
 * This function terminates a process.
 *****************************************************************************************
 * @ status   : terminaison status : 0 / EXIT_SUCCESS / EXIT_FAILURE.
 ****************************************************************************************/
void exit( int status );

/***************************************************************************************** 
 * This function open or create an open file descriptor.
 *****************************************************************************************
 * @ pathname   : pathname (can be relative or absolute).
 * @ flags      : bit vector attributes (see syscalls).
 * @ mode       : access rights (if O_CREAT is set).
 * @ return file descriptor index in fd_array if success / return -1 if failure.
 ****************************************************************************************/
int open( const char * pathname,
          int          flags,
          int          mode );

/***************************************************************************************** 
 * This function map physical memory (or a file) in the calling thread virtual space.
 *****************************************************************************************
 * @ addr       : unused and unsupported : must be NULL.
 * @ length     : requested number of bytes.
 * @ prot       : RWX access modes.
 * @ flags      : MAP_FILE / MAP_ANON / MAP_PRIVATE / MAP_SHARED (defined in syscalls.h)  
 * @ fdid       : file descriptor index (if MAP_FILE).                     
 * @ offset	    : offset in file (if MAP_FILE)
 * @ return 0 if success / return -1 if failure.
 ****************************************************************************************/
void * mmap( void         * addr,
             unsigned int   length,
             int            prot,
             int            flags,
             int            fd,
             unsigned int   offset );

/***************************************************************************************** 
 * This function read bytes from an open file identified by its file descriptor.
 * This file can be a regular file or a character oriented device.
 *****************************************************************************************
 * @ file_id  : open file index in fd_array.
 * @ buf      : buffer virtual address in user space.
 * @ count    : number of bytes.
 * @ return number of bytes actually read if success / returns -1 if failure.
 ****************************************************************************************/
int read( int            fd,
          void         * buf,
          unsigned int   count );

/***************************************************************************************** 
 * This function writes bytes to an open file identified by its file descriptor.
 * This file can be a regular file or character oriented device.
 *****************************************************************************************
 * @ file_id  : open file index in fd_array.
 * @ buf      : buffer virtual address in user space.
 * @ count    : number of bytes.
 * @ return number of bytes actually written if success / returns -1 if failure.
 ****************************************************************************************/
int write( int            fd,
           const void   * buf,
           unsigned int   count );

/***************************************************************************************** 
 * This function repositions the offset of the file descriptor identified by <file_id>,
 * according to the operation type defined by the <whence> argument.
 *****************************************************************************************
 * @ fd       : open file index in fd_array.
 * @ offset   : used to compute new offset value.
 * @ whence   : operation type (SEEK_SET / SEEK_CUR / SEEK_END defined in syscalls.h)
 * @ return 0 if success / returns -1 if failure.
 ****************************************************************************************/
int lseek( int           fd,
           unsigned int  offset,
           int           whence );

/***************************************************************************************** 
 * This function release the memory allocated for the file descriptor identified by 
 * the <file_id> argument, and remove the fd array_entry in all copies of the process 
 * descriptor.
 *****************************************************************************************
 * fd   : file descriptor index in fd_array.
 * @ return 0 if success / returns -1 if failure.
 ****************************************************************************************/
int close( int fd );

/***************************************************************************************** 
 * This function removes a directory entry identified by the <pathname> from the
 * directory, and decrement the link count of the file referenced by the link.
 * If the link count reduces to zero, and no process has the file open, then all resources
 * associated with the file are released.  If one or more process have the file open when
 * the last link is removed, the link is removed, but the removal of the file is delayed 
 * until all references to it have been closed.
 *****************************************************************************************
 * @ pathname   : pathname (can be relative or absolute).
 * @ return 0 if success / returns -1 if failure.
 ****************************************************************************************/
int unlink( const char * pathname );

/***************************************************************************************** 
 * This function creates in the calling thread cluster an unnamed pipe, and two
 * (read and write) file descriptors to access this pipe. The calling function must pass
 * the pointer on the fd[] array.
 * TODO not implemented yet...
 *****************************************************************************************
 * @ file_id[0] : [out] read only file descriptor index.
 * @ file_id[1] : [out] write only file descriptor index.
 * @ return 0 if success / return -1 if failure.
 ****************************************************************************************/
int pipe( int fd[2] );

/***************************************************************************************** 
 * This function change the current working directory in reference process descriptor.
 *****************************************************************************************
 * @ pathname   : pathname (can be relative or absolute).
 * @ return 0 if success / returns -1 if failure.
 ****************************************************************************************/
int chdir( const char * pathname );

/***************************************************************************************** 
 * This function creates a new directory in file system.
 *****************************************************************************************
 * @ pathname   : pathname (can be relative or absolute).
 * @ mode       : access rights (as defined in chmod).
 * @ return 0 if success / returns -1 if failure.
 ****************************************************************************************/
int mkdir( const char * pathname,
           int          mode );

/***************************************************************************************** 
 * This function creates a named FIFO file in the calling thread cluster.
 * The associated read and write file descriptors mut be be  explicitely created
 * using the open() system call.
 *****************************************************************************************
 * @ pathname   : pathname (can be relative or absolute).
 * @ mode       : access rights (as defined in chmod).
 * @ return 0 if success / returns -1 if failure.
 ****************************************************************************************/
int mkfifo( const char * pathname,
            int          mode );

/***************************************************************************************** 
 * This function opens the directory identified by the <pathname> argument, 
 * associates a directory stream with it and returns an user space pointer to identify 
 * this directory stream in subsequent operations.  
 *****************************************************************************************
 * @ pathname   : pathname (can be relative or absolute).
 * @ returns DIR* pointer if success / returns NULL if pathname cannot be accessed.
 ****************************************************************************************/
DIR * opendir( const char * pathname );

/***************************************************************************************** 
 * This function returns a pointer to the next directory entry. 
 *****************************************************************************************
 * @ dirp     : DIR pointer identifying the directory. 
 * @ returns dirent* pointer / returns NULL upon reaching end of directory or on error.
 ****************************************************************************************/
struct dirent * readdir( DIR * dirp );

/***************************************************************************************** 
 * This function closes the directory identified by the <dirp> argument, and releases
 * all structures associated with the <dirp> pointer.
 *****************************************************************************************
 * @ dirp     : DIR pointer identifying the directory. 
 * @ returns 0 if success / returns -1 if failure.
 ****************************************************************************************/
int closedir( DIR * dirp );

/***************************************************************************************** 
 * This function returns the pathname of the current working directory.
 *****************************************************************************************
 * buf     : buffer addres in user space.
 * nbytes  : user buffer size in bytes.
 * @ return 0 if success / returns -1 if failure.
 ****************************************************************************************/
int getcwd( char       * buf,
            unsigned int nbytes );

/***************************************************************************************** 
 * This function removes a directory file whose name is given by <pathname>.
 * The directory must not have any entries other than `.' and `..'.
 *****************************************************************************************
 * @ pathname   : pathname (can be relative or absolute).
 * @ return 0 if success / returns -1 if failure.
 ****************************************************************************************/
int rmdir( char * pathname ); 

/***************************************************************************************** 
 * This function implement the operations related to User Thread Local Storage.
 *****************************************************************************************
 * @ operation  : UTLS operation type as defined in "shared_sycalls.h" file.
 * @ value      : argument value for the UTLS_SET operation.
 * @ return value for the UTLS_GET and UTLS_GET_ERRNO / return -1 if failure.
 ****************************************************************************************/
int utls( unsigned int operation,
          unsigned int value );

/***************************************************************************************** 
 * This function change the acces rights for the file/dir identified by the
 * pathname argument.
 *****************************************************************************************
 * @ pathname   : pathname (can be relative or absolute).
 * @ rights     : acces rights.
 * @ return 0 if success / returns -1 if failure.
 ****************************************************************************************/
int chmod( char     * pathname,
           unsigned int   rights );

/***************************************************************************************** 
 * This function associate a specific signal handler to a given signal type.
 * The handlers for the SIGKILL and SIGSTOP signals cannot be redefined.
 ***************************************************************************************** 
 * @ sig_id    : index defining signal type (from 1 to 31).
 * @ handler   : pointer on fonction implementing the specific handler.
 * @ return 0 if success / returns -1 if failure.
 ****************************************************************************************/
int signal( unsigned int   sig_id,
            void         * handler );

/***************************************************************************************** 
 * This function returns in the structure <tv>, defined in the time.h file,
 * the current time (in seconds & micro-seconds).
 * It is computed from the calling core descriptor.
 * The timezone is not supported.
 *****************************************************************************************
 * @ tv      : pointer on the timeval structure.
 * @ tz      : pointer on the timezone structure : must be NULL.       
 * @ return 0 if success / returns -1 if failure.
 ****************************************************************************************/
int gettimeofday( struct timeval  * tv,
                  struct timezone * tz );

/***************************************************************************************** 
 * This function implements the POSIX "kill" system call.
 * It register the signal defined by the <sig_id> argument in all thread descriptors 
 * of a target process identified by the <pid> argument. This is done in all clusters
 * containing threads for the target process.
 ***************************************************************************************** 
 * @ pid      : target process identifier.
 * @ sig_id   : index defining the signal type (from 1 to 31). 
 * @ return 0 if success / returns -1 if failure.
 ****************************************************************************************/
int kill( unsigned int  pid,
          unsigned int  sig_id );

/***************************************************************************************** 
 * This function implements the POSIX "getpid" system call.
 ***************************************************************************************** 
 * @ returns the process PID for the calling thread.
 ****************************************************************************************/
int getpid();

/***************************************************************************************** 
 * This function implement the POSIX "fork" system call.
 * The calling process descriptor (parent process), and the associated thread descriptor
 * are replicated in the same cluster as the calling thread, but the new process (child 
 * process) is registered in another target cluster, that is the new process owner. 
 * The child process and the associated main thread will be migrated to the target cluster
 * later, when the child process makes an "exec" or any other system call. 
 * The target cluster depends on the "fork_user" flag and "fork_cxy" variable that can be
 * stored in the calling thread descriptor by the specific fork_place() system call.
 * If not, the kernel function makes a query to the DQDT to select the target cluster. 
 ***************************************************************************************** 
 * @ returns child process PID if success / returns -1 if failure
 ****************************************************************************************/
int fork();

/***************************************************************************************** 
 * This function implement the "exec" system call.
 * It is executed in the client cluster, but the new process descriptor and main thread
 * must be created in a server cluster, that is generally another cluster. 
 * - if the server_cluster is the client cluster, call directly the process_make_exec()
 *   function to create a new process, and launch a new thread in local cluster.
 * - if the target_cluster is remote, call rpc_process_exec_client() to execute the
 *   process_make_exec() on the remote cluster.
 * In both case this function build an exec_info_t structure containing all informations
 * required to build the new process descriptor and the associated thread.
 * Finally, the calling process and thread are deleted.
 ***************************************************************************************** 
 * @ filename : string pointer on .elf filename (virtual pointer in user space)
 * @ argv     : array of strings on process arguments (virtual pointers in user space)
 * @ envp     : array of strings on environment variables (virtual pointers in user space)
 * @ returns O if success / returns -1 if failure.
 ****************************************************************************************/
int exec( char  * filename,
          char ** argv,
          char ** envp );

/***************************************************************************************** 
 * This function  returns in the <stat> structure, defined in the "shared_syscalls.h" 
 * file, various informations on the file/directory identified by the <pathname> argument.
 ***************************************************************************************** 
 * @ pathname  : user pointer on file pathname.
 * @ stat      : user pointer on the stat structure.
 * @ returns O if success / returns -1 if failure.
 ****************************************************************************************/
int stat( const char  * pathname,
          struct stat * stat );

/***************************************************************************************** 
 * This function is used to activate / desactivate the trace for a thread 
 * identified by the <trdid> and <pid> arguments.
 * It can be called by any other thread in the same process.
 ***************************************************************************************** 
 * @ operation  : operation type.
 * @ pid        : process identifier.
 * @ trdid      : thread identifier.
 * @ returns O if success / returns -1 if failure.
 ****************************************************************************************/
int trace( unsigned int operation,
           unsigned int pid, 
           unsigned int trdid );


#endif  // _STDIO_H_
