/*
 * sys_munmap.c - unmap a mapping from process virtual address space
 * 
 * Authors       Ghassan Almaless (2008,2009,2010,2011,2012)
 *               Alain Greiner (2016,2017)
 *
 * Copyright (c) UPMC Sorbonne Universites
 *
 * This file is part of ALMOS-MKH.
 *
 * ALMOS-MKH is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2.0 of the License.
 *
 * ALMOS-MKH is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with ALMOS-MKH; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */

#include <hal_types.h>
#include <hal_uspace.h>
#include <shared_syscalls.h>
#include <errno.h>
#include <thread.h>
#include <printk.h>
#include <mapper.h>
#include <vfs.h>
#include <process.h>
#include <vmm.h>

////////////////////////////////
int sys_munmap( void     * vaddr,
                uint32_t   size )
{
    error_t       error;

	uint32_t      tm_start;
	uint32_t      tm_end;

	tm_start = hal_get_cycles();

	thread_t    * this    = CURRENT_THREAD;
	process_t   * process = this->process;

    // call relevant kernel function
    error = vmm_resize_vseg( process , (intptr_t)vaddr , (intptr_t)size );

    if ( error )
    {
        printk("\n[ERROR] in %s : cannot remove mapping\n", __FUNCTION__ );
		this->errno = EINVAL;
		return -1;
    }

    tm_end = hal_get_cycles();

syscall_dmsg("\n[DBG] %s : core[%x,%d] removed vseg in process %x / cycle %d\n"
"      base = %x / size = %x / cost = %d\n",
__FUNCTION__, local_cxy , this->core->lid , process->pid , tm_start ,
vaddr , size , tm_end - tm_start );

        return 0;

}  // end sys_mmap()

