/*
 * sys_thread_join.c - passive wait on the end of a given thread.
 * 
 * Authors    Alain Greiner (2016,2017)
 *
 * Copyright (c) 2011,2012 UPMC Sorbonne Universites
 *
 * This file is part of ALMOS-MKH.
 *
 * ALMOS-MKH is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2.0 of the License.
 *
 * ALMOS-MKH is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with ALMOS-MKH; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */

#include <hal_types.h>
#include <hal_remote.h>
#include <hal_special.h>
#include <thread.h>
#include <vmm.h>
#include <scheduler.h>
#include <errno.h>
#include <printk.h>
#include <remote_spinlock.h>

///////////////////////////////////////
int sys_thread_join ( trdid_t    trdid,
                      void    ** exit_value )
{
    xptr_t        target_xp;
    thread_t    * target_ptr;
    cxy_t         target_cxy;
    ltid_t        target_ltid;
	uint32_t      target_blocked;   // target thread blocked bit-vector
    uint32_t      target_flags;     // target thread flags bit-bector
    paddr_t       paddr;            // required for vmm_v2p_translate()

	thread_t    * this    = CURRENT_THREAD;
    process_t   * process = this->process;

    // get target thread ltid and cxy
    target_ltid = LTID_FROM_TRDID( trdid );
    target_cxy  = CXY_FROM_TRDID( trdid );

    // check trdid argument
	if( (target_ltid >= CONFIG_THREAD_MAX_PER_CLUSTER) || cluster_is_undefined( target_cxy ) )  
	{
        printk("\n[ERROR] in %s : illegal trdid argument\n", __FUNCTION__ );
		this->errno = EINVAL;
		return -1;
	}

    // check exit_value argument
	if( (exit_value != NULL) && (vmm_v2p_translate( false , exit_value , &paddr ) != 0 ) )
	{
        printk("\n[ERROR] in %s : illegal exit_value argument\n", __FUNCTION__ );
		this->errno = EINVAL;
		return -1;
	}

    // check target thread != this thread
    if( this->trdid == trdid )
    {
        printk("\n[ERROR] in %s : this thread == target thread\n", __FUNCTION__ );
        this->errno = EDEADLK;
        return -1;
    }

    // get extended pointer on target thread
	target_xp  = thread_get_xptr( process->pid , trdid );

    if( target_xp == XPTR_NULL )
    {
        printk("\n[ERROR] in %s : target thread not found\n", __FUNCTION__ );
        this->errno = ESRCH;
        return -1;
    }

    // get cluster and local pointer on target thread
    target_ptr = (thread_t *)GET_PTR( target_xp );

    // check target thread joinable
    target_flags = hal_remote_lw( XPTR( target_cxy , &target_ptr->flags ) );
    if( target_flags & THREAD_FLAG_DETACHED )
    {
        printk("\n[ERROR] in %s : target thread not joinable\n", __FUNCTION__ );
        this->errno = EINVAL; 
        return -1;
    }

    // check kernel stack overflow
    if( target_ptr->signature != THREAD_SIGNATURE )
    {
        panic("\n[PANIC] in %s : kernel stack overflow\n", __FUNCTION__ );
    }

    // get the lock protecting the join in target thread
    remote_spinlock_lock( XPTR( target_cxy , &target_ptr->join_lock ) );

    // get the blocked bit_vector from the target thread
    target_blocked = hal_remote_lw( XPTR( target_cxy , &target_ptr->blocked ) );

    if( target_blocked & THREAD_BLOCKED_JOIN )    // target thread arrived first
    {
        // unblock the target thread
        thread_unblock( target_xp , THREAD_BLOCKED_JOIN );

        // release the lock protecting flags      
        remote_spinlock_unlock( XPTR( target_cxy , &target_ptr->join_lock ) );

        // get the exit value from target thread
        *exit_value = hal_remote_lpt( XPTR( target_cxy , &target_ptr->join_value ) );
    }
    else                                          // this thread arrived first
    {
        // register this thread extended pointer in target thread
        hal_remote_swd( XPTR( target_cxy , &target_ptr->join_xp ) , 
                              XPTR( local_cxy , this ) );

        // set the JOIN_DONE flag in target thread
        hal_remote_atomic_or( XPTR( target_cxy , &target_ptr->flags ) ,
                              THREAD_FLAG_JOIN_DONE );

        // block this thread on BLOCKED_EXIT
        thread_block( this , THREAD_BLOCKED_EXIT ); 

        // release the lock protecting flags      
        remote_spinlock_unlock( XPTR( target_cxy , &target_ptr->join_lock ) );

        // deschedule
        sched_yield( "WAITING_EXIT" );
    
        // get the exit value from target thread when resume
        *exit_value = hal_remote_lpt( XPTR( target_cxy , &target_ptr->join_value ) );
    }

    return 0;

}  // end sys_thread_join()
