/*
 * syscalls.h - Shared Kernel/User informations for syscalls.
 * 
 * Author     Alain Greiner (2016,2017)
 *
 * Copyright (c) UPMC Sorbonne Universites
 *
 * This file is part of ALMOS-MKH.
 *
 * ALMOS-MKH is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2.0 of the License.
 *
 * ALMOS-MKH is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with ALMOS-MKH; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */

#ifndef _SHARED_SYSCALLS_H_
#define _SHARED_SYSCALLS_H_

/******************************************************************************************
 * This enum defines the mnemonics for the syscall indexes.
 * It must be kept consistent with the array defined in do_syscalls.c
 *****************************************************************************************/

enum
{
	SYS_THREAD_EXIT    = 0,
	SYS_THREAD_YIELD   = 1,
	SYS_THREAD_CREATE  = 2,
	SYS_THREAD_JOIN    = 3,
	SYS_THREAD_DETACH  = 4,
	SYS_THREAD_CANCEL  = 5,
	SYS_SEM            = 6,
	SYS_CONDVAR        = 7,
	SYS_BARRIER        = 8,
	SYS_MUTEX          = 9,

    SYS_EXIT           = 10,
    SYS_MUNMAP         = 11,
	SYS_OPEN           = 12,
	SYS_MMAP           = 13,
	SYS_READ           = 14,
	SYS_WRITE          = 15,
	SYS_LSEEK          = 16,
	SYS_CLOSE          = 17,
	SYS_UNLINK         = 18,   
	SYS_PIPE           = 19,

	SYS_CHDIR          = 20,
	SYS_MKDIR          = 21,
	SYS_MKFIFO         = 22,    
	SYS_OPENDIR        = 23,
	SYS_READDIR        = 24,
	SYS_CLOSEDIR       = 25,
	SYS_GETCWD         = 26,
	SYS_UNDEFINED_27   = 27,   ///
	SYS_ALARM          = 28,   
	SYS_RMDIR          = 29,

	SYS_UTLS           = 30,  
	SYS_CHMOD          = 31,
	SYS_SIGNAL         = 32,
	SYS_TIMEOFDAY      = 33,
	SYS_KILL           = 34,
	SYS_GETPID         = 35,
	SYS_FORK           = 36,
	SYS_EXEC           = 37,
	SYS_STAT           = 38,     
	SYS_TRACE          = 39,

    SYS_GET_CONFIG     = 40,
    SYS_GET_CORE       = 41,
    SYS_GET_CYCLE      = 42,
    SYS_GET_SCHED      = 43,
    SYS_PANIC          = 44,
	SYS_SLEEP          = 45,
	SYS_WAKEUP         = 46,

	SYSCALLS_NR        = 47,
};

/******************************************************************************************* 
 * This defines the signal type mnemonics for the kill() syscall.
 * WARNING : Only the three SIGKILL / SIGSTOP / SIGCONT are supported (december 2017)
 ******************************************************************************************/

#define SIGHUP     1       /*! hangup                                                     */
#define SIGINT     2       /*! interrupt                                                  */
#define SIGQUIT    3       /*! quit                                                       */
#define SIGILL     4       /*! illegal instruction (not reset when caught)                */
#define SIGTRAP    5       /*! trace trap (not reset when caught)                         */
#define SIGABRT    6       /*! used by abort, replace SIGIOT in the future                */
#define SIGEMT     7       /*! EMT instruction                                            */
#define SIGFPE     8       /*! floating point exception                                   */
#define SIGKILL    9       /*! kill (cannot be caught or ignored)                         */
#define SIGBUS     10      /*! bus error                                                  */
#define SIGSEGV    11      /*! segmentation violation                                     */
#define SIGSYS     12      /*! bad argument to system call                                */
#define SIGPIPE    13      /*! write on a pipe with no one to read it                     */
#define SIGALRM    14      /*! alarm clock                                                */
#define SIGTERM    15      /*! software termination signal from kill                      */
#define SIGURG     16      /*! urgent condition on IO channel                             */
#define SIGSTOP    17      /*! sendable stop signal not from tty                          */
#define SIGTSTP    18      /*! stop signal from tty                                       */
#define SIGCONT    19      /*! continue a stopped process                                 */
#define SIGCHLD    20      /*! to parent on child stop or exit                            */
#define SIGTTIN    21      /*! to readers pgrp upon background tty read                   */
#define SIGTTOU    22      /*! like TTIN for output if (tp->t_local&LTOSTOP)              */
#define SIGIO      23      /*! input/output possible signal                               */
#define SIGXCPU    24      /*! exceeded CPU time limit                                    */
#define SIGXFSZ    25      /*! exceeded file size limit                                   */
#define SIGVTALRM  26      /*! virtual time alarm                                         */
#define SIGPROF    27      /*! profiling time alarm                                       */
#define SIGWINCH   28      /*! window changed                                             */
#define SIGLOST    29      /*! resource lost (eg, record-lock lost)                       */
#define SIGUSR1    30      /*! user defined signal 1                                      */
#define SIGUSR2    31      /*! user defined signal 2                                      */

/******************************************************************************************* 
 * This enum defines the supported terminaison status values for the exit() syscall.
 ******************************************************************************************/

typedef enum
{
	EXIT_SUCCESS,
	EXIT_FAILURE,
} 
exit_statut_t;

/*******************************************************************************************
 * These typedef define the POSIX thread related types. 
 ******************************************************************************************/

typedef unsigned int      sem_t;
typedef unsigned int      pthread_cond_t;
typedef unsigned int      pthread_condattr_t;
typedef unsigned int      pthread_rwlock_t;
typedef unsigned int      pthread_rwlockattr_t;
typedef unsigned int      pthread_key_t;

/*******************************************************************************************
 * This structure and enum define the attributes for the "pthread_create" syscall.
 ******************************************************************************************/

typedef unsigned int  pthread_t;               

typedef struct pthread_attr_s
{
	unsigned int      attributes;      /*! user defined attributes bit vector             */
	unsigned int      cxy;             /*! target cluster identifier                      */
	unsigned int      lid;             /*! target core local index                        */
}
pthread_attr_t;

enum
{
    PT_ATTR_DETACH          = 0x0001,  /*! user defined not joinable                      */
    PT_ATTR_CLUSTER_DEFINED = 0x0002,  /*! user defined target cluster                    */
    PT_ATTR_CORE_DEFINED    = 0x0004,  /*! user defined core index in cluster             */
};

/******************************************************************************************* 
 * This enum defines the operation mnemonics for operations on POSIX unnamed semaphores.
 ******************************************************************************************/

typedef enum
{
	SEM_INIT,
	SEM_DESTROY,
	SEM_GETVALUE,
	SEM_WAIT,
	SEM_POST,
} 
sem_operation_t;

/******************************************************************************************* 
 * This enum defines the operation mnemonics for operations on POSIX condition variables.
 ******************************************************************************************/

typedef enum
{
	CONDVAR_INIT,
	CONDVAR_DESTROY,
    CONDVAR_WAIT,
    CONDVAR_SIGNAL,
    CONDVAR_BROADCAST,
} 
condvar_operation_t;

/******************************************************************************************* 
 * This enum defines the operation mnemonics for operations on POSIX barriers.
 ******************************************************************************************/

typedef enum
{
	BARRIER_INIT,
	BARRIER_DESTROY,
	BARRIER_WAIT,
} 
barrier_operation_t;

/******************************************************************************************* 
 * This enum defines the operation mnemonics for operations on POSIX mutex.
 ******************************************************************************************/

typedef enum
{
	MUTEX_INIT,
	MUTEX_DESTROY,
	MUTEX_LOCK,
	MUTEX_UNLOCK,
} 
mutex_operation_t;

/******************************************************************************************* 
 * This enum defines the attributes bit-vector for an "open" syscall.
 ******************************************************************************************/

typedef enum
{
    O_RDONLY   = 0x0010000,    /*! open file in read-only mode                            */
    O_WRONLY   = 0x0020000,    /*! open file in write-only mode                           */
    O_RDWR     = 0x0030000,    /*! open file in read/write mode                           */
    O_NONBLOCK = 0x0040000,    /*! do not block if data non available                     */
    O_APPEND   = 0x0080000,    /*! append on each write                                   */
    O_CREAT    = 0x0100000,    /*! create file if it does not exist                       */
    O_TRUNC    = 0x0200000,    /*! file length is forced to 0                             */
    O_EXCL     = 0x0400000,    /*! error if VFS_O_CREAT and file exist                    */
    O_SYNC	   = 0x0800000,    /*! synchronize File System on each write                  */
    O_CLOEXEC  = 0x1000000,    /*! set the close-on-exec flag in file descriptor          */
    O_DIR      = 0x2000000,    /*! new file descriptor is for a directory                 */
}
open_attr_t;

/******************************************************************************************* 
 * This structure contains the arguments passed to the "mmap" syscall.
 ******************************************************************************************/

#define      MAP_FAILED  0

typedef enum
{
    PROT_NONE   = 0x0,         /*! no access                                              */
    PROT_EXEC   = 0x1,         /*! executable                                             */
    PROT_WRITE  = 0x2,         /*! writable                                               */
    PROT_READ   = 0x4,         /*! readable                                               */
}
mmap_prot_t;

typedef enum
{
    MAP_FILE    = 0x00000000,  /*! map an open file defined by its fdid                   */ 
    MAP_ANON    = 0x00000001,  /*! map an anonymous vseg in local cluster                 */
    MAP_REMOTE  = 0x00000002,  /*! map an anonymous vseg in remote cluster (cxy == fdid)  */
    MAP_PRIVATE = 0x00000010,  /*!                                                        */
    MAP_SHARED  = 0x00000020,  /*!                                                        */
    MAP_FIXED   = 0x00000100,  /*! non supported                                          */
}
mmap_flags_t;

typedef struct mmap_attr_s
{
	void         * addr;       /*! requested virtual address (unused : should be NULL)    */
	unsigned int   length;     /*! requested vseg size (bytes)                            */
	unsigned int   prot;       /*! access modes                                           */
	unsigned int   flags;      /*! MAP_FILE / MAP_ANON / MAP_PRIVATE / MAP_SHARED         */
	unsigned int   fdid;       /*! file descriptor index (if MAP_FILE)                    */
	unsigned int   offset;     /*! file offset (if MAP_FILE)                              */
}
mmap_attr_t;

/******************************************************************************************* 
 * This enum defines the operation mnemonics for the "lseek" syscall.
 ******************************************************************************************/

typedef enum
{
    SEEK_SET  = 0,             /*! new_offset <= offset                                   */
    SEEK_CUR  = 1,             /*! new_offset <= current_offset + offset                  */
    SEEK_END  = 2,             /*! new_offset <= current_size + offset                    */
}
lseek_operation_t;

/******************************************************************************************* 
 * This enum defines the operation mnemonics for the "utls" syscall (Thread Local Storage).
 ******************************************************************************************/

typedef enum
{
    UTLS_SET       = 1,
    UTLS_GET       = 2,
    UTLS_GET_ERRNO = 3,
}
utls_operation_t;

/******************************************************************************************* 
 * This enum defines the operation mnemonics for the "trace" syscall.
 ******************************************************************************************/

typedef enum
{
    TRACE_ON       = 0,
    TRACE_OFF      = 1,
}
trace_operation_t;

/******************************************************************************************
 * This structure define the informations associated to a file descriptor,
 * returned to user space by the stat() system call.
 *****************************************************************************************/

typedef struct stat 
{
	unsigned int    dev;        /*! ID of device containing file                             */
	unsigned int    inum;       /*! inode number                                             */
	unsigned int    mode;       /*! protection                                               */
	unsigned int    nlink;      /*! number of hard links                                     */
	unsigned int    uid;        /*! user ID of owner                                         */
	unsigned int    gid;        /*! group ID of owner                                        */
	unsigned int    rdev;       /*! device ID (if special file)                              */
	unsigned int    size;       /*! total size, in bytes                                     */
	unsigned int    blksize;    /*! blocksize for file system I/O                            */
	unsigned int    blocks;     /*! number of 512B blocks allocated                          */
}
stat_t;

/*******************************************************************************************
 * These two structure defines the informations returned to user by the opendir()
 * function, used by the readdir() function, and released by the closedir() function.
 * - "DIR" describes the complete directory.
 * - "dirent" describes one directory entry.
 ******************************************************************************************/

#define DIRENT_NAME_MAX_LENGTH  56
#define DIRENT_MAX_NUMBER       63

struct dirent
{
    unsigned int   inum;                                /*! inode identifier              */
    unsigned int   type;                                /*! inode type                    */
    char           name[DIRENT_NAME_MAX_LENGTH];        /*! directory entry name          */
};

typedef struct user_directory
{
    struct dirent   entry[DIRENT_MAX_NUMBER];
    unsigned int    current;
}
DIR;

/*******************************************************************************************
 * These two structure are used by the gettimeofday() function.
 ******************************************************************************************/

struct timeval
{
    unsigned int tv_sec;                    /*! seconds since Jan. 1, 1970                */
    unsigned int tv_usec;                   /*! and microseconds                          */
};

struct timezone 
{
    int          tz_minuteswest;            /*8 of Greenwich                              */
    int          tz_dsttime;                /*! type of dst correction to apply           */
};



#endif  // _SHARED_SYSCALLS_H_
