/*
 * remote_condvar.h - distributed kernel condvar definition
 * 
 * Author  Alain Greiner (2016,2017)
 *
 * Copyright (c) UPMC Sorbonne Universites
 *
 * This file is part of ALMOS-MKH.
 *
 * ALMOS-MKH is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2.0 of the License.
 *
 * ALMOS-MKH is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with ALMOS-MKH; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */

#ifndef _REMOTE_CONDVAR_H_
#define _REMOTE_CONDVAR_H_

#include <kernel_config.h>
#include <hal_types.h>
#include <remote_spinlock.h>
#include <xlist.h>

/*****************************************************************************************
 *          This file defines a POSIX compliant condvar.
 *
 * It is used by multi-threaded applications to synchronise threads running in
 * different clusters, as all access functions uses hal_remote_lw() / hal_remote_sw()
 * portable remote access primitives.
 *
 * A condvar is declared by a given user process as a "pthread_cond_t" global variable.
 * This user type is implemented as an unsigned long, but the value is not used by the 
 * kernel. ALMOS-MKH uses only the condvar virtual address as an identifier.
 * For each user condvar, ALMOS-MKH creates a kernel "remote_condvar_t" structure,
 * dynamically allocated in the reference cluster by the remote_condvar_create() function, 
 * and destroyed by the remote_condvar_destroy() function, using RPC if the calling 
 * thread is not running in the reference cluster. The synchronisation is done by the
 * remote_condvar_wait(), remote_condvar_signal(), remote_convar_broadcast() functions.
 ****************************************************************************************/

/*****************************************************************************************
 * This structure defines the condvar descriptor.
 * - It contains an xlist of all condvars dynamically created by a given process,
 *   rooted in the reference process descriptor.
 * - It contains also the root of another xlist of all threads waiting on the condvar,
 *   resumed by a remote_condvar_signal(), or remote_condvar_broadcast().
 ****************************************************************************************/

typedef struct remote_condvar_s
{
    remote_spinlock_t  lock;     /*! lock protecting the waiting threads list           */
    intptr_t           ident;    /*! virtual address in user space == identifier        */
    xlist_entry_t      list;     /*! member of list of condvars in same process         */
    xlist_entry_t      root;     /*! root of list of waiting threads                    */
}
remote_condvar_t;

/*****************************************************************************************
 * This function returns an extended pointer on the remote condvar identified
 * by its virtual address in a given user process. It makes an associative search, 
 * scanning the list of condvars rooted in the reference process descriptor.
 *****************************************************************************************
 * @ ident    : condvar virtual address, used as identifier.
 * @ returns extended pointer on condvar if success / returns XPTR_NULL if not found. 
 ****************************************************************************************/
xptr_t remote_condvar_from_ident( intptr_t  ident );

/*****************************************************************************************
 * This function implement the pthread_condvar_init() syscall.
 * It allocates memory for the condvar descriptor in the reference cluster for
 * the calling process, it initializes the condvar state, and register it in the 
 * list of condvars owned by the reference process.
 *****************************************************************************************
 * @ ident       : condvar identifier (virtual address in user space).
 * @ return 0 if success / return ENOMEM if failure.
 ****************************************************************************************/
error_t remote_condvar_create( intptr_t ident );

/*****************************************************************************************
 * This function implement the pthread_condvar_destroy() syscall.
 * It releases the memory allocated for the condvar descriptor, and remove the condvar 
 * from the list of condvars owned by the reference process.
 *****************************************************************************************
 * @ condvar_xp  : extended pointer on condvar descriptor.
 ****************************************************************************************/
void remote_condvar_destroy( xptr_t   condvar_xp );

/*****************************************************************************************
 * This function implement the pthread_condvar_wait() syscall.
 * It unlock the mutex.
 * It register the calling thread in the condvar waiting queue, block the calling thread
 * on the THREAD_BLOCKED_CONDVAR condition and deschedule.
 * it lock the mutex.
 *****************************************************************************************
 * @ condvar_xp   : extended pointer on condvar descriptor.
 * @ mutex_xp     : extended pointer on associated mutex descriptor.
 ****************************************************************************************/
void remote_condvar_wait( xptr_t   condvar_xp,  
                          xptr_t   mutex_xp );

/*****************************************************************************************
 * This function implement the pthread_condvar_signal() syscall.
 * It unblocks the first waiting thread in the condvar waiting queue.
 *****************************************************************************************
 * @ condvar_xp  : extended pointer on condvar descriptor.
 ****************************************************************************************/
void remote_condvar_signal( xptr_t   condvar_xp );

/*****************************************************************************************
 * This function implement the pthread_condvar_broadcast() syscall.
 * It unblocks all waiting threads in the condvar waiting queue.
 *****************************************************************************************
 * @ condvar_xp  : extended pointer on condvar descriptor.
 ****************************************************************************************/
void remote_condvar_broadcast( xptr_t   condvar_xp );


#endif	/* _REMOTE_BARRIER_H_ */
