/*
 * sys_kill.c - Kernel function implementing the "kill" system call.
 * 
 * Author    Alain Greiner (2016,2017)
 *
 * Copyright (c)  UPMC Sorbonne Universites
 *
 * This file is part of ALMOS-MKH.
 *
 * ALMOS-MKH is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2.0 of the License.
 *
 * ALMOS-MKH is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with ALMOS-MKH; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */

#include <kernel_config.h>
#include <hal_types.h>
#include <hal_irqmask.h>
#include <errno.h>
#include <thread.h>
#include <printk.h>
#include <process.h>
#include <shared_syscalls.h>
#include <cluster.h>
#include <rpc.h>

///////////////////////////
int sys_kill( pid_t    pid,
              uint32_t sig_id )
{
    uint32_t    save_sr;       // required to enable IRQs
    xptr_t      process_xp;    // extended pointer on target reference process
    cxy_t       process_cxy;   // target process cluster
    process_t * process_ptr;   // local pointer on target process
    xptr_t      parent_xp;     // extended pointer on parent process
    cxy_t       parent_cxy;    // parent process cluster
    process_t * parent_ptr;    // local pointer on parent process
    pid_t       ppid;          // parent process PID

    thread_t  * this    = CURRENT_THREAD;

#if CONFIG_SYSCALL_DEBUG
uint64_t    tm_start;
uint64_t    tm_end;
tm_start = hal_get_cycles();
printk("\n[DBG] %s : core[%x,%d] enter / process %x / sig %d / cycle %d\n",
__FUNCTION__ , local_cxy , this->core->lid , pid, sig_id, (uint32_t)tm_start );
#endif

    // get cluster and pointers on reference process
    process_xp  = cluster_get_reference_process_from_pid( pid );
    process_cxy = GET_CXY( process_xp );
    process_ptr = (process_t *)GET_PTR( process_xp );

    // check process existence
    if( process_xp == XPTR_NULL )
    {
        syscall_dmsg("\n[ERROR] in %s : process %x not found\n",
        __FUNCTION__ , pid );
        this->errno = EINVAL;
        return -1;
    }
    
    // get parent process PID
    parent_xp  = hal_remote_lwd( XPTR( process_cxy , &process_ptr->parent_xp ) );
    parent_cxy = GET_CXY( parent_xp );
    parent_ptr = GET_PTR( parent_xp );
    ppid       = hal_remote_lw( XPTR( parent_cxy , &parent_ptr->pid ) );

    // processes INIT and processes KSH cannot be stoped or killed
    if( ppid < 2 )
    {
        syscall_dmsg("\n[ERROR] in %s : process %x cannot be killed\n",
        __FUNCTION__ , pid );
		this->errno = EINVAL;
        return -1;
    }

    // does nothing if sig_id == 0
    if( sig_id == 0 )  return 0;
    
    // check sig_id
    if( (sig_id != SIGSTOP) && (sig_id != SIGCONT) && (sig_id != SIGKILL) ) 
    {
        syscall_dmsg("\n[ERROR] in %s : illegal signal type for process %x\n",
        __FUNCTION__ , sig_id , pid );
		this->errno = EINVAL;
        return -1;
    }

    // enable IRQs
    hal_enable_irq( &save_sr );

    // execute process_make_kill() function in owner cluster
    if( local_cxy == process_cxy )                            // owner cluster is local 
    {
        process_make_kill( pid , sig_id );
    }
    else                                                      // owner cluster is remote
    {
        rpc_process_make_kill_client( process_cxy , pid , sig_id );
    }

    // restore IRQs
    hal_restore_irq( save_sr );

    hal_fence();

#if CONFIG_SYSCALL_DEBUG 
tm_end = hal_get_cycles();
printk("\n[DBG] %s : core[%x,%d] exit / process %x / sig %d / cost = %d\n",
__FUNCTION__ , local_cxy , this->core->lid , pid, sig_id, (uint32_t)(tm_end - tm_start) );
#endif
 
	return 0;

}  // end sys_kill()

