/*
 * sys_thread_cancel.c - terminate execution of a target thread.
 * 
 * Authors   Alain Greiner (2016,2017)
 *
 * Copyright (c) 2011,2012 UPMC Sorbonne Universites
 *
 * This file is part of ALMOS-MKH.
 *
 * ALMOS-MKH is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2.0 of the License.
 *
 * ALMOS-MKH is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with ALMOS-MKH; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */

#include <hal_types.h>
#include <hal_remote.h>
#include <hal_special.h>
#include <thread.h>
#include <errno.h>
#include <printk.h>

//////////////////////////////////////
int sys_thread_cancel( trdid_t trdid )
{
    xptr_t       target_xp;     // target thread extended pointer
	thread_t   * target_ptr;    // target thread local pointer
    cxy_t        target_cxy;    // target thread cluster
    ltid_t       target_ltid;   // target thread local index

#if CONFIG_SYSCALL_DEBUG
uint32_t     tm_start;
uint32_t     tm_end;
tm_start  = hal_get_cycles();
#endif

	thread_t   * this    = CURRENT_THREAD;
    process_t  * process = this->process;

    // check kernel stack overflow
    assert( (this->signature == THREAD_SIGNATURE), __FUNCTION__, "kernel stack overflow\n" );

    // get target thread ltid and cxy
    target_ltid = LTID_FROM_TRDID( trdid );
    target_cxy  = CXY_FROM_TRDID( trdid );

    // check trdid argument
	if( (target_ltid >= CONFIG_THREAD_MAX_PER_CLUSTER) || cluster_is_undefined( target_cxy ) )  
	{
        printk("\n[ERROR] in %s : illegal trdid argument\n", __FUNCTION__ );
		this->errno = EINVAL;
		return -1;
	}

    // get extended pointer on target thread
	target_xp  = thread_get_xptr( process->pid , trdid );

    if( target_xp == XPTR_NULL )
    {
        printk("\n[ERROR] in %s : target thread not found\n", __FUNCTION__ );
        this->errno = EINVAL;
        return -1;
    }

    // get target thread local pointer
    target_ptr = (thread_t *)GET_PTR( target_xp );

    // cal the relevant kernel function
    if( target_cxy == local_cxy )    // target thread is local
    {
        thread_kill( target_ptr );
    }
    else
    {
        rpc_thread_kill_client( target_cxy , target_ptr );
    }

#if CONFIG_SYSCALL_DEBUG
tm_end = hal_get_cycles();
syscall_dmsg("\n[DBG] %s : core[%x,%d] / thread %x in process %x / cycle %d\n"
"thread %x killed / cost = %d\n",
__FUNCTION__ , local_cxy , this->core->lid , this->trdid , this->process->pid , tm_start ,
trdid , (uint32_t)(tm_end - tm_start) );
#endif

	return 0;

}  // end sys_thread_cancel()
