/*
 * sys_wait.c - wait termination or blocking of a child process.
 * 
 * Author    Alain Greiner (2016,2017)
 *  
 * Copyright (c) UPMC Sorbonne Universites
 *
 * This file is part of ALMOS-MKH.
 *
 * ALMOS-MKH is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2.0 of the License.
 *
 * ALMOS-MKH is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with ALMOS-MKH; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */

#include <hal_types.h>
#include <hal_uspace.h>
#include <core.h>
#include <thread.h>
#include <process.h>
#include <vmm.h>
#include <printk.h>

/////////////////////////////////
int sys_wait( uint32_t * status )
{
	error_t     error;
    paddr_t     paddr;
    xptr_t      iter_xp;
    xptr_t      child_xp;
    process_t * child_ptr;
    cxy_t       child_cxy;
    pid_t       child_pid;
    int         child_state;

    thread_t  * this    = CURRENT_THREAD;
    process_t * process = this->process;

#if CONFIG_SYSCALL_DEBUG
uint64_t    tm_start;
uint64_t    tm_end;
tm_start = hal_get_cycles();
printk("\n[DBG] %s : core[%x,%d] enter / process %x / cycle %d\n",
__FUNCTION__ , local_cxy , this->core->lid , process->pid, (uint32_t)tm_start );
#endif

    // check status in user space
    error = vmm_v2p_translate( false , status , &paddr );

	if( error )
	{
        printk("\n[ERROR] in %s : status buffer unmapped for thread %x in process %x\n",
        __FUNCTION__ , this->trdid , process->pid );
        this->errno = EFAULT;
		return -1;
	}

    // get cluster and local pointer on reference process
    xptr_t      ref_xp  = process->ref_xp;
    cxy_t       ref_cxy = GET_CXY( ref_xp );
    process_t * ref_ptr = GET_PTR( ref_xp );

    // get extended pointer on children list root
    xptr_t root_xp = XPTR( ref_cxy , &ref_ptr->children_root );

    // get extended pointer on lock protecting the children list
    xptr_t lock_xp = XPTR( ref_cxy , &ref_ptr->children_lock );

    // exit this blocking loop only when a child processes change state
    while( 1 )
    {
        // get lock
        remote_spinlock_lock( lock_xp );

        // scan the list of child process
        XLIST_FOREACH( root_xp , iter_xp )
        {
            // get child process cluster and local pointer 
            child_xp  = XLIST_ELEMENT( iter_xp , process_t , children_list );
            child_ptr = GET_PTR( child_xp );
            child_cxy = GET_CXY( child_xp );

            // get the child PID
            child_pid = (int)hal_remote_lw( XPTR( child_cxy , &child_ptr->pid ) );

            // get the child process state
            child_state = hal_remote_lw( XPTR( child_cxy , &child_ptr->state ) );

            // check child process state
            if( child_state != PROCESS_STATE_RUNNING )
            {
                // release lock
                remote_spinlock_unlock( lock_xp );

#if CONFIG_SYSCALL_DEBUG 
tm_end = hal_get_cycles();
printk("\n[DBG] %s : core[%x,%d] exit / process %x / cost = %d\n",
__FUNCTION__ , local_cxy, this->core->lid, process->pid, (uint32_t)(tm_end - tm_start) );
#endif

                // return relevant info to process 
                hal_copy_to_uspace( status , &child_state , sizeof(int) );
                return child_pid;
            }
        }
        
        // release lock
        remote_spinlock_unlock( lock_xp );

        // block the calling thread until a child process change state
        thread_block( this , THREAD_BLOCKED_WAIT );
        sched_yield( "wait child termination" );
    }

    // never executed
	return -1; 

}  // end sys_wait()
