/*
 * cluster.c - Cluster-Manager related operations
 *
 * Author  Ghassan Almaless (2008,2009,2010,2011,2012)
 *         Mohamed Lamine Karaoui (2015)
 *         Alain Greiner (2016,2017)
 *
 * Copyright (c) UPMC Sorbonne Universites
 *
 * This file is part of ALMOS-MKH..
 *
 * ALMOS-MKH. is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2.0 of the License.
 *
 * ALMOS-MKH. is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with ALMOS-MKH.; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */

#include <kernel_config.h>
#include <hal_types.h>
#include <hal_atomic.h>
#include <hal_special.h>
#include <hal_ppm.h>
#include <remote_fifo.h>
#include <printk.h>
#include <errno.h>
#include <spinlock.h>
#include <core.h>
#include <scheduler.h>
#include <list.h>
#include <cluster.h>
#include <boot_info.h>
#include <bits.h>
#include <ppm.h>
#include <thread.h>
#include <kmem.h>
#include <process.h>
#include <dqdt.h>

/////////////////////////////////////////////////////////////////////////////////////
// Extern global variables
/////////////////////////////////////////////////////////////////////////////////////

extern process_t process_zero;     // allocated in kernel_init.c file


/////////////////////////////////////////////////
error_t cluster_init( struct boot_info_s * info )
{
    error_t         error;
    lpid_t          lpid;     // local process_index
    lid_t           lid;      // local core index
    uint32_t        i;        // index in loop on external peripherals
    boot_device_t * dev;      // pointer on external peripheral
    uint32_t        func;     // external peripheral functionnal type

	cluster_t * cluster = LOCAL_CLUSTER;

    // initialize cluster global parameters
	cluster->paddr_width     = info->paddr_width;
	cluster->x_width         = info->x_width;
	cluster->y_width         = info->y_width;
	cluster->x_size          = info->x_size;
	cluster->y_size          = info->y_size;
	cluster->io_cxy          = info->io_cxy;

    // initialize external peripherals channels
    for( i = 0 ; i < info->ext_dev_nr ; i++ )
    {
        dev  = &info->ext_dev[i];
        func = FUNC_FROM_TYPE( dev->type );   
        if( func == DEV_FUNC_TXT ) cluster->nb_txt_channels = dev->channels;
        if( func == DEV_FUNC_NIC ) cluster->nb_nic_channels = dev->channels;
        if( func == DEV_FUNC_IOC ) cluster->nb_ioc_channels = dev->channels;
        if( func == DEV_FUNC_FBF ) cluster->nb_fbf_channels = dev->channels;
    }

    // initialize cluster local parameters
	cluster->cores_nr        = info->cores_nr;

    // initialize the lock protecting the embedded kcm allocator
	spinlock_init( &cluster->kcm_lock );

cluster_dmsg("\n[DBG] %s for cluster %x enters\n",
__FUNCTION__ , local_cxy );

    // initialises DQDT
    cluster->dqdt_root_level = dqdt_init( info->x_size,
                                          info->y_size,
                                          info->y_width );
    cluster->threads_var = 0;
    cluster->pages_var   = 0;

    // initialises embedded PPM
	error = hal_ppm_init( info );

    if( error )
    {
        printk("\n[ERROR] in %s : cannot initialize PPM in cluster %x\n",
               __FUNCTION__ , local_cxy );
        return ENOMEM;
    }

cluster_dmsg("\n[DBG] %s : PPM initialized in cluster %x at cycle %d\n",
__FUNCTION__ , local_cxy , hal_get_cycles() );

    // initialises embedded KHM
	khm_init( &cluster->khm );

    cluster_dmsg("\n[DBG] %s : KHM initialized in cluster %x at cycle %d\n",
                 __FUNCTION__ , local_cxy , hal_get_cycles() );

    // initialises embedded KCM
	kcm_init( &cluster->kcm , KMEM_KCM );

    cluster_dmsg("\n[DBG] %s : KCM initialized in cluster %x at cycle %d\n",
                 __FUNCTION__ , local_cxy , hal_get_cycles() );

    // initialises all cores descriptors 
	for( lid = 0 ; lid < cluster->cores_nr; lid++ )
	{
		core_init( &cluster->core_tbl[lid],    // target core descriptor
	               lid,                        // local core index
	               info->core[lid].gid );      // gid from boot_info_t
	}

cluster_dmsg("\n[DBG] %s : cores initialized in cluster %x at cycle %d\n",
__FUNCTION__ , local_cxy , hal_get_cycles() );

    // initialises RPC fifo
	local_fifo_init( &cluster->rpc_fifo );
    cluster->rpc_threads = 0;

cluster_dmsg("\n[DBG] %s : RPC fifo inialized in cluster %x at cycle %d\n",
__FUNCTION__ , local_cxy , hal_get_cycles() );

    // initialise pref_tbl[] in process manager
	spinlock_init( &cluster->pmgr.pref_lock );
    cluster->pmgr.pref_nr = 0;
    cluster->pmgr.pref_tbl[0] = XPTR( local_cxy , &process_zero );
    for( lpid = 1 ; lpid < CONFIG_MAX_PROCESS_PER_CLUSTER ; lpid++ )
    {
        cluster->pmgr.pref_tbl[lpid] = XPTR_NULL;
    }

    // initialise local_list in process manager
	remote_spinlock_init( XPTR( local_cxy , &cluster->pmgr.local_lock ) );
    xlist_root_init( XPTR( local_cxy , &cluster->pmgr.local_root ) );
    cluster->pmgr.local_nr = 0;

    // initialise copies_lists in process manager
    for( lpid = 0 ; lpid < CONFIG_MAX_PROCESS_PER_CLUSTER ; lpid++ )
    {
	    remote_spinlock_init( XPTR( local_cxy , &cluster->pmgr.copies_lock[lpid] ) );
        cluster->pmgr.copies_nr[lpid] = 0;
        xlist_root_init( XPTR( local_cxy , &cluster->pmgr.copies_root[lpid] ) );
    }

cluster_dmsg("\n[DBG] %s Process Manager initialized in cluster %x at cycle %d\n",
__FUNCTION__ , local_cxy , hal_get_cycles() );

    hal_fence();

	return 0;
} // end cluster_init()

////////////////////////////////////////
bool_t cluster_is_undefined( cxy_t cxy )
{
    cluster_t * cluster = LOCAL_CLUSTER;

    uint32_t y_width = cluster->y_width;

    uint32_t x = cxy >> y_width;
    uint32_t y = cxy & ((1<<y_width)-1);

    if( x >= cluster->x_size ) return true;
    if( y >= cluster->y_size ) return true;

    return false;
}

////////////////////////////////////////////////////////////////////////////////////
//  Cores related functions
////////////////////////////////////////////////////////////////////////////////////

/////////////////////////////////
lid_t cluster_select_local_core()
{
    uint32_t min = 100;
    lid_t    sel = 0;
    lid_t    lid;

    cluster_t * cluster = LOCAL_CLUSTER;

    for( lid = 0 ; lid < cluster->cores_nr ; lid++ )
    {
        if( cluster->core_tbl[lid].usage < min )
        {
            min = cluster->core_tbl[lid].usage;
            sel = lid;
        }
    }
    return sel;
}

////////////////////////////////////////////////////////////////////////////////////
//  Process related functions
////////////////////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////
xptr_t cluster_get_reference_process_from_pid( pid_t pid )
{
    xptr_t ref_xp;   // extended pointer on reference process descriptor

    cluster_t * cluster = LOCAL_CLUSTER;

    // get owner cluster and lpid
    cxy_t  owner_cxy = CXY_FROM_PID( pid );
    lpid_t lpid      = LPID_FROM_PID( pid );

    // Check valid PID
    if( lpid >= CONFIG_MAX_PROCESS_PER_CLUSTER )  return XPTR_NULL;

    if( local_cxy == owner_cxy )   // local cluster is owner cluster
    {
        ref_xp = cluster->pmgr.pref_tbl[lpid];
    }
    else                              // use a remote_lwd to access owner cluster
    {
        ref_xp = (xptr_t)hal_remote_lwd( XPTR( owner_cxy , &cluster->pmgr.pref_tbl[lpid] ) );
    }

    return ref_xp;
}

///////////////////////////////////////////////
error_t cluster_pid_alloc( process_t * process,
                           pid_t     * pid )
{
    lpid_t      lpid;
    bool_t      found;

    pmgr_t    * pm         = &LOCAL_CLUSTER->pmgr;

    // get the process manager lock
    spinlock_lock( &pm->pref_lock );

    // search an empty slot
    found = false;
    for( lpid = 0 ; lpid < CONFIG_MAX_PROCESS_PER_CLUSTER ; lpid++ )
    {
        if( pm->pref_tbl[lpid] == XPTR_NULL )
        {
            found = true;
            break;
        }
    }

    if( found )
    {
        // register process in pref_tbl[]
        pm->pref_tbl[lpid] = XPTR( local_cxy , process );
        pm->pref_nr++;

        // returns pid
        *pid = PID( local_cxy , lpid );

        // release the processs_manager lock
        spinlock_unlock( &pm->pref_lock );

        return 0;
    }
    else
    {
        // release the processs_manager lock
        spinlock_unlock( &pm->pref_lock );

        return -1;
    }

} // end cluster_pid_alloc()

/////////////////////////////////////
void cluster_pid_release( pid_t pid )
{
    cxy_t  owner_cxy  = CXY_FROM_PID( pid );
    lpid_t lpid       = LPID_FROM_PID( pid );

    pmgr_t  * pm = &LOCAL_CLUSTER->pmgr;

    // check pid argument
    assert( (lpid < CONFIG_MAX_PROCESS_PER_CLUSTER) && (owner_cxy == local_cxy) ,
    __FUNCTION__ , "illegal PID" );

    // get the process manager lock
    spinlock_lock( &pm->pref_lock );

    // remove process from pref_tbl[]
    pm->pref_tbl[lpid] = XPTR_NULL;
    pm->pref_nr--;

    // release the processs_manager lock
    spinlock_unlock( &pm->pref_lock );

} // end cluster_pid_release()

///////////////////////////////////////////////////////////
process_t * cluster_get_local_process_from_pid( pid_t pid )
{
    xptr_t         process_xp;
    process_t    * process_ptr;
    xptr_t         root_xp;
    xptr_t         iter_xp;
    bool_t         found;

    found   = false;
    root_xp = XPTR( local_cxy , &LOCAL_CLUSTER->pmgr.local_root );

    XLIST_FOREACH( root_xp , iter_xp )
    {
        process_xp  = XLIST_ELEMENT( iter_xp , process_t , local_list );
        process_ptr = (process_t *)GET_PTR( process_xp );
        if( process_ptr->pid == pid )
        {
            found = true;
            break;
        }
    }

    if (found ) return process_ptr;
    else        return NULL;

}  // end cluster_get_local_process_from_pid()

//////////////////////////////////////////////////////
void cluster_process_local_link( process_t * process )
{
    uint32_t irq_state;
    pmgr_t * pm = &LOCAL_CLUSTER->pmgr;

    // get lock protecting the process manager local list
    remote_spinlock_lock_busy( XPTR( local_cxy , &pm->local_lock ) , & irq_state );

    xlist_add_last( XPTR( local_cxy , &pm->local_root ),
                    XPTR( local_cxy , &process->local_list ) );
    pm->local_nr++;

    // release lock protecting the process manager local list
    remote_spinlock_unlock_busy( XPTR( local_cxy , &pm->local_lock ) , irq_state );
}

////////////////////////////////////////////////////////
void cluster_process_local_unlink( process_t * process )
{
    uint32_t irq_state;
    pmgr_t * pm = &LOCAL_CLUSTER->pmgr;

    // get lock protecting the process manager local list
    remote_spinlock_lock_busy( XPTR( local_cxy , &pm->local_lock ) , &irq_state );

    xlist_unlink( XPTR( local_cxy , &process->local_list ) );
    pm->local_nr--;

    // release lock protecting the process manager local list
    remote_spinlock_unlock_busy( XPTR( local_cxy , &pm->local_lock ) , irq_state );
}

///////////////////////////////////////////////////////
void cluster_process_copies_link( process_t * process )
{
    uint32_t irq_state;
    pmgr_t * pm = &LOCAL_CLUSTER->pmgr;

    // get owner cluster identifier CXY and process LPID
    pid_t    pid        = process->pid;
    cxy_t    owner_cxy  = CXY_FROM_PID( pid );
    lpid_t   lpid       = LPID_FROM_PID( pid );

    // get extended pointer on lock protecting copies_list[lpid]
    xptr_t copies_lock  = XPTR( owner_cxy , &pm->copies_lock[lpid] );

    // get extended pointer on the copies_list[lpid] root
    xptr_t copies_root  = XPTR( owner_cxy , &pm->copies_root[lpid] );

    // get extended pointer on the local copies_list entry
    xptr_t copies_entry = XPTR( local_cxy , &process->copies_list );

    // get lock protecting copies_list[lpid]
    remote_spinlock_lock_busy( copies_lock , &irq_state );

    xlist_add_first( copies_root , copies_entry );
    hal_remote_atomic_add( XPTR( owner_cxy , &pm->copies_nr[lpid] ) , 1 );

    // release lock protecting copies_list[lpid]
    remote_spinlock_unlock_busy( copies_lock , irq_state );
}

/////////////////////////////////////////////////////////
void cluster_process_copies_unlink( process_t * process )
{
    uint32_t irq_state;
    pmgr_t * pm = &LOCAL_CLUSTER->pmgr;

    // get owner cluster identifier CXY and process LPID
    pid_t    pid        = process->pid;
    cxy_t    owner_cxy  = CXY_FROM_PID( pid );
    lpid_t   lpid       = LPID_FROM_PID( pid );

    // get extended pointer on lock protecting copies_list[lpid]
    xptr_t copies_lock  = hal_remote_lwd( XPTR( owner_cxy , &pm->copies_lock[lpid] ) );

    // get extended pointer on the local copies_list entry
    xptr_t copies_entry = XPTR( local_cxy , &process->copies_list );

    // get lock protecting copies_list[lpid]
    remote_spinlock_lock_busy( copies_lock , &irq_state );

    xlist_unlink( copies_entry );
    hal_remote_atomic_add( XPTR( owner_cxy , &pm->copies_nr[lpid] ) , -1 );

    // release lock protecting copies_list[lpid]
    remote_spinlock_unlock_busy( copies_lock , irq_state );
}

///////////////////////////////////////////
void cluster_processes_display( cxy_t cxy )
{
    xptr_t        root_xp;
    xptr_t        iter_xp;
    xptr_t        process_xp;     

    // get extended pointer on root of process in cluster cxy
    root_xp = XPTR( cxy , &LOCAL_CLUSTER->pmgr.local_root );

    // skip one line
    printk("\n");

    // loop on all reference processes in cluster cxy 
    XLIST_FOREACH( root_xp , iter_xp )
    {
        process_xp = XLIST_ELEMENT( iter_xp , process_t , local_list );
        process_display( process_xp );
    }
}  // end cluster_processes_display() 


