/*
 * sys_display.c - display the current state of a kernel structure on TXT0
 *
 * Author    Alain Greiner (2016,2017)
 *  
 * Copyright (c) UPMC Sorbonne Universites
 *
 * This file is part of ALMOS-MKH.
 *
 * ALMOS-MKH is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2.0 of the License.
 *
 * ALMOS-MKH is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with ALMOS-MKH; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */

#include <hal_types.h>
#include <hal_uspace.h>
#include <errno.h>
#include <cluster.h>
#include <thread.h>
#include <process.h>


/////////////////////////////
int sys_display( reg_t  type,
                 reg_t  arg0,
                 reg_t  arg1 )
{
    if( type == DISPLAY_STRING )
    {
        paddr_t   paddr;
        char      kbuf[256];

        char    * string = (char *)arg0;
 
        // check string in user space
        if( vmm_v2p_translate( false , string , &paddr ) ) return -1;

        // ckeck string length
        if( hal_strlen_from_uspace( string ) >= 256 ) return -1;

        // copy string in kernel space
        hal_strcpy_from_uspace( kbuf , string , 256 );

        // get thread, process and core
        thread_t  * this    = CURRENT_THREAD;
        process_t * process = this->process;
        core_t    * core    = this->core;

        // print message on TXT0 kernel terminal
        printk("\n[USER] thread %x / process %x / core[%x,%d] / cycle %d\n       %s",
        this->trdid , process->pid , local_cxy, core->lid ,
        (uint32_t)hal_get_cycles() , kbuf );

	    return 0; 
    }
    else if( type == DISPLAY_VMM )
    {
        pid_t pid = (pid_t)arg0;

        // get extended pointer on reference process
        xptr_t process_xp = cluster_get_reference_process_from_pid( pid );

	    if( process_xp == XPTR_NULL ) return -1;

        // get cluster and local pointer on process
        cxy_t       process_cxy = GET_CXY( process_xp );
        process_t * process_ptr = (process_t *)GET_PTR( process_xp );

        // call kernel function
        if( process_cxy == local_cxy )
        {
	        vmm_display( process_ptr , true );
        }
        else
        {
            rpc_vmm_display_client( process_cxy , process_ptr , true );
        }

	    return 0; 
    }
    else if( type == DISPLAY_SCHED )
    {
        cxy_t cxy = (cxy_t)arg0;
        lid_t lid = (lid_t)arg1;

        // check cluster argument
	    if( cluster_is_undefined( cxy ) ) return -1;

        // check core argument
        if( lid >= LOCAL_CLUSTER->cores_nr ) return -1;

        // call kernel function
        if( cxy == local_cxy )
        {
	        sched_display( lid );
        }
        else
        {
            rpc_sched_display_client( cxy , lid );
        }

	    return 0; 
    }
    else if( type == DISPLAY_PROCESS )
    {
        cxy_t cxy = (cxy_t)arg0;

        // check cluster argument
	    if( cluster_is_undefined( cxy ) ) return -1;

        // call kernel function
        cluster_processes_display( cxy );

        return 0;
    }
    else if( type == DISPLAY_VFS )
    {
        // call kernel function
        process_t * process = CURRENT_THREAD->process;
        vfs_display( process->vfs_root_xp );

        return 0;
    }
    else if( type == DISPLAY_CHDEV )
    {
        // call kernel function
        chdev_dir_display();

        return 0;
    }
    else return -1;

}  // end sys_get_sched()
