/*
 * chdev.c - channel device descriptor operations implementation.
 *
 * Authors  Alain Greiner   (2016)
 *
 * Copyright (c) UPMC Sorbonne Universites
 *
 * This file is part of ALMOS-MKH.
 *
 * ALMOS-MKH.is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2.0 of the License.
 *
 * ALMOS-MKH.is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with ALMOS-MKH.; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */

#include <kernel_config.h>
#include <hal_types.h>
#include <hal_special.h>
#include <hal_irqmask.h>
#include <printk.h>
#include <boot_info.h>
#include <xlist.h>
#include <kmem.h>
#include <scheduler.h>
#include <thread.h>
#include <rpc.h>
#include <chdev.h>
#include <devfs.h>


extern chdev_directory_t    chdev_dir;   // allocated in kernel_init.c

#if CONFIG_READ_DEBUG
extern uint32_t enter_chdev_cmd;
extern uint32_t exit_chdev_cmd;
extern uint32_t enter_chdev_server;
extern uint32_t exit_chdev_server;
#endif

////////////////////////////////////////////
char * chdev_func_str( uint32_t func_type ) 
{
  	if     ( func_type == DEV_FUNC_RAM ) return "RAM";
  	else if( func_type == DEV_FUNC_ROM ) return "ROM";
  	else if( func_type == DEV_FUNC_FBF ) return "FBF";
  	else if( func_type == DEV_FUNC_IOB ) return "IOB";
  	else if( func_type == DEV_FUNC_IOC ) return "IOC";
  	else if( func_type == DEV_FUNC_MMC ) return "MMC";
  	else if( func_type == DEV_FUNC_DMA ) return "DMA";
  	else if( func_type == DEV_FUNC_NIC ) return "NIC";
  	else if( func_type == DEV_FUNC_TIM ) return "TIM";
  	else if( func_type == DEV_FUNC_TXT ) return "TXT";
  	else if( func_type == DEV_FUNC_ICU ) return "ICU";
  	else if( func_type == DEV_FUNC_PIC ) return "PIC";
    else                                 return "undefined";
}

/////////////////////////////////////////
chdev_t * chdev_create( uint32_t    func,
                        uint32_t    impl,
                        uint32_t    channel,
                        uint32_t    is_rx,
                        xptr_t      base )
{
    chdev_t    * chdev;
    kmem_req_t   req;

    // allocate memory for chdev
    req.type   = KMEM_DEVICE;
    req.flags  = AF_ZERO;
    chdev      = (chdev_t *)kmem_alloc( &req );

    if( chdev == NULL ) return NULL;

    // initialize waiting threads queue and associated lock
    remote_spinlock_init( XPTR( local_cxy , &chdev->wait_lock ) );
    xlist_root_init( XPTR( local_cxy , &chdev->wait_root ) );

    // initialize attributes
    chdev->func    =  func;
    chdev->impl    =  impl;
    chdev->channel =  channel;
    chdev->is_rx   =  is_rx;
    chdev->base    =  base; 

    return chdev;

}  // end chdev_create()

///////////////////////////////////
void chdev_print( chdev_t * chdev )
{
    printk("\n - func      = %s"
           "\n - channel   = %d"
           "\n - base      = %l"
           "\n - cmd       = %x"
           "\n - isr       = %x"
           "\n - chdev     = %x\n",
           chdev_func_str(chdev->func),
           chdev->channel,
           chdev->base,
           chdev->cmd,
           chdev->isr,
           chdev );
}

//////////////////////////////////////////////////
void chdev_register_command( xptr_t     chdev_xp )
{
    thread_t * server_ptr;    // local pointer on server thread associated to chdev
    core_t   * core_ptr;      // local pointer on core running the server thread
    uint32_t   lid;           // core running the server thread local index
    xptr_t     lock_xp;       // extended pointer on lock protecting the chdev queue
    uint32_t   different;     // non zero if server thread core != client thread core
    uint32_t   save_sr;       // for critical section

#if CONFIG_READ_DEBUG
enter_chdev_cmd = hal_time_stamp();
#endif

    thread_t * this = CURRENT_THREAD;

#if CONFIG_DEBUG_CHDEV_REGISTER_COMMAND
uint32_t cycle = (uint32_t)hal_get_cycles();
if( CONFIG_DEBUG_CHDEV_REGISTER_COMMAND < cycle )
printk("\n[DBG] %s : client_thread %x (%s) enter / cycle %d\n",
__FUNCTION__, this, thread_type_str(this->type) , cycle );
#endif

    // get device descriptor cluster and local pointer
    cxy_t     chdev_cxy = GET_CXY( chdev_xp );
    chdev_t * chdev_ptr = (chdev_t *)GET_PTR( chdev_xp );

    // build extended pointers on client thread xlist and device root
    xptr_t  list_xp = XPTR( local_cxy , &this->wait_list );
    xptr_t  root_xp = XPTR( chdev_cxy , &chdev_ptr->wait_root );

    // get local pointer on server thread
    server_ptr = (thread_t *)hal_remote_lpt( XPTR( chdev_cxy , &chdev_ptr->server) );

    // build extended pointer on chdev lock protecting queue
    lock_xp = XPTR( chdev_cxy , &chdev_ptr->wait_lock );

    // get local pointer on core running the server thread
    core_ptr = (core_t *)hal_remote_lpt( XPTR( chdev_cxy , &server_ptr->core ) );

    // get core local index
    lid = hal_remote_lw( XPTR( chdev_cxy , &core_ptr->lid ) );

    // compute server core != thread core
    different = (lid != this->core->lid) || (local_cxy != chdev_cxy);

    // enter critical section to make atomic : 
    // (1) client blocking
    // (2) client registration in server queue 
    // (3) IPI to force server scheduling
    // (4) descheduling 
    // ... in this order
    hal_disable_irq( &save_sr );

    // block current thread
    thread_block( CURRENT_THREAD , THREAD_BLOCKED_IO );

    // register client thread in waiting queue 
    remote_spinlock_lock( lock_xp );
    xlist_add_last( root_xp , list_xp );
    remote_spinlock_unlock( lock_xp );

    // send IPI to core running the server thread if required
    if( different ) dev_pic_send_ipi( chdev_cxy , lid ); 
    
#if CONFIG_DEBUG_CHDEV_REGISTER_COMMAND
cycle = (uint32_t)hal_get_cycles();
if( CONFIG_DEBUG_CHDEV_REGISTER_COMMAND < cycle )
printk("\n[DBG] %s : client_thread %x (%s) exit / cycle %d\n",
__FUNCTION__, this, thread_type_str(this->type) , cycle );
#endif

    // deschedule
    assert( thread_can_yield( this ) , __FUNCTION__ , "illegal sched_yield\n" );
    sched_yield("blocked on I/O");

    // exit critical section
    hal_restore_irq( save_sr );

#if CONFIG_DEBUG_CHDEV_REGISTER_COMMAND
cycle = (uint32_t)hal_get_cycles();
if( CONFIG_DEBUG_CHDEV_REGISTER_COMMAND < cycle )
printk("\n[DBG] %s : client_thread %x (%s) resumes / cycle %d\n",
__FUNCTION__, this, thread_type_str(this->type) , cycle );
#endif

#if CONFIG_READ_DEBUG
exit_chdev_cmd = hal_time_stamp();
#endif

}  // end chdev_register_command()

///////////////////////////////////////////////
void chdev_sequencial_server( chdev_t * chdev )
{
    xptr_t          client_xp;    // extended pointer on waiting thread
    cxy_t           client_cxy;   // cluster of client thread
    thread_t      * client_ptr;   // local pointer on client thread
    thread_t      * server;       // local pointer on server thread
    xptr_t          root_xp;      // extended pointer on device waiting queue root
    xptr_t          lock_xp;      // extended pointer on lock ptotecting chdev queue

    server = CURRENT_THREAD;

#if CONFIG_DEBUG_CHDEV_SEQUENCIAL_SERVER
uint32_t cycle = (uint32_t)hal_get_cycles();
if( CONFIG_DEBUG_CHDEV_SEQUENCIAL_SERVER < cycle )
printk("\n[DBG] %s : server_thread %x enter / chdev = %x / cycle %d\n",
__FUNCTION__ , server , chdev , cycle );
#endif

    root_xp = XPTR( local_cxy , &chdev->wait_root );
    lock_xp = XPTR( local_cxy , &chdev->wait_lock );

	// This infinite loop is executed by the DEV thread
    // to handle commands registered in the chdev queue.
    while( 1 )
    {
        // get the lock protecting the waiting queue
        remote_spinlock_lock( lock_xp );

        // check waiting queue state
        if( xlist_is_empty( root_xp ) ) // waiting queue empty 
        {
            // release lock 
            remote_spinlock_unlock( lock_xp );

chdev_dmsg("\n[DBG] %s : thread %x deschedule /cycle %d\n",
__FUNCTION__ , server , hal_time_stamp() );

            // deschedule
            sched_yield("I/O queue empty");

chdev_dmsg("\n[DBG] %s : thread %x resume /cycle %d\n",
__FUNCTION__ , server , hal_time_stamp() );

        } 
        else                            // waiting queue not empty
        {

#if CONFIG_READ_DEBUG
enter_chdev_server = hal_time_stamp();
#endif
            // release lock 
            remote_spinlock_unlock( lock_xp );

            // get extended pointer on first client thread
            client_xp = XLIST_FIRST_ELEMENT( root_xp , thread_t , wait_list );

            // get client thread cluster, local pointer, and identifier 
            client_cxy = GET_CXY( client_xp );
            client_ptr = (thread_t *)GET_PTR( client_xp );

            // call driver command function to execute I/O operation
            chdev->cmd( client_xp );
        
            // remove the client thread from waiting queue
            remote_spinlock_lock( lock_xp );
            xlist_unlink( XPTR( client_cxy , &client_ptr->wait_list ) );
            remote_spinlock_unlock( lock_xp );

            // unblock client thread
            thread_unblock( client_xp , THREAD_BLOCKED_IO );

#if CONFIG_DEBUG_CHDEV_SEQUENCIAL_SERVER
cycle = (uint32_t)hal_get_cycles();
if( CONFIG_DEBUG_CHDEV_SEQUENCIAL_SERVER < cycle )
printk("\n[DBG] %s : server_thread %x complete operation for client %x / cycle %d\n",
__FUNCTION__ , server , client_ptr , cycle );
#endif

#if CONFIG_READ_DEBUG
exit_chdev_server = hal_time_stamp();
#endif

        }
    }  // end while
}  // end chdev_sequencial_server()

////////////////////////////////////////
xptr_t chdev_from_file( xptr_t file_xp )
{
    cxy_t         file_cxy;
    vfs_file_t  * file_ptr;
    uint32_t      inode_type;
    vfs_inode_t * inode_ptr;
    chdev_t     * chdev_ptr;

    // get cluster and local pointer on remote file descriptor
    // associated inode and chdev are stored in same cluster as the file desc.
    file_cxy  = GET_CXY( file_xp );
    file_ptr  = (vfs_file_t *)GET_PTR( file_xp );

    // get inode type from file descriptor
    inode_type = hal_remote_lw( XPTR( file_cxy , &file_ptr->type ) );
    inode_ptr  = (vfs_inode_t *)hal_remote_lpt( XPTR( file_cxy , &file_ptr->inode ) );

    assert( (inode_type == INODE_TYPE_DEV) , __FUNCTION__ ,
    "inode type %d is not INODE_TYPE_DEV", inode_type );

    // get chdev local pointer from inode extension
    chdev_ptr = (chdev_t *)hal_remote_lpt( XPTR( file_cxy , &inode_ptr->extend ) );

    return XPTR( file_cxy , chdev_ptr );

}  // end chdev_from_file()

////////////////////////
void chdev_dir_display()
{
    uint32_t  i;
    cxy_t     cxy;
    chdev_t * ptr;
    uint32_t  base;
    reg_t     save_sr;

    // get pointers on TXT0 chdev
    xptr_t    txt0_xp  = chdev_dir.txt_tx[0];
    cxy_t     txt0_cxy = GET_CXY( txt0_xp );
    chdev_t * txt0_ptr = GET_PTR( txt0_xp );

    // get extended pointer on remote TXT0 chdev lock
    xptr_t  lock_xp = XPTR( txt0_cxy , &txt0_ptr->wait_lock );

    // get TXT0 lock in busy waiting mode
    remote_spinlock_lock_busy( lock_xp , &save_sr );

    // header
    nolock_printk("\n***** external chdevs directory *****\n");

    // IOB
    cxy  = GET_CXY( chdev_dir.iob );
    ptr  = GET_PTR( chdev_dir.iob );
    base = (uint32_t)hal_remote_lwd( XPTR( cxy , &ptr->base ) );
    nolock_printk("  - iob       : cxy = %X / ptr = %X / base = %X\n", cxy, ptr, base);

    // PIC
    cxy  = GET_CXY( chdev_dir.pic );
    ptr  = GET_PTR( chdev_dir.pic );
    base = (uint32_t)hal_remote_lwd( XPTR( cxy , &ptr->base ) );
    nolock_printk("  - pic       : cxy = %X / ptr = %X / base = %X\n", cxy, ptr, base);

    // TXT
    for( i = 0 ; i < LOCAL_CLUSTER->nb_txt_channels ; i++ )
    {
        cxy = GET_CXY( chdev_dir.txt_rx[i] );
        ptr = GET_PTR( chdev_dir.txt_rx[i] );
        base = (uint32_t)hal_remote_lwd( XPTR( cxy , &ptr->base ) );
        nolock_printk("  - txt_rx[%d] : cxy = %X / ptr = %X / base = %X\n", i, cxy, ptr, base);

        cxy = GET_CXY( chdev_dir.txt_tx[i] );
        ptr = GET_PTR( chdev_dir.txt_tx[i] );
        base = (uint32_t)hal_remote_lwd( XPTR( cxy , &ptr->base ) );
        nolock_printk("  - txt_tx[%d] : cxy = %X / ptr = %X / base = %X\n", i, cxy, ptr, base);
    }

    // IOC
    for( i = 0 ; i < LOCAL_CLUSTER->nb_ioc_channels ; i++ )
    {
        cxy = GET_CXY( chdev_dir.ioc[i] );
        ptr = GET_PTR( chdev_dir.ioc[i] );
        base = (uint32_t)hal_remote_lwd( XPTR( cxy , &ptr->base ) );
        nolock_printk("  - ioc[%d]    : cxy = %X / ptr = %X / base = %X\n", i, cxy, ptr, base);
    }

    // FBF
    for( i = 0 ; i < LOCAL_CLUSTER->nb_fbf_channels ; i++ )
    {
        cxy  = GET_CXY( chdev_dir.fbf[i] );
        ptr  = GET_PTR( chdev_dir.fbf[i] );
        base = (uint32_t)hal_remote_lwd( XPTR( cxy , &ptr->base ) );
        nolock_printk("  - fbf[%d]    : cxy = %X / ptr = %X / base = %X\n", i, cxy, ptr, base);
    }

    // NIC
    for( i = 0 ; i < LOCAL_CLUSTER->nb_nic_channels ; i++ )
    {
        cxy = GET_CXY( chdev_dir.nic_rx[i] );
        ptr = GET_PTR( chdev_dir.nic_rx[i] );
        base = (uint32_t)hal_remote_lwd( XPTR( cxy , &ptr->base ) );
        nolock_printk("  - nic_rx[%d] : cxy = %X / ptr = %X / base = %X\n", i, cxy, ptr, base);

        cxy = GET_CXY( chdev_dir.nic_tx[i] );
        ptr = GET_PTR( chdev_dir.nic_tx[i] );
        base = (uint32_t)hal_remote_lwd( XPTR( cxy , &ptr->base ) );
        nolock_printk("  - nic_tx[%d] : cxy = %X / ptr = %X / base = %X\n", i, cxy, ptr, base);
    }

    // release lock
    remote_spinlock_unlock_busy( lock_xp , save_sr );

}  // end chdev_dir_display()

