/*
 * sys_mmap.c - map files, memory or devices into process virtual address space
 * 
 * Authors       Ghassan Almaless (2008,2009,2010,2011,2012)
 *               Alain Greiner (2016,2017)
 *
 * Copyright (c) UPMC Sorbonne Universites
 *
 * This file is part of ALMOS-MKH.
 *
 * ALMOS-MKH is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2.0 of the License.
 *
 * ALMOS-MKH is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with ALMOS-MKH; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */

#include <hal_types.h>
#include <hal_uspace.h>
#include <hal_irqmask.h>
#include <shared_syscalls.h>
#include <errno.h>
#include <thread.h>
#include <printk.h>
#include <mapper.h>
#include <vfs.h>
#include <process.h>
#include <vmm.h>

//////////////////////////////////
int sys_mmap( mmap_attr_t * attr )
{
    vseg_t      * vseg;
    cxy_t         vseg_cxy;
    vseg_type_t   vseg_type;
    mmap_attr_t   k_attr;       // attributes copy in kernel space
    xptr_t        mapper_xp;
    error_t       error;
    paddr_t       paddr;        // unused, but required for user space checking
    reg_t         save_sr;      // required to enable IRQs

	thread_t    * this    = CURRENT_THREAD;
	process_t   * process = this->process;

#if CONFIG_DEBUG_SYS_MMAP
uint64_t      tm_start;
uint64_t      tm_end;
tm_start = hal_get_cycles();
if ( CONFIG_DEBUG_SYS_MMAP < tm_start )
printk("\n[DBG] %s : thread %x enter / process %x / cycle %d\n",
__FUNCTION__, this, process->pid, (uint32_t)tm_start );
#endif

    // check arguments in user space
    error = vmm_v2p_translate( false , attr , &paddr );

    if ( error )
    {

#if CONFIG_DEBUG_SYSCALLS_ERROR
printk("\n[ERROR] in %s : arguments not in used space = %x\n", __FUNCTION__ , (intptr_t)attr );
#endif
		this->errno = EINVAL;
		return -1;
    }

    // copy arguments from uspace
    hal_copy_from_uspace( &k_attr , attr , sizeof(mmap_attr_t) );

    // get fdid, offset, and length arguments
    uint32_t fdid   = k_attr.fdid;
    uint32_t offset = k_attr.offset;
    uint32_t length = k_attr.length;

    // get flags
    bool_t     map_fixed   = ( (k_attr.flags & MAP_FIXED)   != 0 );
    bool_t     map_anon    = ( (k_attr.flags & MAP_ANON)    != 0 );
    bool_t     map_remote  = ( (k_attr.flags & MAP_REMOTE)  != 0 );
    bool_t     map_shared  = ( (k_attr.flags & MAP_SHARED)  != 0 );
    bool_t     map_private = ( (k_attr.flags & MAP_PRIVATE) != 0 );

    // MAP_FIXED not supported
    if( map_fixed )
    {

#if CONFIG_DEBUG_SYSCALLS_ERROR
printk("\n[ERROR] in %s : MAP_FIXED not supported\n", __FUNCTION__ );
#endif
        this->errno = EINVAL;
        return -1;
    }

    if( map_shared == map_private )
    {

#if CONFIG_DEBUG_SYSCALLS_ERROR
printk("\n[ERROR] in %s : MAP_SHARED xor MAP_PRIVATE\n", __FUNCTION__ );
#endif
        this->errno = EINVAL;
        return -1;
    }

    // FIXME handle Copy_On_Write for MAP_PRIVATE...

    // get access rigths
    bool_t     prot_read   = ( (k_attr.prot & PROT_READ )   != 0 );
    bool_t     prot_write  = ( (k_attr.prot & PROT_WRITE)   != 0 );

    // test mmap type : can be FILE / ANON / REMOTE

    if( (map_anon == false) && (map_remote == false) )   // FILE
    {
	    // FIXME: handle concurent delete of file by another thread closing it 

		if( fdid >= CONFIG_PROCESS_FILE_MAX_NR ) 
		{

#if CONFIG_DEBUG_SYSCALLS_ERROR
printk("\n[ERROR] in %s: bad file descriptor = %d\n", __FUNCTION__ , fdid );
#endif
            this->errno = EBADFD;
            return -1;
        }

        // get extended pointer on file descriptor
        xptr_t file_xp = process_fd_get_xptr( process , fdid );

        if( file_xp == XPTR_NULL )
        {

#if CONFIG_DEBUG_SYSCALLS_ERROR
printk("\n[ERROR] in %s: file %d not found\n", __FUNCTION__ , fdid );
#endif
            this->errno = EBADFD;
            return -1;
        }

        // get file cluster and local pointer
        cxy_t        file_cxy = GET_CXY( file_xp );
        vfs_file_t * file_ptr = (vfs_file_t *)GET_PTR( file_xp );

        // get inode pointer, mapper pointer and file attributes
        vfs_inode_t * inode_ptr  = hal_remote_lpt(XPTR(file_cxy , &file_ptr->inode ));
        uint32_t      file_attr  = hal_remote_lw (XPTR(file_cxy , &file_ptr->attr  ));
        mapper_t    * mapper_ptr = hal_remote_lpt(XPTR(file_cxy , &file_ptr->mapper));

        // get file size
		uint32_t size = hal_remote_lw( XPTR( file_cxy , &inode_ptr->size ) );

        // chek offset and length arguments
		if( (offset + length) > size)
		{

#if CONFIG_DEBUG_SYSCALLS_ERROR
printk("\n[ERROR] in %s: offset (%d) + len (%d) >= file's size (%d)\n", 
__FUNCTION__, k_attr.offset, k_attr.length, size );
#endif
            this->errno = ERANGE;
            return -1;
		}

        // check access rights
		if( (prot_read  && !(file_attr & FD_ATTR_READ_ENABLE)) ||
		    (prot_write && !(file_attr & FD_ATTR_WRITE_ENABLE)) )
		{

#if CONFIG_DEBUG_SYSCALLS_ERROR
printk("\n[ERROR] in %s: prot = %x / file_attr = %x)\n", 
__FUNCTION__ , k_attr.prot , file_attr );
#endif
			this->errno = EACCES;
			return -1;
		}

		// increment file refcount
		vfs_file_count_up( file_xp );

        mapper_xp = XPTR( file_cxy , mapper_ptr );
        vseg_type = VSEG_TYPE_FILE;
        vseg_cxy  = file_cxy;
    }
    else                                                // ANON or REMOTE
    {
        // no mapper for ANON or REMOTE
        mapper_xp = XPTR_NULL;

        if( map_anon )
        {
            vseg_type = VSEG_TYPE_ANON;
            vseg_cxy  = local_cxy;
        }
        else
        {
            vseg_type = VSEG_TYPE_REMOTE;
            vseg_cxy  = k_attr.fdid;
 
            if( cluster_is_undefined( vseg_cxy ) )
            {

#if CONFIG_DEBUG_SYSCALLS_ERROR
printk("\n[ERROR] in %s : illegal cxy for MAP_REMOTE\n", __FUNCTION__ );
#endif
                this->errno = EINVAL;
                return -1;
            }
        }
    }

    // enable IRQs
    hal_enable_irq( &save_sr );

    // get reference process cluster and local pointer
    xptr_t      ref_xp  = process->ref_xp;
    cxy_t       ref_cxy = GET_CXY( ref_xp );
    process_t * ref_ptr = (process_t *)GET_PTR( ref_xp );

    // create the vseg in reference cluster
    if( local_cxy == ref_cxy )
    {
        vseg = vmm_create_vseg( process,
                                vseg_type,
                                0,               // base address unused for mmap()
                                length,
                                offset,
                                0,               // file_size unused for mmap()
                                mapper_xp,
                                vseg_cxy );
    }
    else
    {
        rpc_vmm_create_vseg_client( ref_cxy,
                                    ref_ptr,
                                    vseg_type,
                                    0,            // base address unused for mmap()
                                    length,
                                    offset,
                                    0,            // file size unused for mmap()
                                    mapper_xp,
                                    vseg_cxy,
                                    &vseg ); 
    }
    
    // restore IRQs
    hal_restore_irq( save_sr );

    if( vseg == NULL )
    {

#if CONFIG_DEBUG_SYSCALLS_ERROR
printk("\n[ERROR] in %s : cannot create vseg\n", __FUNCTION__ );
#endif
        this->errno = ENOMEM;
        return -1;
    }

    // copy vseg base address to user space
    hal_copy_to_uspace( &attr->addr , &vseg->min , sizeof(intptr_t) );

    hal_fence();

#if CONFIG_DEBUG_SYS_MMAP
tm_end = hal_get_cycles();
if ( CONFIG_DEBUG_SYS_MMAP < tm_start )
printk("\n[DBG] %s : thread %x enter / process %x / cycle %d\n"
"vseg %s / cluster %x / base %x / size %x / cost %d\n", 
__FUNCTION__, this, process->pid, (uint32_t)tm_end,
vseg_type_str(vseg->type), vseg->cxy, vseg->min, length, (uint32_t)(tm_end - tm_start) );
#endif

        return 0;

}  // end sys_mmap()

