/*
 * rpc.h - RPC (Remote Procedure Call) operations definition.
 *
 * Author  Alain Greiner (2016,2017,2018)
 *
 * Copyright (c) UPMC Sorbonne Universites
 * 
 * This file is part of ALMOS-MKH.
 *
 * ALMOS-MKH is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2.0 of the License.
 *
 * ALMOS-MKH is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with ALMOS-MKH; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */

#ifndef _RPC_H_
#define _RPC_H_

#include <kernel_config.h>
#include <hal_types.h>
#include <hal_atomic.h>
#include <bits.h>
#include <spinlock.h>
#include <vseg.h>
#include <remote_fifo.h>

/**** Forward declarations ****/

struct process_s;
struct page_s;
struct vseg_s;
struct exec_info_s;
struct pthread_attr_s;
struct remote_sem_s;
struct fragment_s;
struct vfs_inode_s;
struct vfs_dentry_s;
struct vfs_file_s;
struct thread_s;
struct mapper_s;


/**********************************************************************************/
/**************  structures for Remote Procedure Calls ****************************/
/**********************************************************************************/

/***********************************************************************************
 * This enum defines all RPC indexes.
 * It must be consistent with the rpc_server[] array defined in in the rpc.c file.
 **********************************************************************************/

typedef enum
{
    RPC_PMEM_GET_PAGES         = 0,
    RPC_PMEM_RELEASE_PAGES     = 1,
    RPC_UNDEFINED_2            = 2,      
    RPC_PROCESS_MAKE_FORK      = 3,
    RPC_UNDEFINED_4            = 4,
    RPC_UNDEFINED_5            = 5,
    RPC_THREAD_USER_CREATE     = 6,
    RPC_THREAD_KERNEL_CREATE   = 7,
    RPC_UNDEFINED_8            = 8,
    RPC_PROCESS_SIGACTION      = 9,

    RPC_VFS_INODE_CREATE       = 10,
    RPC_VFS_INODE_DESTROY      = 11,
    RPC_VFS_DENTRY_CREATE      = 12,
    RPC_VFS_DENTRY_DESTROY     = 13,
    RPC_VFS_FILE_CREATE        = 14,
    RPC_VFS_FILE_DESTROY       = 15,
    RPC_VFS_INODE_LOAD         = 16,
    RPC_VFS_MAPPER_LOAD_ALL    = 17,
    RPC_FATFS_GET_CLUSTER      = 18,
    RPC_UNDEFINED_19           = 19,

    RPC_VMM_GET_VSEG           = 20,
    RPC_VMM_GET_PTE            = 21,
    RPC_KCM_ALLOC              = 22,
    RPC_KCM_FREE               = 23,
    RPC_MAPPER_MOVE_BUFFER     = 24,
    RPC_MAPPER_GET_PAGE        = 25,
    RPC_VMM_CREATE_VSEG        = 26,
    RPC_SCHED_DISPLAY          = 27,
    RPC_VMM_SET_COW            = 28,
    RPC_VMM_DISPLAY            = 29,

    RPC_MAX_INDEX              = 30,
}
rpc_index_t;

/***********************************************************************************
 * This defines the prototype of the rpc_server functions,
 * defined by the rpc_server[] array in the rpc.c file. 
 **********************************************************************************/

typedef  void (rpc_server_t) ( xptr_t xp );

/***********************************************************************************
 *  This structure defines the RPC descriptor
 **********************************************************************************/

typedef struct rpc_desc_s
{
	rpc_index_t         index;       /*! index of requested RPC service           */
	volatile uint32_t   response;    /*! all responses received when 0            */
    struct thread_s   * thread;      /*! local pointer on client thread           */
    uint32_t            lid;         /*! index of core running the calling thread */ 
    bool_t              blocking;    /*! blocking RPC when true                   */
    uint64_t            args[10];    /*! input/output arguments buffer            */
} 
rpc_desc_t;

/**********************************************************************************/
/******* Generic functions supporting RPCs : client side **************************/
/**********************************************************************************/

/***********************************************************************************
 * This function is executed by the client thread in the client cluster. 
 * It puts one RPC descriptor defined by the <desc> argument in the remote fifo
 * defined by the <cxy> argument.  It sends an IPI to the server if fifo is empty.
 * The RPC descriptor must be allocated in the caller's stack, and initialised by
 * the caller. It exit with a Panic message if remote fifo is still full after 
 * (CONFIG_RPC_PUT_MAX_ITERATIONS) retries. 
 * - When the RPC <blocking> field is true, this function blocks and deschedule. 
 *   It returns only when the server acknowledges the RPC by writing in the RPC
 *   "response" field, and unblocks the client.
 * - When the <blocking> field is false, this function returns as soon as the RPC
 *   has been registered in the FIFO, and the server thread must directly signal
 *   completion to the client thread.
 ***********************************************************************************
 * @ cxy   : server cluster identifier
 * @ desc  : local pointer on RPC descriptor in client cluster
 **********************************************************************************/
void rpc_send( cxy_t        cxy,    
               rpc_desc_t * desc );



/**********************************************************************************/
/******* Generic functions supporting RPCs : server side **************************/
/**********************************************************************************/

/***********************************************************************************
 * This function is the entry point for RPC handling on the server side.
 * It is executed by a core receiving an IPI, and each time the core enters,
 * or exit the kernel to handle .
 * It does nothing and return if the RPC_FIFO is empty.
 * The calling thread checks if it exist at least one non-blocked RPC thread,
 * creates a new RPC if required, and deschedule to allow the RPC thead to execute.
 **********************************************************************************/
void rpc_check();

/***********************************************************************************
 * This function contains the loop to execute all pending RPCs on the server side.
 * It is called by the rpc_thread_func() function with irq disabled, and after
 * RPC_FIFO ownership acquisition.
 ***********************************************************************************
 * @ rpc_fifo  : pointer on the local RPC fifo
 **********************************************************************************/
void rpc_execute_all( remote_fifo_t * rpc_fifo );

/***********************************************************************************
 * This function contains the infinite loop executed by a RPC thread.
 **********************************************************************************/
void rpc_thread_func();

/***********************************************************************************
 * This function is executed in case of illegal RPC index.
 **********************************************************************************/
void __attribute__((noinline)) rpc_undefined();




/**********************************************************************************/
/******* Marshalling functions attached to the various RPCs ***********************/
/**********************************************************************************/

/***********************************************************************************
 * [0] The RPC_PMEM_GET_PAGES allocates one or several pages in a remote cluster,
 * and returns the local pointer on the page descriptor.
 ***********************************************************************************
 * @ cxy     : server cluster identifier
 * @ order   : [in]  ln2( number of requested pages )
 * @ page    : [out] local pointer on page descriptor / NULL if failure
 **********************************************************************************/
void rpc_pmem_get_pages_client( cxy_t             cxy,
                                uint32_t          order,
                                struct page_s  ** page );

void rpc_pmem_get_pages_server( xptr_t xp );

/***********************************************************************************
 * [1] The RPC_PMEM_RELEASE_PAGES release one or several pages to a remote cluster.
 ***********************************************************************************
 * @ cxy     : server cluster identifier
 * @ page    : [in] local pointer on page descriptor to release.
 **********************************************************************************/
void rpc_pmem_release_pages_client( cxy_t            cxy,
                                    struct page_s  * page );

void rpc_pmem_release_pages_server( xptr_t xp );

/***********************************************************************************
 * [2] undefined slot
 **********************************************************************************/

/***********************************************************************************
 * [3] The RPC_PROCESS_MAKE_FORK creates a "child" process descriptor, and the
 * associated "child" thread descriptor in a target remote cluster that can be 
 * any cluster.  The child process is initialized from informations found in the 
 * "parent" process descriptor (that must be the parent reference cluster),
 * and from the "parent" thread descriptor that can be in any cluster.
 ***********************************************************************************
 * @ cxy              : server cluster identifier.
 * @ ref_process_xp   : [in]  extended pointer on reference parent process.
 * @ parent_thread_xp : [in]  extended pointer on parent thread.
 * @ child_pid        : [out] child process identifier.
 * @ child_thread_ptr : [out] local pointer on child thread.
 * @ error            : [out]  error status (0 if success).
 **********************************************************************************/
void rpc_process_make_fork_client( cxy_t              cxy,
                                   xptr_t             ref_process_xp,
                                   xptr_t             parent_thread_xp,
                                   pid_t            * child_pid,
                                   struct thread_s ** child_thread_ptr,
                                   error_t          * error );

void rpc_process_make_fork_server( xptr_t xp );

/***********************************************************************************
 * [4] undefined slot
 **********************************************************************************/

/***********************************************************************************
 * [5] undefined slot
 **********************************************************************************/

/***********************************************************************************
 * [6] The RPC_THREAD_USER_CREATE creates an user thread in the server cluster,
 * as specified by the arguments. It returns an extended pointer on the new
 * thread descriptor in server cluster, and an error code.
 * It is called by the sys_thread_create() system call.
 ***********************************************************************************
 * @ cxy       : server cluster identifier.
 * @ attr      : [in]  local pointer on pthread_attr_t in client cluster.
 * @ thread_xp : [out] buffer for thread extended pointer. 
 * @ error     : [out] error status (0 if success).
 **********************************************************************************/
void rpc_thread_user_create_client( cxy_t                   cxy,
                                    pid_t                   pid,
                                    void                  * start_func,
                                    void                  * start_arg,
                                    pthread_attr_t        * attr,
                                    xptr_t                * thread_xp,
                                    error_t               * error );

void rpc_thread_user_create_server( xptr_t xp );

/***********************************************************************************
 * [7] The RPC_THREAD_KERNEL_CREATE creates a kernel thread in the server cluster,
 * as specified by the type, func and args arguments. It returns the local pointer
 * on the thread descriptor in server cluster and an error code.
 * It is used by the dev_init() function to create the device server thread.
 ***********************************************************************************
 * @ cxy       : server cluster identifier.
 * @ type      : [in]  type of kernel thread.
 * @ func      : [in]  local pointer on thread function.
 * @ args      : [in]  local pointer on function arguments.
 * @ thread_xp : [out] pointer on buffer for thread extended pointer. 
 * @ error     : [out] error status (0 if success).
 **********************************************************************************/
void rpc_thread_kernel_create_client( cxy_t     cxy,
                                      uint32_t  type,
                                      void    * func,
                                      void    * args,
                                      xptr_t  * thread_xp,
                                      error_t * error );

void rpc_thread_kernel_create_server( xptr_t xp );

/***********************************************************************************
 * [8] undefined slot
 **********************************************************************************/

/***********************************************************************************
 * [9] The RPC_PROCESS_SIGACTION allows a thread running in any cluster 
 * to request a cluster identified by the <cxy> argument (local or remote)
 * to execute a given sigaction for a given cluster. The <action_type> and 
 * the <pid> arguments are defined in the shared RPC descriptor, that must be 
 * initialised by the client thread.
 *
 * WARNING : It is implemented as a NON BLOCKING multicast RPC, that can be sent 
 * in parallel to all process copies. The various RPC server threads atomically
 * decrement the <response> field in the shared RPC descriptor.
 * The last server thread unblock the client thread that blocked (after sending 
 * all RPC requests) in the process_sigaction() function.
 ***********************************************************************************
 * @ cxy     : server cluster identifier.
 * @ rpc     : pointer on ishared RPC descriptor initialized by the client thread.
 **********************************************************************************/
void rpc_process_sigaction_client( cxy_t               cxy,
                                   struct rpc_desc_s * rpc );
                             
void rpc_process_sigaction_server( xptr_t xp );

/*********************************************************************************** 
 * [10] The RPC_VFS_INODE_CREATE creates an inode and the associated mapper in a
 * remote cluster. The parent dentry must have been previously created.
 * It returns an extended pointer on the created inode.
 ***********************************************************************************
 * @ cxy        :  server cluster identifier.
 * @ dentry_xp  : [in]  extended pointer on parent dentry.
 * @ fs_type    : [in]  file system type.
 * @ inode_type : [in]  file system type.
 * @ extend     : [in]  fs_type_specific inode extension.
 * @ attr       : [in]  inode attributes.
 * @ rights     : [in]  access rights
 * @ uid        : [in]  user ID
 * @ gid        : [in]  group ID
 * @ inode_xp   : [out] buffer for extended pointer on created inode.
 * @ error      : [out] error status (0 if success).
 **********************************************************************************/
void rpc_vfs_inode_create_client( cxy_t      cxy,
                                  xptr_t     dentry_xp,
                                  uint32_t   fs_type,
                                  uint32_t   inode_type,
                                  void     * extend,
                                  uint32_t   attr,    
                                  uint32_t   rights,  
                                  uint32_t   uid,
                                  uint32_t   gid,
                                  xptr_t   * inode_xp,
                                  error_t  * error );

void rpc_vfs_inode_create_server( xptr_t xp );

/*********************************************************************************** 
 * [11] The RPC_VFS_INODE_DESTROY releases memory allocated for an inode descriptor
 * and for the associated mapper in a remote cluster. 
 ***********************************************************************************
 * @ cxy       :  server cluster identifier
 * @ inode     : [in]  local pointer on inode.
 **********************************************************************************/
void rpc_vfs_inode_destroy_client( cxy_t                 cxy,
                                   struct vfs_inode_s * inode );

void rpc_vfs_inode_destroy_server( xptr_t xp );

/*********************************************************************************** 
 * [12] The RPC_VFS_DENTRY_CREATE creates a dentry in a remote cluster. 
 * It returns an extended pointer on the created dentry.
 ***********************************************************************************
 * @ cxy        :  server cluster identifier
 * @ type       : [in]  file system type.
 * @ name       : [in]  directory entry name.
 * @ parent     : [in]  local pointer on parent inode.
 * @ dentry_xp  : [out] buffer for extended pointer on created dentry.
 * @ error      : [out] error status (0 if success).
 **********************************************************************************/
void rpc_vfs_dentry_create_client( cxy_t                  cxy,
                                   uint32_t               type,
                                   char                 * name,    
                                   struct vfs_inode_s   * parent,
                                   xptr_t               * dentry_xp,
                                   error_t              * error );

void rpc_vfs_dentry_create_server( xptr_t xp );

/*********************************************************************************** 
 * [13] The RPC_VFS_DENTRY_DESTROY releases memory allocated for an dentry descriptor
 * in a remote cluster. 
 ***********************************************************************************
 * @ cxy       :  server cluster identifier
 * @ dentry     : [in]  local pointer on dentry.
 **********************************************************************************/
void rpc_vfs_dentry_destroy_client( cxy_t                 cxy,
                                    struct vfs_dentry_s * dentry );

void rpc_vfs_dentry_destroy_server( xptr_t xp );

/*********************************************************************************** 
 * [14] The RPC_VFS_FILE_CREATE creates a file descriptor in a remote cluster. 
 * It returns an extended pointer on the created file structure.
 ***********************************************************************************
 * @ cxy        :  server cluster identifier
 * @ inode      : [in]  local pointer on parent inode.
 * @ file_attr  : [in]  new file attributes.
 * @ file_xp    : [out] buffer for extended pointer on created file.
 * @ error      : [out] error status (0 if success).
 **********************************************************************************/
void rpc_vfs_file_create_client( cxy_t                  cxy,
                                 struct vfs_inode_s   * inode,
                                 uint32_t               file_attr,
                                 xptr_t               * file_xp,
                                 error_t              * error );

void rpc_vfs_file_create_server( xptr_t xp );

/*********************************************************************************** 
 * [15] The RPC_VFS_FILE_DESTROY releases memory allocated for a file descriptor
 * in a remote cluster. 
 ***********************************************************************************
 * @ cxy       :  server cluster identifier
 * @ file       : [in]  local pointer on file.
 **********************************************************************************/
void rpc_vfs_file_destroy_client( cxy_t               cxy,
                                  struct vfs_file_s * file );

void rpc_vfs_file_destroy_server( xptr_t xp );

/*********************************************************************************** 
 * [16] The RPC_VFS_LOAD_INODE calls the vfs_inode_load() kernel function in a
 * remote cluster containing a parent inode directory to scan the associated
 * mapper, find a directory entry, identified by its name, and update the remote
 * child inode.
 ***********************************************************************************
 * @ cxy            :  server cluster identifier
 * @ parent_inode   : [in]  local pointer on parent inode.
 * @ name           : [in]  local pointer on child name (in client cluster).
 * @ child_inode_xp : [in]  extended pointer on child inode (in another cluster).
 * @ error          : [out] error status (0 if success).
 **********************************************************************************/
void rpc_vfs_inode_load_client( cxy_t                cxy,
                                struct vfs_inode_s * parent_inode,
                                char               * name,
                                xptr_t               child_inode_xp,
                                error_t            * error );

void rpc_vfs_inode_load_server( xptr_t xp );

/*********************************************************************************** 
 * [17] The RPC_VFS_MAPPER_LOAD_ALL calls the vfs_mapper_load_all() kernel function
 * in a remote cluster containing an inode, to load all pages of the associated 
 * mapper from the file system on device.
 ***********************************************************************************
 * @ cxy     :  server cluster identifier
 * @ inode   : [in]  local pointer on inode in server cluster.
 * @ error   : [out] error status (0 if success).
 **********************************************************************************/
void rpc_vfs_mapper_load_all_client( cxy_t                cxy,
                                     struct vfs_inode_s * inode,
                                     error_t            * error );

void rpc_vfs_mapper_load_all_server( xptr_t xp );

/***********************************************************************************
 * [18] The RPC_FATFS_GET_CLUSTER can be send by any thread running in a "client" 
 * cluster to scan the FAT mapper, stored in a remote "server" cluster, and get 
 * from the mapper the local pointer on a given page.
 ***********************************************************************************
 * @ cxy      : server cluster identifier.
 * @ mapper   : [in]  local pointer on FAT mapper.
 * @ first    : [in]  FATFS cluster index allocated to first page of file.
 * @ page     : [in]  page index in file.
 * @ cluster  : [out] local pointer on buffer for found FATFS cluster index.
 * @ error    : [out] local pointer on buffer for error code (in client cluster). 
 **********************************************************************************/
void rpc_fatfs_get_cluster_client( cxy_t             cxy,
                                   struct mapper_s * mapper,
                                   uint32_t          first,
                                   uint32_t          page,
                                   uint32_t        * cluster,
                                   error_t         * error );    

void rpc_fatfs_get_cluster_server( xptr_t xp );

/***********************************************************************************
 * [19] undefined slot
 **********************************************************************************/

/***********************************************************************************
 * [20] The RPC_VMM_GET_VSEG returns an extended pointer 
 * on the vseg containing a given virtual address in a given process.
 * The server cluster is supposed to be the reference cluster.
 * It returns a non zero error value if no vseg has been founded.
 ***********************************************************************************
 * @ cxy     : server cluster identifier.
 * @ process : [in]   pointer on process descriptor in server cluster. 
 * @ vaddr   : [in]   virtual address to be searched.
 * @ vseg_xp : [out]  buffer for extended pointer on vseg in client cluster.
 * @ error   : [out] local pointer on buffer for error code (in client cluster). 
 **********************************************************************************/
void rpc_vmm_get_vseg_client( cxy_t              cxy,
                              struct process_s * process,
                              intptr_t           vaddr,
                              xptr_t           * vseg_xp,
                              error_t          * error );

void rpc_vmm_get_vseg_server( xptr_t xp );

/***********************************************************************************
 * [21] The RPC_VMM_GET_PTE returns in the <ppn> and <attr> arguments the PTE value
 * for a given <vpn> in a given <process> (page_fault or copy_on_write event).
 * The server cluster is supposed to be the reference cluster, and the vseg
 * containing the VPN must be registered in the reference VMM.
 * It returns an error if physical memory cannot be allocated for the missing PTE2, 
 * or for the missing page itself.
 ***********************************************************************************
 * @ cxy     : server cluster identifier.
 * @ process : [in]   pointer on process descriptor in server cluster. 
 * @ vaddr   : [in]   virtual address to be searched.
 * @ cow     : [in]   "copy_on_write" event if true / "page_fault" event if false.
 * @ attr    : [out]  address of buffer for attributes.
 * @ ppn     : [out]  address of buffer for PPN.
 * @ error   : [out]  address of buffer for error code.
 **********************************************************************************/
void rpc_vmm_get_pte_client( cxy_t              cxy,
                             struct process_s * process,
                             vpn_t              vpn,
                             bool_t             cow,
                             uint32_t         * attr,
                             ppn_t            * ppn,
                             error_t          * error );

void rpc_vmm_get_pte_server( xptr_t xp );

/***********************************************************************************
 * [22] The RPC_KCM_ALLOC allocates memory from a given KCM in a remote cluster,
 * and returns an extended pointer on the allocated object.
  It returns XPTR_NULL if physical memory cannot be allocated. 
 ***********************************************************************************
 * @ cxy       : server cluster identifier.
 * @ kmem_type : [in]  KCM object type (as defined in kmem.h).
 * @ buf_xp    : [out] buffer for extended pointer on allocated buffer.
 **********************************************************************************/
void rpc_kcm_alloc_client( cxy_t      cxy,
                           uint32_t   kmem_type,
                           xptr_t   * buf_xp );  

void rpc_kcm_alloc_server( xptr_t xp );

/***********************************************************************************
 * [23] The RPC_KCM_FREE releases memory allocated for a KCM object of a given type,
 * in a remote cluster.
 ***********************************************************************************
 * @ cxy       : server cluster identifier.
 * @ buf       : [in] local pointer on allocated buffer. 
 * @ kmem_type : [in]  KCM object type (as defined in kmem.h).
 **********************************************************************************/
void rpc_kcm_free_client( cxy_t     cxy,
                          void    * buf,
                          uint32_t  kmem_type );

void rpc_kcm_free_server( xptr_t xp );

/***********************************************************************************
 * [24] The RPC_MAPPER_MOVE_BUFFER allows a client thread to require a remote
 * mapper to move data to/from a kernel or user buffer.
 * - It calls the mapper_move_user() function for a - possibly distributed -
 *   user buffer identified by a user-space pointer, and casted to uint64_t.
 * - It calls the mapper_move_kernel() function for a - possibly remote -
 *   kernel buffer identified by an extended pointer, and casted to uint64_t.
 * It is used by the vfs_move_user() function to move data between a mapper
 * and an user buffer required by a sys_read() or a sys_write().
 * It is used by the vmm_get_one_ppn() function to initialise a physical page
 * from a .elf file mapper, for a CODE or DATA vseg page fault.
 ***********************************************************************************
 * @ cxy         : server cluster identifier.
 * @ mapper      : [in]  local pointer on mapper
 * @ to_buffer   : [in]  move data from mapper to buffer if non zero.
 * @ is_user     : [in]  buffer in user space if true
 * @ file_offset : [in]  first byte to move in mapper 
 * @ buffer      : [in]  user space pointer / kernel extended pointer
 * @ size        : [in]  number of bytes to move
 * @ error       : [out] error status (0 if success).
 **********************************************************************************/
void rpc_mapper_move_buffer_client( cxy_t             cxy,
                                    struct mapper_s * mapper,
                                    bool_t            to_buffer,
                                    bool_t            is_user,
                                    uint32_t          file_offset,
                                    uint64_t          buffer,
                                    uint32_t          size,
                                    error_t         * error );

void rpc_mapper_move_buffer_server( xptr_t xp );

/***********************************************************************************
 * [25] The RPC_MAPPER_GET_PAGE allows a client thread to get the local pointer
 * on a remote page descriptor, for a page, identified by the page index in mapper.
 * It is used by the vmm_get_one_ppn() function to handle a page fault on 
 * a FILE type vseg.
 ***********************************************************************************
 * @ cxy         : server cluster identifier.
 * @ mapper      : [in]  local pointer on mapper.
 * @ index       : [in]  page index in mapper.
 * @ page        : [out] local pointer on page descriptor / NULL if failure.
 **********************************************************************************/
void rpc_mapper_get_page_client( cxy_t             cxy,
                                 struct mapper_s * mapper,
                                 uint32_t          index,
                                 struct page_s  ** page );

void rpc_mapper_get_page_server( xptr_t xp );

/***********************************************************************************
 * [26] The RPC_VMM_CREATE_VSEG allows a client thread to request the remote 
 * reference cluster of a given process to allocate and register in the reference
 * process VMM a new vseg descriptor.
 * On the server side, this RPC uses the vmm_create_vseg() function, and returns
 * to the client the local pointer on the created vseg descriptor.
 ***********************************************************************************
 * @ cxy         : server cluster identifier.
 * @ process     : [in]  local pointer on process descriptor in server.
 * @ type        : [in]  vseg type.
 * @ base        : [in]  base address (unused for dynamically allocated vsegs).
 * @ size        : [in]  number of bytes.
 * @ file_offset : [in]  offset in file (for CODE, DATA, FILE types).
 * @ file_size   : [in]  can be smaller than size for DATA type.
 * @ mapper_xp   : [in]  extended pointer on mapper (for CODE, DATA, FILE types).
 * @ vseg_cxy    : [in]  target cluster for mapping (if not data type).
 * @ vseg        : [out] local pointer on vseg descriptor / NULL if failure.
 **********************************************************************************/
void rpc_vmm_create_vseg_client( cxy_t              cxy,
                                 struct process_s * process,
                                 vseg_type_t        type,
                                 intptr_t           base,
                                 uint32_t           size,
                                 uint32_t           file_offset,
                                 uint32_t           file_size,
                                 xptr_t             mapper_xp,
                                 cxy_t              vseg_cxy,
                                 struct vseg_s   ** vseg );

void rpc_vmm_create_vseg_server( xptr_t xp );

/***********************************************************************************
 * [27] The RPC_SCHED_DISPLAY allows a client thread to request the display
 * of a remote scheduler, identified by the <lid> argument.
 ***********************************************************************************
 * @ cxy         : server cluster identifier.
 * @ lid         : [in]  local index of target core in client cluster.
 **********************************************************************************/
void rpc_sched_display_client( cxy_t              cxy,
                               lid_t              lid );

void rpc_sched_display_server( xptr_t xp );

/***********************************************************************************
 * [28] The RPC_VMM_SET_COW allows a client thread to request the remote reference
 * cluster to set the COW flag and reset the WRITABLE flag of all GPT entries for
 * the DATA, MMAP and REMOTE vsegs of process identified by the <process> argument.

 * of a remote scheduler, identified by the <lid> argument.
 ***********************************************************************************
 * @ cxy         : server cluster identifier.
 * @ process     : [in]  local pointer on reference process descriptor.
 **********************************************************************************/
void rpc_vmm_set_cow_client( cxy_t              cxy,
                             struct process_s * process );

void rpc_vmm_set_cow_server( xptr_t xp );

/***********************************************************************************
 * [29] The RPC_VMM_DISPLAY allows any client thread to display the VMM state
 * of a remote reference process identified by the <cxy> and <process> arguments.
 * The type of display is defined by the <detailed> boolean argument.
 ***********************************************************************************
 * @ cxy         : server cluster identifier.
 * @ process     : [in]  local pointer on reference process descriptor.
 * @ detailed    : [in]  detailed display if true.
 **********************************************************************************/
void rpc_vmm_display_client( cxy_t              cxy,
                             struct process_s * process,
                             bool_t             detailed );

void rpc_vmm_display_server( xptr_t xp );


#endif
