/*
 * sys_thread_cancel.c - terminate execution of an user thread.
 * 
 * Authors   Alain Greiner (2016,2017, 2018)
 *
 * Copyright (c) UPMC Sorbonne Universites
 *
 * This file is part of ALMOS-MKH.
 *
 * ALMOS-MKH is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2.0 of the License.
 *
 * ALMOS-MKH is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with ALMOS-MKH; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */

#include <hal_types.h>
#include <hal_remote.h>
#include <hal_special.h>
#include <thread.h>
#include <errno.h>
#include <printk.h>

//////////////////////////////////////
int sys_thread_cancel( trdid_t trdid )
{
    xptr_t       target_xp;     // target thread extended pointer

    // get killer thread pointers
	thread_t   * this    = CURRENT_THREAD;
    process_t  * process = this->process;

    // get extended pointer on target thread
	target_xp  = thread_get_xptr( process->pid , trdid );

    // check target_xp
    if( target_xp == XPTR_NULL )
    {

#if DEBUG_SYSCALLS_ERROR
printk("\n[ERROR] in %s : target thread %x not found\n", __FUNCTION__, trdid );
#endif
        this->errno = EINVAL;
        return -1;
    }

#if DEBUG_SYS_THREAD_CANCEL
uint64_t     tm_start;
uint64_t     tm_end;
tm_start = hal_get_cycles();
if( DEBUG_SYS_THREAD_CANCEL < tm_start )
printk("\n[DBG] %s : thread %x enter to kill thread %x / cycle %d\n",
__FUCTION__, this, GET_PTR( target_xp ), (uint32_t)tm_start );
#endif

    // cal the relevant kernel function
    thread_kill( target_xp,
                 0,           // is_exit
                 0 );         // is forced

#if DEBUG_SYS_THREAD_CANCEL
tm_end = hal_get_cycles();
if( DEBUG_SYS_THREAD_CANCEL < tm_end )
printk("\n[DBG] %s : thread %x exit after kill thread %x / cycle %d\n",
__FUCTION__, this, GET_PTR( target_xp ), (uint32_t)tm_end );
#endif

	return 0;

}  // end sys_thread_cancel()
