/*
 * stdio.c - User side system calls implementation.
 * 
 * Author     Alain Greiner (2016,2017)
 *
 * Copyright (c) UPMC Sorbonne Universites
 *
 * This file is part of ALMOS-MKH.
 *
 * ALMOS-MKH is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2.0 of the License.
 *
 * ALMOS-MKH is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with ALMOS-MKH; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */

#include <shared_syscalls.h>
#include <hal_user.h>
#include <stdio.h>
#include <stdlib.h>

#define  reg_t     int

///////////// POSIX standard system calls ////////////////////////////////////

///////////////////////
void exit( int status )
{
    hal_user_syscall( SYS_EXIT,
                      (reg_t)status, 0, 0, 0 );
}

/////////////////////////////////
int munmap( void         * addr,
            unsigned int   size )
{
    return hal_user_syscall( SYS_MUNMAP,
                             (reg_t)addr,
                             (reg_t)size, 0, 0 );
}

////////////////////////////////
int open( const char * pathname,
          int          flags,
          int          mode )
{
    return hal_user_syscall( SYS_OPEN,
                             (reg_t)pathname,
                             (reg_t)flags,
                             (reg_t)mode, 0 );
}

///////////////////////////////
void * mmap( void       * addr,
             unsigned int length,
             int          prot,
             int          flags,
             int          fd,
             unsigned int offset )
{
    mmap_attr_t attr;
    attr.addr   = addr;
    attr.length = length;
    attr.prot   = prot;
    attr.flags  = flags;
    attr.fdid   = fd;
    attr.offset = offset;

    if( hal_user_syscall( SYS_MMAP,
                          (reg_t)&attr, 0, 0, 0 ) ) return NULL;
    else                                            return attr.addr;
}

//////////////////////////
int read( int          fd,
          void       * buf,
          unsigned int count )
{
    return hal_user_syscall( SYS_READ,
                             (reg_t)fd,
                             (reg_t)buf,
                             (reg_t)count, 0 );
}

///////////////////////////
int write( int          fd,
           const void * buf,
           unsigned int count )
{
    return hal_user_syscall( SYS_WRITE,
                             (reg_t)fd,
                             (reg_t)buf,
                             (reg_t)count, 0 );
}

///////////////////////////
int lseek( int          fd,
           unsigned int offset,
           int          whence )
{
    return hal_user_syscall( SYS_LSEEK,
                             (reg_t)fd,
                             (reg_t)offset,
                             (reg_t)whence, 0 );
}

///////////////////
int close( int fd )
{
    return hal_user_syscall( SYS_CLOSE,
                             (reg_t)fd, 0, 0, 0 );
}

///////////////////////////////////
int unlink( const char * pathname )
{
    return hal_user_syscall( SYS_UNLINK,
                             (reg_t)pathname, 0, 0, 0 );
}

/////////////////////
int pipe( int fd[2] )
{
    return -1;
}

//////////////////////////////////
int chdir( const char * pathname )
{
    return hal_user_syscall( SYS_CHDIR,
                             (reg_t)pathname, 0, 0, 0 );
}

/////////////////////////////////
int mkdir( const char * pathname,
           int          mode )
{
    return hal_user_syscall( SYS_MKDIR,
                             (reg_t)pathname,
                             (reg_t)mode, 0, 0 );
}

//////////////////////////////////
int mkfifo( const char * pathname,
            int          mode )
{
    return hal_user_syscall( SYS_MKFIFO,
                             (reg_t)pathname,
                             (reg_t)mode, 0, 0 );
}

//////////////////////////////////////
DIR * opendir( const char * pathname )
{
    DIR   * dirp; 
    int     error;
    error = hal_user_syscall( SYS_OPENDIR,
                              (reg_t)pathname,
                              (reg_t)&dirp, 0, 0 );
    if( error ) return NULL;
    else        return dirp;
}

/////////////////////////////////////
struct dirent * readdir( DIR * dirp )
{
    struct dirent * dentp;
    int             error;
    error = hal_user_syscall( SYS_READDIR,
                              (reg_t)dirp,
                              (reg_t)&dentp, 0, 0 );
    if( error ) return NULL;
    else        return dentp;
}

//////////////////////////
int closedir( DIR * dirp )
{
    return hal_user_syscall( SYS_CLOSEDIR,
                             (reg_t)dirp, 0, 0, 0 );
}

/////////////////////////////
int getcwd( char       * buf,
            unsigned int bytes )
{
    return hal_user_syscall( SYS_GETCWD,
                             (reg_t)buf,
                             (reg_t)bytes, 0, 0 );
}

////////////////////////////
int rmdir( char * pathname )
{
    return hal_user_syscall( SYS_RMDIR,
                             (reg_t)pathname, 0, 0, 0 );
} 

/////////////////////////////////
int utls( unsigned int operation,
          unsigned int value )
{
    return hal_user_syscall( SYS_UTLS,
                             (reg_t)operation,
                             (reg_t)value, 0, 0 );
}

///////////////////////////////
int chmod( char     * pathname,
           unsigned int   rights )
{
    return hal_user_syscall( SYS_CHMOD,
                             (reg_t)pathname,
                             (reg_t)rights, 0, 0 );
}

/////////////////////////////////
int signal( unsigned int   sigid,
            void         * handler )
{
    return hal_user_syscall( SYS_SIGNAL,
                             (reg_t)sigid,
                             (reg_t)handler, 0, 0 );
}

///////////////////////////////////////
int gettimeofday( struct timeval  * tv,
                  struct timezone * tz )
{
    return hal_user_syscall( SYS_SIGNAL,
                             (reg_t)tv,
                             (reg_t)tz, 0, 0 );
}

///////////////////////////
int kill( unsigned int pid,
          unsigned int sig_id )
{
    return hal_user_syscall( SYS_KILL,
                             (reg_t)pid,
                             (reg_t)sig_id, 0, 0 );
}

////////////
int getpid()
{
    return hal_user_syscall( SYS_GETPID, 0, 0, 0, 0 );
}

//////////
int fork()
{
    return hal_user_syscall( SYS_FORK, 0, 0, 0, 0 );
}

///////////////////////////
int exec( char  * pathname,
          char ** argv,
          char ** envp )
{
    return hal_user_syscall( SYS_EXEC,
                             (reg_t)pathname,
                             (reg_t)argv,
                             (reg_t)envp, 0 );
}

/////////////////////////////////
int stat( const char  * pathname,
          struct stat * stat )
{
    return hal_user_syscall( SYS_STAT,
                             (reg_t)pathname,
                             (reg_t)stat, 0, 0 );
}

////////////////////////
int wait( int * status )
{
    return hal_user_syscall( SYS_WAIT,
                             (reg_t)status, 0, 0, 0 );
}



/////////////     Non standard system calls ////////////////////////////////////


//////////////////////////
int fg( unsigned int pid )
{
    return hal_user_syscall( SYS_FG,
                             (reg_t)pid, 0, 0, 0 );
}

//////////////////////////////////////
int get_config( unsigned int * x_size,
                unsigned int * y_size,
                unsigned int * ncores )
{
    return hal_user_syscall( SYS_GET_CONFIG,
                             (reg_t)x_size,
                             (reg_t)y_size,
                             (reg_t)ncores, 0 );
}

/////////////////////////////////
int get_core( unsigned int * cxy,
              unsigned int * lid )
{
    return hal_user_syscall( SYS_GET_CORE,
                             (reg_t)cxy,
                             (reg_t)lid, 0, 0 );
}

////////////////////////////////////
void display_string( char * string )
{
    hal_user_syscall( SYS_DISPLAY,
                      DISPLAY_STRING,
                      (reg_t)string, 0, 0 );
}

///////////////////////////////////
int display_vmm( unsigned int pid )
{
    return hal_user_syscall( SYS_DISPLAY,
                             DISPLAY_VMM,
                             (reg_t)pid, 0, 0 );
} 

////////////////////////////////
int display_sched( unsigned int cxy,
                   unsigned int lid )
{
    return hal_user_syscall( SYS_DISPLAY,
                             DISPLAY_SCHED,
                             (reg_t)cxy,
                             (reg_t)lid, 0 );
} 

/////////////////////////////////////////////////
int display_cluster_processes( unsigned int cxy )
{
    return hal_user_syscall( SYS_DISPLAY,
                             DISPLAY_CLUSTER_PROCESSES,
                             (reg_t)cxy, 0, 0 );
} 

///////////////////
int display_chdev()
{
    return hal_user_syscall( SYS_DISPLAY,
                             DISPLAY_CHDEV, 0, 0, 0 );
} 

/////////////////
int display_vfs()
{
    return hal_user_syscall( SYS_DISPLAY,
                             DISPLAY_VFS, 0, 0, 0 );
} 

////////////////////////////////////////////////
int display_txt_processes( unsigned int txt_id )
{
    return hal_user_syscall( SYS_DISPLAY,
                             DISPLAY_TXT_PROCESSES,
                             (reg_t)txt_id, 0, 0 );
} 

///////////////////////////////////////////
int get_cycle( unsigned long long * cycle )
{
    return hal_user_syscall( SYS_GET_CYCLE,
                             (reg_t)cycle, 0, 0, 0 );
}

//////////////////////////////////
int trace( unsigned int operation,
           unsigned int pid, 
           unsigned int trdid )
{
    return hal_user_syscall( SYS_TRACE,
                             (reg_t)pid,
                             (reg_t)trdid, 0, 0 );
}


