/*
 * sys_exit.c - Kernel function implementing the "exit" system call.
 * 
 * Author    Alain Greiner (2016,2017)
 *
 * Copyright (c)  UPMC Sorbonne Universites
 *
 * This file is part of ALMOS-MKH.
 *
 * ALMOS-MKH is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2.0 of the License.
 *
 * ALMOS-MKH is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with ALMOS-MKH; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */

#include <kernel_config.h>
#include <hal_types.h>
#include <hal_irqmask.h>
#include <errno.h>
#include <thread.h>
#include <printk.h>
#include <process.h>
#include <shared_syscalls.h>
#include <cluster.h>
#include <rpc.h>

///////////////////////////////
int sys_exit( uint32_t status )
{
    reg_t       save_sr;       // required to enable IRQs

    thread_t  * this    = CURRENT_THREAD;
    process_t * process = this->process;
    pid_t       pid     = process->pid;
    trdid_t     trdid   = this->trdid;

#if DEBUG_SYS_EXIT
uint64_t    tm_start;
uint64_t    tm_end;
tm_start = hal_get_cycles();
if( DEBUG_SYS_EXIT < tm_start )
printk("\n[DBG] %s : thread %x enter / process %x / status %x / cycle %d\n",
__FUNCTION__ , this, pid , status , (uint32_t)tm_start );
#endif

    // get owner cluster
    cxy_t  owner_cxy = CXY_FROM_PID( pid );

    // exit must be called by the main thread
    if( (owner_cxy != local_cxy) || (LTID_FROM_TRDID( trdid ) != 0) )
    {

#if DEBUG_SYSCALLS_ERROR
printk("\n[ERROR] in %s : calling thread %x is not thread 0 in owner cluster %x\n",
__FUNCTION__, trdid, owner_cxy );
#endif
         this->errno = EINVAL;
         return -1;
    }

    // enable IRQs
    hal_enable_irq( &save_sr );

    // register exit_status in owner process descriptor
    process->term_state = status;

#if( DEBUG_SYS_EXIT & 1)
printk("\n[DBG] %s : set exit status in process term_state\n", __FUNCTION__);
#endif

    // remove process from TXT list
    process_txt_detach( XPTR( local_cxy , process ) );

#if( DEBUG_SYS_EXIT & 1)
printk("\n[DBG] %s : removed from TXT list\n", __FUNCTION__);
#endif

    // mark for delete all process threads in all clusters (but the main) 
    process_sigaction( pid , DELETE_ALL_THREADS );

#if( DEBUG_SYS_EXIT & 1)
printk("\n[DBG] %s : deleted all other threads than main\n", __FUNCTION__);
#endif

    // restore IRQs
    hal_restore_irq( save_sr );

    // block the main thread itself 
    thread_block( XPTR( local_cxy , this ) , THREAD_BLOCKED_GLOBAL );

#if( DEBUG_SYS_EXIT & 1)
printk("\n[DBG] %s : blocked the main thread\n", __FUNCTION__);
#endif

    // atomically update owner process descriptor term_state to ask
    // the parent process sys_wait() function to delete this main thread 
    hal_remote_atomic_or( XPTR( local_cxy , &process->term_state ) ,
                          PROCESS_TERM_EXIT );

#if( DEBUG_SYS_EXIT & 1)
printk("\n[DBG] %s : set EXIT flag in process term_state\n", __FUNCTION__);
#endif

    hal_fence();

#if DEBUG_SYS_EXIT
tm_end = hal_get_cycles();
if( DEBUG_SYS_EXIT < tm_end )
printk("\n[DBG] %s : thread %x exit / process %x / status %x / cost = %d / cycle %d\n",
__FUNCTION__, this, pid, status, (uint32_t)(tm_end - tm_start), (uint32_t)tm_end );
#endif

    // main thread deschedule
    sched_yield( "process exit" );

    // this code should never be executed
    assert( false , __FUNCTION__ , "this code should not be executed...\n" );
	return 0;

}  // end sys_exit()

