/*
 * soclib_tty.c - soclib tty driver definition.
 *
 * Author  Alain Greiner (2016)
 *
 * Copyright (c)  UPMC Sorbonne Universites
 *
 * This file is part of ALMOS-MKH.
 *
 * ALMOS-MKH is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2.0 of the License.
 *
 * ALMOS-MKH is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with ALMOS-kernel; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */

#include <dev_txt.h>
#include <chdev.h>
#include <spinlock.h>


/****************************************************************************************
 * This driver supports the vci_tty_tsar component.
 * It implements the generic TXT device API:
 * - transfer one single character from TTY to command "buffer" if to_mem is non-zero.
 * - transfer "count" characters from command "buffer" to TTY if "to_mem is zero.
 ***************************************************************************************/

/****************************************************************************************
 *     SOCLIB_TTY registers offsets
 ***************************************************************************************/

#define TTY_WRITE              0
#define TTY_STATUS             1
#define TTY_READ               2
#define TTY_RX_IRQ_ENABLE      3
#define TTY_TX_IRQ_ENABLE      4

#define TTY_SPAN               8       // number of registers per channel

/****************************************************************************************
 * masks for TTY_STATUS_REG
 ***************************************************************************************/

#define TTY_STATUS_RX_FULL     1       // TTY_READ_REG full if 1
#define TTY_STATUS_TX_FULL     2       // TTY_WRITE_REG full if 1

/****************************************************************************************
 * This Rstructure is used by the soclib_tty_isr for the RX channel.
 ***************************************************************************************/

#define TTY_FIFO_DEPTH  128

typedef struct tty_fifo_s     // 32 bytes
{
    char          data[TTY_FIFO_DEPTH];   // one char per slot
    unsigned int  ptr;                    // next free slot index
    unsigned int  ptw;                    // next full slot index
    unsigned int  sts;                    // number of full slots
} tty_fifo_t;
 

/****************************************************************************************
 * This function masks both the TTY_RX and TTY_TX IRQs.
 * These IRQs are unmasked by the soclib_tty_cmd() function.
 ****************************************************************************************
 * @ chdev     : pointer on the TXT chdev descriptor.
 ***************************************************************************************/
void soclib_tty_init( chdev_t * chdev );

/****************************************************************************************
 * This function implements both the TXT_READ & TXT_WRITE commands registered in the
 * client thread descriptor (in the txt_cmd field), even if ALMOS-MKH defines two
 * different chdevs (and consequently two diffeerent server threads) for the RX and TX
 * directions. The client thread is identified by the <thread_xp> argument.
 * Depending on the command type, it unmasks the relevant TTY_RX / TTY_TX IRQ,
 * and blocks the TXT device server thread on the THREAD_BLOCKED_DEV_ISR, as the data
 * transfer is done by the ISR.
 * ****************************************************************************************
 * @ thread_xp : extended pointer on client thread descriptor.
 ***************************************************************************************/
void soclib_tty_cmd( xptr_t thread_xp );

/****************************************************************************************
 * This function implements the TXT_SYNC_WRITE command registered in the txt_aux_t
 * structure, using a busy waiting policy, without using the TTY IRQ.
 * It is used by the kernel do display debug messages on TXT0 terminal, without
 * interference with another TXT access to another terminal done by the same thread.
 ****************************************************************************************
 * @ thread_xp : pointer on the txt_aux_t structure containing the arguments.
 ***************************************************************************************/
void soclib_tty_aux( void * args );

/****************************************************************************************
 * This ISR is executed to handle both the TTY_TX_IRQ and the TTY_RX_IRQ, even if
 *   The RX_IRQ is activated as soon as the TTY_STATUS_RX_FULL bit is 1 in the
 *   TTY_STATUS register, when the TTY_RX_IRQ_ENABLE is non zero, indicating that
 *   the TTY_READ buffer is full and can be read.
 *   The TX_IRQ is activated as soon as the TTY_STATUS_TX_FULL bit is 0 in the
 *   TTY_STATUS register, when the TTY_TX_IRQ_ENABLE is non zero, indicating that
 *   the TTY_WRITE buffer is empty, and can be written.
 * WARNING : In ALMOS-MKH, the RX_IRQ is always enabled to catch the control signals,
 * but the TX_IRQ is dynamically enabled by the TXT_WRITE command, and disabled when
 * the command is completed.
 *
 * 1) The ISR first read the TTY_STATUS to get the current state of the TTY_READ and
 *   the TTY_WRITE buffers.
 *
 * 2) It try to read the first command registered in the server thread queue associated 
 *    to the TTY channel
 *
 * 2) The ISR handles the RX when the TTY_READ buffer is full :
 *   . it read the available character from the TTY_READ buffer, and this
 *     acknowledges the RX_IRQ.
 *   . if it is a control character ( ^C / ^D / ^Z ) it translate it to the proper
 *     signal and execute the relevant sigaction for the foreground process.
 *   . if it is a normal character, it try to get the first command registered in the 
 *     server thread queue. If it is a TXT_READ, it returns this character to the 
 *     command buffer in the client thread.
 * 
 * 3) The ISR handles the TX when the TTY_WRITE buffer is empty and a TXT_WRITE
 *   . it try to get it copies the
 *     character to the command buffer, acknowledges the TTY_RX_IRQ, and unblock the 
 *     associated server thread.
     
 *   . the control characters ^C / ^D / ^Z  are directly handled by the ISR and
 *     translated to the foreground process.

 * - the 
 the TXT_READ and TXT_WRITE commands.
 * It gets the command arguments from the first client thread in the TXT chdev queue:
 * - if TXT_READ, it transfers one byte from the TTY_READ_REG to the command buffer.
 *   It simply returns for retry if TTY_READ_REG is empty.
 * - if TXT_WRITE, it tries to transfer several bytes from the command buffer to the
 *   TTY_WRITE_REG. If the TTY_WRITE_REG is full, it updates the "count" and "buffer"
 *   command arguments and returns for retry.
 * When the I/O operation is completed, it sets the status field in the command, unblocks
 * the server thread, and unblocks the client thread.
 ****************************************************************************************
 * @ chdev     : local pointer on TXT chdev descriptor.
 ***************************************************************************************/
void soclib_tty_isr( chdev_t * chdev );

