/*
 * sys_exit.c - Kernel function implementing the "exit" system call.
 * 
 * Author    Alain Greiner (2016,2017,2018)
 *
 * Copyright (c)  UPMC Sorbonne Universites
 *
 * This file is part of ALMOS-MKH.
 *
 * ALMOS-MKH is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2.0 of the License.
 *
 * ALMOS-MKH is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with ALMOS-MKH; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */

#include <kernel_config.h>
#include <hal_types.h>
#include <hal_irqmask.h>
#include <errno.h>
#include <thread.h>
#include <printk.h>
#include <process.h>
#include <shared_syscalls.h>
#include <cluster.h>
#include <rpc.h>

///////////////////////////////
int sys_exit( uint32_t status )
{
    reg_t       save_sr;       // required to enable IRQs

    thread_t  * this    = CURRENT_THREAD;
    process_t * process = this->process;
    pid_t       pid     = process->pid;

#if DEBUG_SYS_EXIT
uint64_t    tm_start;
uint64_t    tm_end;
tm_start = hal_get_cycles();
if( DEBUG_SYS_EXIT < tm_start )
printk("\n[DBG] %s : thread %x enter / process %x / status %x / cycle %d\n",
__FUNCTION__ , this, pid , status , (uint32_t)tm_start );
#endif

    // get owner process descriptor pointers and cluster
    xptr_t      owner_xp  = cluster_get_owner_process_from_pid( pid );
    cxy_t       owner_cxy = GET_CXY( owner_xp );
    process_t * owner_ptr = GET_PTR( owner_xp );

    // get pointers on the process main thread
    thread_t * main    = hal_remote_lpt( XPTR( owner_cxy , &owner_ptr->th_tbl[0] ) );

    // enable IRQs
    hal_enable_irq( &save_sr );

    // mark for delete all process threads in all clusters 
    // (but the main thread and this calling thread) 
    process_sigaction( pid , DELETE_ALL_THREADS );

    // disable IRQs
    hal_restore_irq( save_sr );

#if( DEBUG_SYS_EXIT & 1)
if( tm_start > DEBUG_SYS_EXIT )
printk("\n[DBG] %s : thread %x deleted threads / process %x\n",
__FUNCTION__ , this, pid );
#endif

    // mark for delete this calling thread when it is not the main
    if( (owner_cxy != local_cxy) || (main != this) )
    {

#if( DEBUG_SYS_EXIT & 1)
if( tm_start > DEBUG_SYS_EXIT )
printk("\n[DBG] %s : calling thread %x deleted itself / process %x\n",
__FUNCTION__ , this, pid );
#endif
        thread_delete( XPTR( local_cxy , this ) , pid , true );
    }
         
    // remove process from TXT list
    process_txt_detach( owner_xp );

#if( DEBUG_SYS_EXIT & 1)
if( tm_start > DEBUG_SYS_EXIT )
printk("\n[DBG] %s : thread %x removed process %x from TXT list\n",
__FUNCTION__ , this, pid );
#endif

    // block the main thread 
    thread_block( XPTR( owner_cxy , main ) , THREAD_BLOCKED_GLOBAL );

#if( DEBUG_SYS_EXIT & 1)
if( tm_start > DEBUG_SYS_EXIT )
printk("\n[DBG] %s : thread %x blocked main thread for process %x\n",
__FUNCTION__, this , pid );
#endif

    // atomically update owner process descriptor term_state to ask
    // the parent process sys_wait() function to delete the main thread 
    hal_remote_atomic_or( XPTR( owner_cxy , &process->term_state ) ,
                          PROCESS_TERM_EXIT | (status & 0xFF) );

#if( DEBUG_SYS_EXIT & 1)
if( tm_start > DEBUG_SYS_EXIT )
printk("\n[DBG] %s : thread %x set exit status in process %x term_state\n",
__FUNCTION__ , this, pid );
#endif

    hal_fence();

#if DEBUG_SYS_EXIT
tm_end = hal_get_cycles();
if( DEBUG_SYS_EXIT < tm_end )
printk("\n[DBG] %s : thread %x exit / process %x / status %x / cost = %d / cycle %d\n",
__FUNCTION__, this, pid, status, (uint32_t)(tm_end - tm_start), (uint32_t)tm_end );
#endif

    // this thread deschedule
    sched_yield( "process exit" );

	return 0;

}  // end sys_exit()

