/*
 * stdio.h - User side syscalls definition.
 * 
 * Author     Alain Greiner (2016,2017)
 *
 * Copyright (c) UPMC Sorbonne Universites
 *
 * This file is part of ALMOS-MKH.
 *
 * ALMOS-MKH is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2.0 of the License.
 *
 * ALMOS-MKH is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with ALMOS-MKH; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */

#ifndef _STDIO_H_
#define _STDIO_H_

#include <shared_syscalls.h>

#define  NULL  (void *)0

/****************** Standard (POSIX) system calls  **************************************/

/***************************************************************************************** 
 * This function terminates a process.
 *****************************************************************************************
 * @ status   : terminaison status : 0 / EXIT_SUCCESS / EXIT_FAILURE.
 ****************************************************************************************/
void exit( int status );

/***************************************************************************************** 
 * This function open or create an open file descriptor.
 *****************************************************************************************
 * @ pathname   : pathname (can be relative or absolute).
 * @ flags      : bit vector attributes (see syscalls).
 * @ mode       : access rights (if O_CREAT is set).
 * @ return file descriptor index in fd_array if success / return -1 if failure.
 ****************************************************************************************/
int open( const char * pathname,
          int          flags,
          int          mode );

/***************************************************************************************** 
 * This function map physical memory (or a file) in the calling thread virtual space.
 *****************************************************************************************
 * @ addr       : unused and unsupported : must be NULL.
 * @ length     : requested number of bytes.
 * @ prot       : RWX access modes.
 * @ flags      : MAP_FILE / MAP_ANON / MAP_PRIVATE / MAP_SHARED (defined in syscalls.h)  
 * @ fdid       : file descriptor index (if MAP_FILE).                     
 * @ offset	    : offset in file (if MAP_FILE)
 * @ return 0 if success / return -1 if failure.
 ****************************************************************************************/
void * mmap( void         * addr,
             unsigned int   length,
             int            prot,
             int            flags,
             int            fd,
             unsigned int   offset );

/***************************************************************************************** 
 * This function read bytes from an open file identified by its file descriptor.
 * This file can be a regular file or a character oriented device.
 *****************************************************************************************
 * @ file_id  : open file index in fd_array.
 * @ buf      : buffer virtual address in user space.
 * @ count    : number of bytes.
 * @ return number of bytes actually read if success / returns -1 if failure.
 ****************************************************************************************/
int read( int            fd,
          void         * buf,
          unsigned int   count );

/***************************************************************************************** 
 * This function writes bytes to an open file identified by its file descriptor.
 * This file can be a regular file or character oriented device.
 *****************************************************************************************
 * @ file_id  : open file index in fd_array.
 * @ buf      : buffer virtual address in user space.
 * @ count    : number of bytes.
 * @ return number of bytes actually written if success / returns -1 if failure.
 ****************************************************************************************/
int write( int            fd,
           const void   * buf,
           unsigned int   count );

/***************************************************************************************** 
 * This function repositions the offset of the file descriptor identified by <file_id>,
 * according to the operation type defined by the <whence> argument.
 *****************************************************************************************
 * @ fd       : open file index in fd_array.
 * @ offset   : used to compute new offset value.
 * @ whence   : operation type (SEEK_SET / SEEK_CUR / SEEK_END defined in syscalls.h)
 * @ return 0 if success / returns -1 if failure.
 ****************************************************************************************/
int lseek( int           fd,
           unsigned int  offset,
           int           whence );

/***************************************************************************************** 
 * This function release the memory allocated for the file descriptor identified by 
 * the <file_id> argument, and remove the fd array_entry in all copies of the process 
 * descriptor.
 *****************************************************************************************
 * fd   : file descriptor index in fd_array.
 * @ return 0 if success / returns -1 if failure.
 ****************************************************************************************/
int close( int fd );

/***************************************************************************************** 
 * This function removes a directory entry identified by the <pathname> from the
 * directory, and decrement the link count of the file referenced by the link.
 * If the link count reduces to zero, and no process has the file open, then all resources
 * associated with the file are released.  If one or more process have the file open when
 * the last link is removed, the link is removed, but the removal of the file is delayed 
 * until all references to it have been closed.
 *****************************************************************************************
 * @ pathname   : pathname (can be relative or absolute).
 * @ return 0 if success / returns -1 if failure.
 ****************************************************************************************/
int unlink( const char * pathname );

/***************************************************************************************** 
 * This function creates in the calling thread cluster an unnamed pipe, and two
 * (read and write) file descriptors to access this pipe. The calling function must pass
 * the pointer on the fd[] array.
 * TODO not implemented yet...
 *****************************************************************************************
 * @ file_id[0] : [out] read only file descriptor index.
 * @ file_id[1] : [out] write only file descriptor index.
 * @ return 0 if success / return -1 if failure.
 ****************************************************************************************/
int pipe( int fd[2] );

/***************************************************************************************** 
 * This function change the current working directory in reference process descriptor.
 *****************************************************************************************
 * @ pathname   : pathname (can be relative or absolute).
 * @ return 0 if success / returns -1 if failure.
 ****************************************************************************************/
int chdir( const char * pathname );

/***************************************************************************************** 
 * This function creates a new directory in file system.
 *****************************************************************************************
 * @ pathname   : pathname (can be relative or absolute).
 * @ mode       : access rights (as defined in chmod).
 * @ return 0 if success / returns -1 if failure.
 ****************************************************************************************/
int mkdir( const char * pathname,
           int          mode );

/***************************************************************************************** 
 * This function creates a named FIFO file in the calling thread cluster.
 * The associated read and write file descriptors mut be be  explicitely created
 * using the open() system call.
 *****************************************************************************************
 * @ pathname   : pathname (can be relative or absolute).
 * @ mode       : access rights (as defined in chmod).
 * @ return 0 if success / returns -1 if failure.
 ****************************************************************************************/
int mkfifo( const char * pathname,
            int          mode );

/***************************************************************************************** 
 * This function opens the directory identified by the <pathname> argument, 
 * associates a directory stream with it and returns an user space pointer to identify 
 * this directory stream in subsequent operations.  
 *****************************************************************************************
 * @ pathname   : pathname (can be relative or absolute).
 * @ returns DIR* pointer if success / returns NULL if pathname cannot be accessed.
 ****************************************************************************************/
DIR * opendir( const char * pathname );

/***************************************************************************************** 
 * This function returns a pointer to the next directory entry. 
 *****************************************************************************************
 * @ dirp     : DIR pointer identifying the directory. 
 * @ returns dirent* pointer / returns NULL upon reaching end of directory or on error.
 ****************************************************************************************/
struct dirent * readdir( DIR * dirp );

/***************************************************************************************** 
 * This function closes the directory identified by the <dirp> argument, and releases
 * all structures associated with the <dirp> pointer.
 *****************************************************************************************
 * @ dirp     : DIR pointer identifying the directory. 
 * @ returns 0 if success / returns -1 if failure.
 ****************************************************************************************/
int closedir( DIR * dirp );

/***************************************************************************************** 
 * This function returns the pathname of the current working directory.
 *****************************************************************************************
 * buf     : buffer addres in user space.
 * nbytes  : user buffer size in bytes.
 * @ return 0 if success / returns -1 if failure.
 ****************************************************************************************/
int getcwd( char       * buf,
            unsigned int nbytes );

/***************************************************************************************** 
 * This function removes a directory file whose name is given by <pathname>.
 * The directory must not have any entries other than `.' and `..'.
 *****************************************************************************************
 * @ pathname   : pathname (can be relative or absolute).
 * @ return 0 if success / returns -1 if failure.
 ****************************************************************************************/
int rmdir( char * pathname ); 

/***************************************************************************************** 
 * This function implement the operations related to User Thread Local Storage.
 *****************************************************************************************
 * @ operation  : UTLS operation type as defined in "shared_sycalls.h" file.
 * @ value      : argument value for the UTLS_SET operation.
 * @ return value for the UTLS_GET and UTLS_GET_ERRNO / return -1 if failure.
 ****************************************************************************************/
int utls( unsigned int operation,
          unsigned int value );

/***************************************************************************************** 
 * This function change the acces rights for the file/dir identified by the
 * pathname argument.
 *****************************************************************************************
 * @ pathname   : pathname (can be relative or absolute).
 * @ rights     : acces rights.
 * @ return 0 if success / returns -1 if failure.
 ****************************************************************************************/
int chmod( char     * pathname,
           unsigned int   rights );

/***************************************************************************************** 
 * This function associate a specific signal handler to a given signal type.
 * The handlers for the SIGKILL and SIGSTOP signals cannot be redefined.
 ***************************************************************************************** 
 * @ sig_id    : index defining signal type (from 1 to 31).
 * @ handler   : pointer on fonction implementing the specific handler.
 * @ return 0 if success / returns -1 if failure.
 ****************************************************************************************/
int signal( unsigned int   sig_id,
            void         * handler );

/***************************************************************************************** 
 * This function returns in the structure <tv>, defined in the time.h file,
 * the current time (in seconds & micro-seconds).
 * It is computed from the calling core descriptor.
 * The timezone is not supported.
 *****************************************************************************************
 * @ tv      : pointer on the timeval structure.
 * @ tz      : pointer on the timezone structure : must be NULL.       
 * @ return 0 if success / returns -1 if failure.
 ****************************************************************************************/
int gettimeofday( struct timeval  * tv,
                  struct timezone * tz );

/***************************************************************************************** 
 * This function implements the "kill" system call on the user side.
 * It register the signal defined by the <sig_id> argument in all thread descriptors 
 * of a target process identified by the <pid> argument. This is done in all clusters
 * containing threads for the target process.
 * It can be executed by any thread running in any cluster, as this function uses
 * remote access to traverse the list of process copies stored in the owner cluster,
 * and the RPC_SIGNAL_RISE to signal the remote threads. 
 * This function does nothing for (sig_id == 0). This can be used to check process pid.
 * TODO : This first implementation supports only SIGKILL / SIGSTOP / SIGCONT values.
 ***************************************************************************************** 
 * @ pid      : target process identifier.
 * @ sig_id   : index defining the signal type.
 * @ return 0 if success / returns -1 if failure.
 ****************************************************************************************/
int kill( unsigned int  pid,
          unsigned int  sig_id );

/***************************************************************************************** 
 * This function implements the "getpid" system call on the user side.
 ***************************************************************************************** 
 * @ returns the process PID for the calling thread process.
 ****************************************************************************************/
int getpid();

/***************************************************************************************** 
 * This function implement the "fork" system call on the user side.
 * The calling process descriptor (parent process), and the associated thread descriptor
 * are replicated in a - likely - remote cluster, that becomes the new process owner. 
 * The child process get a new PID is linked to the parent PID. The child process inherit
 * from the parent process the memory image, and all open files (including the TXT).
 * The child process becomes the TXT terminal owner.
 * The target cluster depends on the "fork_user" flag and "fork_cxy" variable that can be
 * stored in the calling thread descriptor by the specific fork_place() system call.
 * If not, the kernel function makes a query to the DQDT to select the target cluster. 
 ***************************************************************************************** 
 * @ if success, returns child process PID to parent, and return O to child.
 * @ if failure, returns -1 to parent / no child process is created.
 ****************************************************************************************/
int fork();

/***************************************************************************************** 
 * This function implement the "exec" system call on the user side.
 * It creates, in the same cluster as the calling thread, a new process descriptor,
 * and a new associated main thread descriptor, executing a new memory image defined 
 * by the <filename> argument. This new process inherit from the old process the PID
 * and the PPID, as well as all open files (including the TXT). 
 * The old process descriptor, and all its threads are blocked, and marked for deletion.
 * Therefore the exec syscall does not return to the calling thread in case of success.
 * This function build an exec_info_t structure containing the new process arguments,
 * as defined by the <arv> argument, and the new process environment variables, 
 * as defined by the <envp>  argument.
 * TODO : the <argv> and <envp> arguments are not supported yet (both must be NULL).
 ***************************************************************************************** 
 * @ filename : string pointer on .elf filename (virtual pointer in user space)
 * @ argv     : array of strings on process arguments (virtual pointers in user space)
 * @ envp     : array of strings on environment variables (virtual pointers in user space)
 * @ does not return if success / returns -1 if failure.
 ****************************************************************************************/
int exec( char  * filename,
          char ** argv,
          char ** envp );

/***************************************************************************************** 
 * This function  returns in the <stat> structure, defined in the "shared_syscalls.h" 
 * file, various informations on the file/directory identified by the <pathname> argument.
 ***************************************************************************************** 
 * @ pathname  : user pointer on file pathname.
 * @ stat      : user pointer on the stat structure.
 * @ returns O if success / returns -1 if failure.
 ****************************************************************************************/
int stat( const char  * pathname,
          struct stat * stat );

/***************************************************************************************** 
 * This blocking function returns only when one child process of the calling process
 * changes state (from RUNNING to STOPPED / EXITED / KILLED). It returns the terminating
 * child process PID, and set in the <status> buffer the new child process state.
 ***************************************************************************************** 
 * @ status    : [out] terminating child process state.
 * @ returns terminating child process pid.
 ****************************************************************************************/
int wait( int * status );


/****************** Non standard (ALMOS_MKH specific) system calls **********************/


/***************************************************************************************** 
 * This function is used to give the process identified by the <pid> argument the 
 * exclusive ownership of the attached TXT_RX terminal.
 ***************************************************************************************** 
 * @ pid        : process identifier.
 * @ returns O if success / returns -1 if process not found.
 ****************************************************************************************/
int fg( unsigned int pid );

/*************************************************************************************** 
 * This function returns the hardware platform parameters.
 ***************************************************************************************
 * @ x_size   : [out] number of clusters in a row.
 * @ y_size   : [out] number of clusters in a column.
 * @ ncores   : [out] number of cores per cluster.
 * @ return always 0.
 **************************************************************************************/
int get_config( unsigned int * x_size,
                unsigned int * y_size,
                unsigned int * ncores );

/*************************************************************************************** 
 * This function returns the cluster an local index for the calling core.
 ***************************************************************************************
 * @ cxy      : [out] cluster identifier.
 * @ lid      : [out] core local index in cluster.
 * @ return always 0.
 **************************************************************************************/
int get_core( unsigned int * cxy,
              unsigned int * lid );

/***************************************************************************************
 * This function returns the calling core cycles counter,
 * taking into account a possible overflow on 32 bits architectures.
 ***************************************************************************************
 * @ cycle    : [out] current cycle value.
 * @ return always 0.
 **************************************************************************************/
int get_cycle( unsigned long long * cycle );

/*************************************************************************************** 
 * This debug function displays on the kernel terminal TXT0
 * the thread / process / core identifiers, the current cycle, plus a user defined
 * message as specified by the <string> argument. 
 ***************************************************************************************
 * @ string    : [in] user defined message.
 **************************************************************************************/
void display_string( char * string );

/*************************************************************************************** 
 * This debug function displays on the kernel terminal TXT0 
 * the state of the VMM forthe process <pid>, in cluster <cxy>.
 * It can be called by any thread running in any cluster.
 ***************************************************************************************
 * @ pid      : [in] process identifier.
 * @ return 0 if success / return -1 if illegal argument.
 **************************************************************************************/
int display_vmm( unsigned int  cxy,
                 unsigned int  pid );

/*************************************************************************************** 
 * This debug function displays on the kernel terminal TXT0 
 * the state of the core scheduler identified by the <cxy> and <lid> arguments.
 * It can be called by any thread running in any cluster.
 ***************************************************************************************
 * @ cxy      : [in] target cluster identifier.
 * @ lid      : [in] target core local index.
 * @ return 0 if success / return -1 if illegal arguments.
 **************************************************************************************/
int display_sched( unsigned int  cxy,
                   unsigned int  lid );

/*************************************************************************************** 
 * This debug function displays on the kernel terminal TXT0 
 * the list of process registered in a given cluster identified by the <cxy> argument.
 * It can be called by any thread running in any cluster.
 ***************************************************************************************
 * @ cxy      : [in] target cluster identifier.
 * @ return 0 if success / return -1 if illegal argument.
 **************************************************************************************/
int display_cluster_processes( unsigned int  cxy );

/*************************************************************************************** 
 * This debug function displays on the kernel terminal TXT0 
 * the list of channel devices available in the architecture.
 * It can be called by any thread running in any cluster.
 ***************************************************************************************
 * @ return always 0.
 **************************************************************************************/
int display_chdev();

/*************************************************************************************** 
 * This debug function displays on the kernel terminal TXT0 
 * the list of channel device or pseudo-files registered in the VFS cache.
 * It can be called by any thread running in any cluster.
 ***************************************************************************************
 * @ return always 0.
 **************************************************************************************/
int display_vfs();

/*************************************************************************************** 
 * This debug function displays on the kernel terminal TXT0 
 * the list of processes attached to a given TXT channel.
 * It can be called by any thread running in any cluster.
 ***************************************************************************************
 * @ return always 0.
 **************************************************************************************/
int display_txt_processes( unsigned int txt_id );

/***************************************************************************************** 
 * This debug function is used to activate / desactivate the context switches trace
 * for a core identified by the <cxy> and <lid> arguments.
 * It can be called by any thread running in any cluster.
 ***************************************************************************************** 
 * @ active     : activate trace if non zero / desactivate if zero.
 * @ cxy        : cluster identifier.
 * @ lid        : core local index.
 * @ returns O if success / returns -1 if illegal arguments.
 ****************************************************************************************/
int trace( unsigned int active,
           unsigned int cxy, 
           unsigned int lid );


#endif  // _STDIO_H_
