/*
 * sys_barrier.c - Access a POSIX barrier.
 * 
 * authors       Alain Greiner (2016,2017,2018)
 *
 * Copyright (c) UPMC Sorbonne Universites
 *
 * This file is part of ALMOS-MKH.
 *
 * ALMOS-MKH is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2.0 of the License.
 *
 * ALMOS-MKH is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with ALMOS-MKH; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */

#include <hal_types.h>
#include <hal_special.h>
#include <errno.h>
#include <thread.h>
#include <printk.h>
#include <vmm.h>
#include <syscalls.h>
#include <remote_barrier.h>

//////////////////////////////////
int sys_barrier( void     * vaddr,
                 uint32_t   operation,
                 uint32_t   count )
{
	error_t      error;
    vseg_t     * vseg;
  
    thread_t   * this    = CURRENT_THREAD;
    process_t  * process = this->process;

    // check vaddr in user vspace
	error = vmm_get_vseg( process , (intptr_t)vaddr , &vseg );

	if( error )
    {

#if DEBUG_SYSCALLS_ERROR
printk("\n[ERROR] in %s : unmapped barrier %x / thread %x / process %x\n",
__FUNCTION__ , (intptr_t)vaddr , this->trdid , process->pid );
vmm_display( process , false );
#endif
        this->errno = error;
        return -1;
    }

    // execute requested operation
	switch( operation )
	{
        //////////////////
	    case BARRIER_INIT:
        {
            error = remote_barrier_create( (intptr_t)vaddr , count );
    
		    if( error )
            {

#if DEBUG_SYSCALLS_ERROR
printk("\n[ERROR] in %s : cannot create barrier %x / thread %x / process %x\n",
__FUNCTION__ , (intptr_t)vaddr , this->trdid , process->pid );
#endif
                this->errno = error;
                return -1;
            }
			break;
        }
        //////////////////
	    case BARRIER_WAIT:
        {
            xptr_t barrier_xp = remote_barrier_from_ident( (intptr_t)vaddr );

            if( barrier_xp == XPTR_NULL )     // user error
            {

#if DEBUG_SYSCALLS_ERROR
printk("\n[ERROR] in %s : barrier %x not registered / thread %x / process %x\n",
__FUNCTION__ , (intptr_t)vaddr , this->trdid , process->pid );
#endif
                this->errno = EINVAL;
                return -1;
            }
            else                          // success
            {
                remote_barrier_wait( barrier_xp ); 
            }
            break;
        }
        /////////////////////
	    case BARRIER_DESTROY:
        {
            xptr_t barrier_xp = remote_barrier_from_ident( (intptr_t)vaddr );

            if( barrier_xp == XPTR_NULL )     // user error
            {

#if DEBUG_SYSCALLS_ERROR
printk("\n[ERROR] in %s : barrier %x not registered / thread %x / process %x\n",
__FUNCTION__ , (intptr_t)vaddr , this->trdid , process->pid );
#endif
                this->errno = EINVAL;
                return -1;
            }
            else                          // success
            {
                remote_barrier_destroy( barrier_xp ); 
            }
            break;
        }
        ////////
	    default:
        {
            printk("\n[PANIC] in %s : illegal operation type\n", __FUNCTION__ );
            hal_core_sleep();
        }
	}  // end switch

	return 0;

}  // end sys_barrier()

