/*
 * remote_spinlock.h - kernel remote spinlock definition.
 *
 * Author  Alain Greiner (2016,2017,2018)
 * 
 * Copyright (c) UPMC Sorbonne Universites
 *
 * This file is part of ALMOS-MKH.
 *
 * ALMOS-MKH is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2.0 of the License.
 *
 * ALMOS-MKH is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with ALMOS-MKH; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */

#ifndef _REMOTE_SPINLOCK_H_
#define _REMOTE_SPINLOCK_H_

#include <kernel_config.h>
#include <hal_kernel_types.h>
#include <xlist.h>

/***************************************************************************************
 * This structure defines a remote spinlock, that can be used to protect
 * exclusive access to a trans-cluster shared resource. It can be taken by any 
 * thread running in any cluster. All access functions use remote pointers.
 * The "owner" and "list" are optionnal fields used for debug. 
 * It register the list of all remote spinlocks taken by a given thread.
 **************************************************************************************/

typedef struct remote_spinlock_s
{
    volatile uint32_t     taken;       /*! free if 0 / taken if non zero             */

#if DEBUG_REMOTE_SPINLOCKS
    xptr_t                owner;       /*! extended pointer on the owner thread      */
    xlist_entry_t         list;        /*! list of all remote_lock taken by owner    */
#endif

} 
remote_spinlock_t;

/***************************************************************************************
 * This function initializes a remote spinlock.
 ***************************************************************************************
 * @ lock_xp : extended pointer on the remote spinlock
 **************************************************************************************/
void remote_spinlock_init( xptr_t   lock_xp );

/*******************************************************************************************
 * This blocking function uses a busy waiting strategy to lock a remote spinlock.
 * It polls the lock and returns only when the lock has been taken.
 * All IRQs are disabled and will keep disabled until the lock is released. 
 * It increments the calling thread local_locks count when the lock has been taken.
 *******************************************************************************************
 * @ lock_xp    : extended pointer on the remote spinlock.
 * @ irq_state  : buffer to save the SR state (in the calling thread stack)
 ******************************************************************************************/
void remote_spinlock_lock_busy( xptr_t     lock_xp,
                                uint32_t * irq_state );

/*******************************************************************************************
 * This function releases a remote busy_waiting spinlock.
 * It restores the CPU SR state.
 * It decrements the calling thread locks count.
 *******************************************************************************************
 * @ lock_xp    : extended pointer on remote spinlock.
 * @ irq_state  : value to be resrored in CPU SR 
 ******************************************************************************************/
void remote_spinlock_unlock_busy( xptr_t     lock_xp,
                                  uint32_t   irq_state );

/***************************************************************************************
 * This blocking function locks a remote spinlock.
 * If the lock is already taken, the calling thread deschedule, and retry
 * when it is rescheduled.
 * It increments the calling thread locks count when the lock has been taken.
 ***************************************************************************************
 * @ lock_xp   : extended pointer on the remote spinlock
 **************************************************************************************/
void remote_spinlock_lock( xptr_t lock_xp );

/***************************************************************************************
 * This non blocking function try once to lock a remote spinlock.
 * It increments the calling thread locks count in case of success.
 ***************************************************************************************
 * @ lock_xp    : extended pointer on the remote spinlock
 * @ returns O if success / returns non zero if lock already taken.
 **************************************************************************************/
error_t remote_spinlock_trylock( xptr_t  lock_xp );

/***************************************************************************************
 * This function releases a remote spinlock.
 * It decrements the calling thread locks count.
 ***************************************************************************************
 * @ lock_xp    : extended pointer on the remote spinlock
 **************************************************************************************/
void remote_spinlock_unlock( xptr_t  lock_xp );

#endif
