/*
 * sys_chdir: change process current working directory
 *
 * Author    Alain Greiner (2016,2017)
 *
 * Copyright (c) 2011,2012 UPMC Sorbonne Universites
 *
 * This file is part of ALMOS-MKH.
 *
 * ALMOS-MKH is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2.0 of the License.
 *
 * ALMOS-MKH is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with ALMOS-MKH; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */

#include <hal_kernel_types.h>
#include <hal_uspace.h>
#include <chdev.h>
#include <rwlock.h>
#include <vfs.h>
#include <errno.h>
#include <thread.h>
#include <printk.h>
#include <process.h>

/////////////////////////////////
int sys_chdir ( char * pathname )
{
    error_t   error;
    char      kbuf[CONFIG_VFS_MAX_PATH_LENGTH];

    thread_t  * this    = CURRENT_THREAD;
    process_t * process = this->process;

    // check pathname length
    if( hal_strlen_from_uspace( pathname ) >= CONFIG_VFS_MAX_PATH_LENGTH )
    {
        printk("\n[ERROR] in %s : pathname too long\n", __FUNCTION__ );
        this->errno = ENFILE;
        return -1;
    }

    // copy pathname in kernel space
    hal_strcpy_from_uspace( kbuf , pathname , CONFIG_VFS_MAX_PATH_LENGTH );

    // get cluster and local pointer on reference process
    xptr_t      ref_xp  = process->ref_xp;
    process_t * ref_ptr = (process_t *)GET_PTR( ref_xp );
    cxy_t       ref_cxy = GET_CXY( ref_xp );

    // get extended pointer on cwd lock in reference process
    xptr_t lock_xp = hal_remote_lwd( XPTR( ref_cxy , &ref_ptr->cwd_lock ) );

    // get cwd lock in read mode
    remote_rwlock_rd_lock( lock_xp );

    // call relevant VFS function
    error = vfs_chdir( process->vfs_cwd_xp , kbuf );

    // release cwd lock
    remote_rwlock_rd_unlock( lock_xp );

    if( error )
    {
        printk("\n[ERROR] in %s : cannot change current directory\n", __FUNCTION__ );
        this->errno = error;
        return -1;
    }

    return 0;
}
