/*
 * hal_special.h - Generic Special Registers Access API definition.
 *
 * Authors   Ghassan Almaless (2008,2009,2010,2011,2012)
 *           Alain Greiner    (2016)
 *
 * Copyright (c)  UPMC Sorbonne Universites
 *
 * This file is part of ALMOS-MKH.
 *
 * ALMOS-MKH is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2.0 of the License.
 *
 * ALMOS-MKH is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with ALMOS-MKH; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */

#ifndef  _HAL_CPU_H_
#define  _HAL_CPU_H_

#include <hal_types.h>

/****  Forward declarations  ***/

struct thread_s;

///////////////////////////////////////////////////////////////////////////////////////////
//      Generic Special Registers Access API (implementation in hal_special.c)
//
// ALMOS-MKH uses the following API to access the MMU and other core protected registers.
///////////////////////////////////////////////////////////////////////////////////////////

/*****************************************************************************************
 * This function returns the global core identifier from the calling core register.
 ****************************************************************************************/
gid_t hal_get_gid();

/*****************************************************************************************
 * This function returns the content of the calling core cycles counter register.
 ****************************************************************************************/
uint32_t hal_time_stamp();

/*****************************************************************************************
 * This function returns the current thread pointer from the calling core register.
 ****************************************************************************************/
struct thread_s * hal_get_current_thread();

/*****************************************************************************************
 * This function registers a thread pointer in the calling core register.
 ****************************************************************************************/
void hal_set_current_thread( struct thread_s * thread );

/*****************************************************************************************
 * This function writes into the proper core register to enable the floating point unit.
 ****************************************************************************************/
void hal_fpu_enable();

/*****************************************************************************************
 * This function writes into the proper core register to disable the floating point unit.
 ****************************************************************************************/
void hal_fpu_disable();

/*****************************************************************************************
 * This function returns the current value of stack pointer from core register.
 ****************************************************************************************/
uint32_t hal_get_stack();

/*****************************************************************************************
 * This function registers a new value in the core stack pointer and returns previous one.
 ****************************************************************************************/
extern inline uint32_t hal_set_stack( void * new_val );

/*****************************************************************************************
 * This function returns the faulty address in case of address exception.
 ****************************************************************************************/
uint32_t hal_get_bad_vaddr();

/*****************************************************************************************
 * This function makes an uncachable read to a 32 bits variable in local memory.
 * @ ptr     : pointer on the variable
 * @ returns the value
 ****************************************************************************************/
uint32_t hal_uncached_read( uint32_t * ptr );

/*****************************************************************************************
 * This function invalidates the cache line containing a given address.
 * @ ptr     : address in local memory
 ****************************************************************************************/
void hal_invalid_dcache_line( void * ptr );

/*****************************************************************************************
 * This blocking function flushes the write buffer to synchronize all pending writes.
 ****************************************************************************************/
void hal_wbflush();

/*****************************************************************************************
 * This forbids code reordering by the compiler.
 ****************************************************************************************/
void hal_rdbar();

/*****************************************************************************************
 * This function forces the calling core in idle-low-power mode.
 ****************************************************************************************/
void hal_core_sleep();

/*****************************************************************************************
 * This function returns after a fixed delay of (4 * delay) cycles.
 ****************************************************************************************/
void hal_fixed_delay();

/*****************************************************************************************
 * This function returns information on MMU exceptions :
 * @ mmu_ins_excp_code : [out] instruction fetch exception code
 * @ mmu_ins_bad_vaddr : [out] instruction fetch faulty virtual address
 * @ mmu_dat_excp_code : [out] data access exception code
 * @ mmu_dat_bad_vaddr : [out] data access faulty virtual address
 ****************************************************************************************/
void hal_get_mmu_excp( intptr_t * mmu_ins_excp_code,
                       intptr_t * mmu_ins_bad_vaddr,
                       intptr_t * mmu_dat_excp_code,
                       intptr_t * mmu_dat_bad_vaddr );


#endif	/* _HAL_SPECIAL_H_ */
