/*
 * sysfs.h - sysfs interface 
 *
 * Authors   Ghassan Almaless (2008,2009,2010,2011,2012)
 *           Mohamed Lamine Karaoui (2015)
 *           Alain Greiner (2016)
 *
 * Copyright (c) 2011,2012 UPMC Sorbonne Universites
 *
 * This file is part of ALMOS-MKH.
 *
 * ALMOS-MKH is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2.0 of the License.
 *
 * ALMOS-MKH is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with ALMOS-MKH; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */

#ifndef _SYSFS_H_
#define _SYSFS_H_

#include <hal_types.h>
#include <metafs.h>


struct sysfs_op_s;
struct sysfs_entry_s;
struct sysfs_request_s;


/*******************************************************************************************
 * This structure TODO
 ******************************************************************************************/

typedef struct sysfs_context_s
{
	void * empty;
}
sysfs_context_t;

/*******************************************************************************************
 * Operations definition must be provided by underlying kernel subsystem.
 ******************************************************************************************/

typedef error_t (sysfs_request_func_t)( struct sysfs_entry_s   * entry, 
                                        struct sysfs_request_s * rq, 
                                        uint32_t               * offset );

typedef struct sysfs_op_s
{
	sysfs_request_func_t *open;
	sysfs_request_func_t *read;
	sysfs_request_func_t *write;
	sysfs_request_func_t *close;
}
sysfs_op_t;

/*******************************************************************************************
 * This structure TODO  
 ******************************************************************************************/

typedef struct sysfs_request_s
{
	void     * data;
	uint32_t   count;
	uint8_t    buffer[CONFIG_SYSFS_BUFFER_SIZE];
}
sysfs_request_t;

/*******************************************************************************************
 * This structure TODO  
 ******************************************************************************************/

typedef struct sysfs_entry_s
{
	sysfs_op_t   op;
	metafs_t     node;
}
sysfs_entry_t;

/*******************************************************************************************
 *        Extern Variable : sysfs root node.
 ******************************************************************************************/

extern sysfs_entry_t       sysfs_root_entry;

/*******************************************************************************************
 * This function initializes the sysfs root.
 ******************************************************************************************/
static inline void sysfs_root_init()
{
    if( CONFIG_ROOTFS_IS_VFAT ) metafs_init( &sysfs_root_entry.node , "SYS" );
    else                        metafs_init( &sysfs_root_entry.node , "sys" );
}


/*******************************************************************************************
 * This function initializes a given sysfs entry.
 ******************************************************************************************/
static inline void sysfs_entry_init( sysfs_entry_t * entry,
                                     sysfs_op_t    * op,
                                     char          * name )
{
	metafs_init( &entry->node , name );

	if(op == NULL) return;
  
	entry->op.open  = op->open;
	entry->op.read  = op->read;
	entry->op.write = op->write;
	entry->op.close = op->close;
}

/*******************************************************************************************
 * This function registers a given sysfs entry into it's parent list.
 ******************************************************************************************/
static inline void sysfs_entry_register( sysfs_entry_t * parent,
                                         sysfs_entry_t * entry )
{
	metafs_register( &parent->node , &entry->node );
}

/*******************************************************************************************
 * This function removes a given sysfs entry from it's parent list.
 ******************************************************************************************/
static inline void sysfs_entry_unregister( sysfs_entry_t * parent,
                                           sysfs_entry_t * entry )
{
	metafs_unregister( &parent->node , &entry->node );
}

#endif	/* _SYSFS_H_ */
