/*
 * sys_thread_exit.c - terminates the execution of calling thread
 * 
 * Authors   Alain Greiner (2016,2017,2018)
 *
 * Copyright (c) UPMC Sorbonne Universites
 *
 * This file is part of ALMOS-MKH.
 *
 * ALMOS-MKH is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2.0 of the License.
 *
 * ALMOS-MKH is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with ALMOS-MKH; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */

#include <hal_kernel_types.h>
#include <hal_irqmask.h>
#include <thread.h>
#include <process.h>
#include <core.h>
#include <vmm.h>
#include <scheduler.h>
#include <printk.h>

////////////////////////////////////////
int sys_thread_exit( void * exit_value )
{
    reg_t       save_sr;    // required to enable IRQs
    xptr_t      owner_xp;   // extended pointer on owner process
 
	thread_t  * this      = CURRENT_THREAD;
    trdid_t     trdid     = this->trdid;
    process_t * process   = this->process;
    pid_t       pid       = process->pid;
    cxy_t       owner_cxy = CXY_FROM_PID( pid );

    // check exit_value argument
    if( exit_value != NULL )
    {

#if DEBUG_SYSCALLS_ERROR
printk("\n[ERROR] in %s : exit_value argument must be NULL / thread %x in process %x\n",
__FUNCTION__ , this , pid );
#endif
        this->errno = EINVAL;
        return -1;
    }

#if DEBUG_SYS_THREAD_EXIT
uint64_t     tm_start;
uint64_t     tm_end;
tm_start = hal_get_cycles();
if( DEBUG_SYS_THREAD_EXIT < tm_start )
printk("\n[DBG] %s : thread %x enter / process %x / cycle %d\n",
__FUNCTION__ , this, pid , (uint32_t)tm_start );
#endif

    // If calling thread is the main thread, the process must be deleted. 
    // This require to delete all process threads and synchronise with parent process
    if( (local_cxy == owner_cxy) && (LTID_FROM_TRDID(trdid) == 0) )
    {
        // get extended pointer on owner cluster
        owner_xp = cluster_get_owner_process_from_pid( pid );

        // mark for delete all threads but the main 
        hal_enable_irq( &save_sr );
        process_sigaction( pid , DELETE_ALL_THREADS );
        hal_restore_irq( save_sr );

        // remove process from TXT list
        process_txt_detach( owner_xp );

        // block the main thread 
        thread_block( XPTR( local_cxy , this ) , THREAD_BLOCKED_GLOBAL );

        // atomically update owner process descriptor term_state to ask
        // the parent process sys_wait() function to delete the main thread 
        hal_remote_atomic_or( XPTR( local_cxy , &process->term_state ) ,
                              PROCESS_TERM_EXIT );
    }
    else
    {
        // block calling thread and mark it for delete,
        thread_delete( XPTR( local_cxy , this ) , pid , false );
    }

#if DEBUG_SYS_THREAD_EXIT
tm_end = hal_get_cycles();
if( DEBUG_SYS_THREAD_EXIT < tm_end )
printk("\n[DBG] %s : thread %x exit / process %x / cost %d / cycle %d\n",
__FUNCTION__, this, pid, (uint32_t)(tm_end - tm_start), (uint32_t)tm_end );
#endif

    // deschedule <=> suicide, because blocked by thread_delete()
    sched_yield( "suicide after thread_exit" );
    
    return 0;   // never executed but required by compiler

}  // end sys_thread exit
