/*
 * scheduler.c - Core scheduler implementation.
 * 
 * Author    Alain Greiner (2016)
 *
 * Copyright (c)  UPMC Sorbonne Universites
 *
 * This file is part of ALMOS-MKH.
 *
 * ALMOS-MKH. is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2.0 of the License.
 *
 * ALMOS-MKH. is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with ALMOS-MKH.; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */

#include <kernel_config.h>
#include <hal_kernel_types.h>
#include <hal_switch.h>
#include <hal_irqmask.h>
#include <hal_context.h>
#include <printk.h>
#include <list.h>
#include <core.h>
#include <thread.h>
#include <chdev.h>
#include <scheduler.h>


///////////////////////////////////////////////////////////////////////////////////////////
// Extern global variables
///////////////////////////////////////////////////////////////////////////////////////////

uint32_t   idle_thread_count;
uint32_t   idle_thread_count_active;

extern chdev_directory_t    chdev_dir;          // allocated in kernel_init.c file
extern uint32_t             switch_save_sr[];   // allocated in kernel_init.c file

////////////////////////////////
void sched_init( core_t * core )
{
    scheduler_t * sched = &core->scheduler;

    sched->u_threads_nr   = 0;
    sched->k_threads_nr   = 0;

    sched->current        = CURRENT_THREAD;
    sched->idle           = NULL;               // initialized in kernel_init()
    sched->u_last         = NULL;               // initialized in sched_register_thread()
    sched->k_last         = NULL;               // initialized in sched_register_thread()

    // initialise threads lists
    list_root_init( &sched->u_root );
    list_root_init( &sched->k_root );

    // init spinlock
    spinlock_init( &sched->lock );

    sched->req_ack_pending = false;             // no pending request
    sched->trace           = false;             // context switches trace desactivated

}  // end sched_init()

////////////////////////////////////////////
void sched_register_thread( core_t   * core,
                            thread_t * thread )
{
    scheduler_t * sched = &core->scheduler;
    thread_type_t type  = thread->type;

    // take lock protecting sheduler lists
    spinlock_lock( &sched->lock );

    if( type == THREAD_USER )
    {
        list_add_last( &sched->u_root , &thread->sched_list );
        sched->u_threads_nr++;
        if( sched->u_last == NULL ) sched->u_last = &thread->sched_list;
    }
    else // kernel thread
    {
        list_add_last( &sched->k_root , &thread->sched_list );
        sched->k_threads_nr++;
        if( sched->k_last == NULL ) sched->k_last = &thread->sched_list; 
    }

    // release lock 
    hal_fence();
    spinlock_unlock( &sched->lock );

}  // end sched_register_thread()

//////////////////////////////////////////////
thread_t * sched_select( scheduler_t * sched )
{
    thread_t     * thread;
    list_entry_t * current;
    list_entry_t * last;
    list_entry_t * root;
    bool_t         done;
    uint32_t       count;

    // take lock protecting sheduler lists
    spinlock_lock( &sched->lock );

    // first : scan the kernel threads list if not empty 
    if( list_is_empty( &sched->k_root ) == false )
    {
        root    = &sched->k_root;
        last    = sched->k_last;
        done    = false;
        count   = 0;
        current = last;

        while( done == false )
        {
            assert( (count < sched->k_threads_nr), "bad kernel threads list" );

            // get next entry in kernel list
            current = current->next;

            // check exit condition
            if( current == last ) done = true;

            // skip the root that does not contain a thread
            if( current == root ) continue;
            else                  count++;

            // get thread pointer for this entry
            thread = LIST_ELEMENT( current , thread_t , sched_list );

            // select kernel thread if non blocked and non THREAD_IDLE
            if( (thread->blocked == 0)  && (thread->type != THREAD_IDLE) )
            {
                spinlock_unlock( &sched->lock );
                return thread;
            }
        } // end loop on kernel threads
    } // end kernel threads

    // second : scan the user threads list if not empty 
    if( list_is_empty( &sched->u_root ) == false )
    {
        root    = &sched->u_root;
        last    = sched->u_last;
        done    = false;
        count   = 0;
        current = last;

        while( done == false )
        {
            assert( (count < sched->u_threads_nr), "bad user threads list" );

            // get next entry in user list
            current = current->next;

            // check exit condition
            if( current == last ) done = true;

            // skip the root that does not contain a thread
            if( current == root ) continue;
            else                  count++;

            // get thread pointer for this entry
            thread = LIST_ELEMENT( current , thread_t , sched_list );

            // select thread if non blocked
            if( thread->blocked == 0 )
            {
                spinlock_unlock( &sched->lock );
                return thread;
            }
        } // end loop on user threads
    } // end user threads

    // third : return idle thread if no other runnable thread
    spinlock_unlock( &sched->lock );
    return sched->idle;

}  // end sched_select()

///////////////////////////////////////////
void sched_handle_signals( core_t * core )
{

    list_entry_t * iter;
    list_entry_t * root;
    thread_t     * thread;
    process_t    * process;
    bool_t         last_thread;

    // get pointer on scheduler
    scheduler_t  * sched = &core->scheduler;

    // get pointer on user threads root
    root = &sched->u_root;

    // take lock protecting threads lists
    spinlock_lock( &sched->lock );

    // We use a while to scan the user threads, to control the iterator increment,
    // because some threads will be destroyed, and we cannot use a LIST_FOREACH()

    // initialise list iterator
    iter = root->next;

    // scan all user threads
    while( iter != root )
    {
        // get pointer on thread
        thread = LIST_ELEMENT( iter , thread_t , sched_list );

        // increment iterator
        iter = iter->next;

        // handle REQ_ACK 
        if( thread->flags & THREAD_FLAG_REQ_ACK )
        {
            // check thread blocked
            assert( (thread->blocked & THREAD_BLOCKED_GLOBAL) , 
            "thread not blocked" );
 
            // decrement response counter
            hal_atomic_add( thread->ack_rsp_count , -1 );

            // reset REQ_ACK in thread descriptor
            thread_reset_req_ack( thread );
        }

        // handle REQ_DELETE
        if( thread->flags & THREAD_FLAG_REQ_DELETE )
        {
            // get thread process descriptor
            process = thread->process;

	        // release FPU if required
	        if( thread->core->fpu_owner == thread )  thread->core->fpu_owner = NULL;

            // remove thread from scheduler (scheduler lock already taken)
            uint32_t threads_nr = sched->u_threads_nr;

            assert( (threads_nr != 0) , "u_threads_nr cannot be 0\n" );

            sched->u_threads_nr = threads_nr - 1;
            list_unlink( &thread->sched_list );
            if( sched->u_last == &thread->sched_list )
            {
                if( threads_nr == 1 ) 
                {
                    sched->u_last = NULL;
                }
                else if( sched->u_root.next == &thread->sched_list )
                {
                    sched->u_last = sched->u_root.pred;
                }
                else
                {
                    sched->u_last = sched->u_root.next;
                }
            }

            // delete thread descriptor
            last_thread = thread_destroy( thread );

#if DEBUG_SCHED_HANDLE_SIGNALS
uint32_t cycle = (uint32_t)hal_get_cycles();
if( DEBUG_SCHED_HANDLE_SIGNALS < cycle )
printk("\n[DBG] %s : thread %x in process %x on core[%x,%d] deleted / cycle %d\n",
__FUNCTION__ , thread->trdid , process->pid , local_cxy , thread->core->lid , cycle );
#endif
            // destroy process descriptor if no more threads
            if( last_thread ) 
            {
                // delete process    
                process_destroy( process );

#if DEBUG_SCHED_HANDLE_SIGNALS
cycle = (uint32_t)hal_get_cycles();
if( DEBUG_SCHED_HANDLE_SIGNALS < cycle )
printk("\n[DBG] %s : process %x in cluster %x deleted / cycle %d\n",
__FUNCTION__ , process->pid , local_cxy , cycle );
#endif
            }
        }
    }

    // release lock 
    hal_fence();
    spinlock_unlock( &sched->lock );

} // end sched_handle_signals()

////////////////////////////////
void sched_yield( const char * cause )
{
    thread_t    * next;
    thread_t    * current = CURRENT_THREAD;
    core_t      * core    = current->core;
    scheduler_t * sched   = &core->scheduler;
 
#if (DEBUG_SCHED_YIELD & 0x1)
if( sched->trace )
sched_display( core->lid );
#endif

    // delay the yield if current thread has locks
    if( (current->local_locks != 0) || (current->remote_locks != 0) )
    {
        current->flags |= THREAD_FLAG_SCHED;
        return;
    }

    // enter critical section / save SR in current thread descriptor
    hal_disable_irq( &CURRENT_THREAD->save_sr );

    // loop on threads to select next thread 
    next = sched_select( sched );

    // check next thread kernel_stack overflow
    assert( (next->signature == THREAD_SIGNATURE),
    "kernel stack overflow for thread %x on core[%x,%d] \n", next, local_cxy, core->lid );

    // check next thread attached to same core as the calling thread
    assert( (next->core == current->core),
    "next core %x != current core %x\n", next->core, current->core );

    // check next thread not blocked when type != IDLE
    assert( ((next->blocked == 0) || (next->type == THREAD_IDLE)) ,
    "next thread %x (%s) is blocked on core[%x,%d]\n", 
    next->trdid , thread_type_str(next->type) , local_cxy , core->lid );

    // switch contexts and update scheduler state if next != current
	if( next != current )
    {

#if DEBUG_SCHED_YIELD
if( sched->trace )
printk("\n[DBG] %s : core[%x,%d] / cause = %s\n"
"      thread %x (%s) (%x,%x) => thread %x (%s) (%x,%x) / cycle %d\n",
__FUNCTION__, local_cxy, core->lid, cause, 
current, thread_type_str(current->type), current->process->pid, current->trdid,next ,
thread_type_str(next->type) , next->process->pid , next->trdid , (uint32_t)hal_get_cycles() );
#endif

        // update scheduler 
        sched->current = next;
        if( next->type == THREAD_USER ) sched->u_last = &next->sched_list;
        else                            sched->k_last = &next->sched_list;

        // handle FPU ownership
	    if( next->type == THREAD_USER )
        {
	        if( next == current->core->fpu_owner )  hal_fpu_enable();
	        else                                    hal_fpu_disable();
        }

        // switch CPU from current thread context to new thread context
        hal_do_cpu_switch( current->cpu_context, next->cpu_context );
    }
    else
    {

#if DEBUG_SCHED_YIELD 
if( sched->trace )
printk("\n[DBG] %s : core[%x,%d] / cause = %s\n"
"      thread %x (%s) (%x,%x) continue / cycle %d\n",
__FUNCTION__, local_cxy, core->lid, cause, current, thread_type_str(current->type),
current->process->pid, current->trdid, (uint32_t)hal_get_cycles() );
#endif

    }

    // handle pending requests for all threads executing on this core.
    sched_handle_signals( core );

    // exit critical section / restore SR from current thread descriptor
    hal_restore_irq( CURRENT_THREAD->save_sr );

}  // end sched_yield()


///////////////////////////////
void sched_display( lid_t lid )
{
    list_entry_t * iter;
    thread_t     * thread;
    uint32_t       save_sr;

    assert( (lid < LOCAL_CLUSTER->cores_nr), "illegal core index %d\n", lid);

    core_t       * core    = &LOCAL_CLUSTER->core_tbl[lid];
    scheduler_t  * sched   = &core->scheduler;
    
    // get pointers on TXT0 chdev
    xptr_t    txt0_xp  = chdev_dir.txt_tx[0];
    cxy_t     txt0_cxy = GET_CXY( txt0_xp );
    chdev_t * txt0_ptr = GET_PTR( txt0_xp );

    // get extended pointer on remote TXT0 chdev lock
    xptr_t  lock_xp = XPTR( txt0_cxy , &txt0_ptr->wait_lock );

    // get TXT0 lock in busy waiting mode
    remote_spinlock_lock_busy( lock_xp , &save_sr );

    nolock_printk("\n***** threads on core[%x,%d] / current %x / cycle %d\n",
    local_cxy , core->lid, sched->current, (uint32_t)hal_get_cycles() );

    // display kernel threads
    LIST_FOREACH( &sched->k_root , iter )
    {
        thread = LIST_ELEMENT( iter , thread_t , sched_list );
        if (thread->type == THREAD_DEV) 
        {
            nolock_printk(" - %s / pid %X / trdid %X / desc %X / block %X / flags %X / %s\n",
            thread_type_str( thread->type ), thread->process->pid, thread->trdid,
            thread, thread->blocked, thread->flags, thread->chdev->name );
        }
        else
        {
            nolock_printk(" - %s / pid %X / trdid %X / desc %X / block %X / flags %X\n",
            thread_type_str( thread->type ), thread->process->pid, thread->trdid,
            thread, thread->blocked, thread->flags );
        }
    }

    // display user threads
    LIST_FOREACH( &sched->u_root , iter )
    {
        thread = LIST_ELEMENT( iter , thread_t , sched_list );
        nolock_printk(" - %s / pid %X / trdid %X / desc %X / block %X / flags %X\n",
        thread_type_str( thread->type ), thread->process->pid, thread->trdid,
        thread, thread->blocked, thread->flags );
    }

    // release TXT0 lock
    remote_spinlock_unlock_busy( lock_xp , save_sr );

}  // end sched_display()

/////////////////////////////////////
void sched_remote_display( cxy_t cxy,
                           lid_t lid )
{
    thread_t     * thread;
    uint32_t       save_sr;

    // check cxy
    bool_t undefined = cluster_is_undefined( cxy );
    assert( (undefined == false), "illegal cluster %x\n", cxy );

    // check lid
    uint32_t cores = hal_remote_lw( XPTR( cxy , &LOCAL_CLUSTER->cores_nr ) );
    assert( (lid < cores), "illegal core index %d\n", lid);

    // get local pointer on target scheduler
    core_t      * core  = &LOCAL_CLUSTER->core_tbl[lid];
    scheduler_t * sched = &core->scheduler;

    // get local pointer on current thread in target scheduler
    thread_t * current = hal_remote_lpt( XPTR( cxy, &sched->current ) );

    // get local pointer on the first kernel and user threads list_entry
    list_entry_t * k_entry = hal_remote_lpt( XPTR( cxy , &sched->k_root.next ) );
    list_entry_t * u_entry = hal_remote_lpt( XPTR( cxy , &sched->u_root.next ) );
    
    // get pointers on TXT0 chdev
    xptr_t    txt0_xp  = chdev_dir.txt_tx[0];
    cxy_t     txt0_cxy = GET_CXY( txt0_xp );
    chdev_t * txt0_ptr = GET_PTR( txt0_xp );

    // get extended pointer on remote TXT0 chdev lock
    xptr_t  lock_xp = XPTR( txt0_cxy , &txt0_ptr->wait_lock );

    // get TXT0 lock in busy waiting mode
    remote_spinlock_lock_busy( lock_xp , &save_sr );

    // display header
    nolock_printk("\n***** threads on core[%x,%d] / current %x / cycle %d\n",
    cxy , lid, current, (uint32_t)hal_get_cycles() );

    // display kernel threads
    while( k_entry != &sched->k_root )
    {
        // get local pointer on kernel_thread
        thread = LIST_ELEMENT( k_entry , thread_t , sched_list );

        // get relevant thead info
        thread_type_t type    = hal_remote_lw ( XPTR( cxy , &thread->type ) );
        trdid_t       trdid   = hal_remote_lw ( XPTR( cxy , &thread->trdid ) );
        uint32_t      blocked = hal_remote_lw ( XPTR( cxy , &thread->blocked ) );
        uint32_t      flags   = hal_remote_lw ( XPTR( cxy , &thread->flags ) );
        process_t *   process = hal_remote_lpt( XPTR( cxy , &thread->process ) );
        pid_t         pid     = hal_remote_lw ( XPTR( cxy , &process->pid ) );

        // display thread info
        if (type == THREAD_DEV) 
        {
            char      name[16];
            chdev_t * chdev = hal_remote_lpt( XPTR( cxy , &thread->chdev ) );
            hal_remote_strcpy( XPTR( local_cxy , name ), XPTR( cxy , &chdev->name ) );

            nolock_printk(" - %s / pid %X / trdid %X / desc %X / block %X / flags %X / %s\n",
            thread_type_str( type ), pid, trdid, thread, blocked, flags, name );
        }
        else
        {
            nolock_printk(" - %s / pid %X / trdid %X / desc %X / block %X / flags %X\n",
            thread_type_str( type ), pid, trdid, thread, blocked, flags );
        }

        // get next remote kernel thread list_entry
        k_entry = hal_remote_lpt( XPTR( cxy , &k_entry->next ) );
    }

    // display user threads
    while( u_entry != &sched->u_root )
    {
        // get local pointer on user_thread
        thread = LIST_ELEMENT( u_entry , thread_t , sched_list );

        // get relevant thead info
        thread_type_t type    = hal_remote_lw ( XPTR( cxy , &thread->type ) );
        trdid_t       trdid   = hal_remote_lw ( XPTR( cxy , &thread->trdid ) );
        uint32_t      blocked = hal_remote_lw ( XPTR( cxy , &thread->blocked ) );
        uint32_t      flags   = hal_remote_lw ( XPTR( cxy , &thread->flags ) );
        process_t *   process = hal_remote_lpt( XPTR( cxy , &thread->process ) );
        pid_t         pid     = hal_remote_lw ( XPTR( cxy , &process->pid ) );

        nolock_printk(" - %s / pid %X / trdid %X / desc %X / block %X / flags %X\n",
        thread_type_str( type ), pid, trdid, thread, blocked, flags );

        // get next user thread list_entry
        u_entry = hal_remote_lpt( XPTR( cxy , &u_entry->next ) );
    }

    // release TXT0 lock
    remote_spinlock_unlock_busy( lock_xp , save_sr );

}  // end sched_remote_display()

