/*
 * sys_stat.c - Return statistics on a file or directory.
 * 
 * Author    Alain Greiner  (2016,2017,2018)
 *
 * Copyright (c) UPMC Sorbonne Universites
 *
 * This file is part of ALMOS-MKH.
 *
 * ALMOS-MKH is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2.0 of the License.
 *
 * ALMOS-MKH is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with ALMOS-MKH; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */

#include <hal_kernel_types.h>
#include <hal_uspace.h>
#include <hal_special.h>
#include <errno.h>
#include <thread.h>
#include <printk.h>
#include <vfs.h>
#include <vmm.h>
#include <process.h>

#include <syscalls.h>

/////////////////////////////////////
int sys_stat( const char  * pathname,
              struct stat * u_stat )
{
    error_t       error;
    vseg_t      * vseg;         // for user space checking
    struct stat   k_stat;       // kernel space 
    xptr_t        file_xp;
    char          kbuf[CONFIG_VFS_MAX_PATH_LENGTH];
	
	thread_t  * this    = CURRENT_THREAD;
	process_t * process = this->process;

    // check stat structure in user space
    error = vmm_get_vseg( process , (intptr_t)u_stat , &vseg );

	if( error )
	{

#if DEBUG_SYCALL_ERROR
printk("\n[ERROR] in %s : stat structure unmapped %x / thread %x / process %x\n",
__FUNCTION__ , (intptr_t)u_stat , this->trdid , process->pid );
vmm_display( process , false );
#endif
		this->errno = EINVAL;
		return -1;
	}	

    // check pathname length
    if( hal_strlen_from_uspace( pathname ) >= CONFIG_VFS_MAX_PATH_LENGTH )
    {
        printk("\n[ERROR] in %s : pathname too long\n", __FUNCTION__ );
        this->errno = ENFILE;
        return -1;
    }

    // copy pathname in kernel space
    hal_strcpy_from_uspace( kbuf , pathname , CONFIG_VFS_MAX_PATH_LENGTH );

    // get cluster and local pointer on reference process
    xptr_t      ref_xp  = process->ref_xp;
    process_t * ref_ptr = (process_t *)GET_PTR( ref_xp );
    cxy_t       ref_cxy = GET_CXY( ref_xp );

    // get extended pointer on cwd inode
    xptr_t cwd_xp = hal_remote_lwd( XPTR( ref_cxy , &ref_ptr->vfs_cwd_xp ) );

    // get the cwd lock in read mode from reference process
    remote_rwlock_rd_lock( XPTR( ref_cxy , &ref_ptr->cwd_lock ) );

    // get extended pointer on remote file descriptor
    error = vfs_lookup( cwd_xp,
                        pathname,
                        0,
                        &file_xp );

    // release the cwd lock
    remote_rwlock_rd_unlock( XPTR( ref_cxy , &ref_ptr->cwd_lock ) );

    if( error )
    {
        printk("\n[ERROR] in %s : cannot found file <%s> for thread %x in process %x\n",
               __FUNCTION__ , pathname , this->trdid , process->pid );
        this->errno = error;
        return -1;
    }

    // call VFS function to get stat info
    error = vfs_stat( file_xp,
                      &k_stat );
    if( error )
	{
        printk("\n[ERROR] in %s : cannot get stats for file %s\n",
               __FUNCTION__ , pathname );
		this->errno = error;
		return -1;
	}
   
    // copy k_stat to u_stat 
    hal_copy_to_uspace( u_stat , &k_stat , sizeof(struct stat) );

    hal_fence();

	return 0;

}  // end sys_stat()

