/*
 * scheduler.h - Core scheduler definition.
 * 
 * Author    Alain Greiner (2016)
 *
 * Copyright (c) UPMC Sorbonne Universites
 *
 * This file is part of ALMOS-MKH.
 *
 * ALMOS-MKH is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2.0 of the License.
 *
 * ALMOS-MKH is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with ALMOS-MKH; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */

#ifndef _SCHEDULER_H_
#define _SCHEDULER_H_

#include <hal_kernel_types.h>
#include <list.h>
#include <spinlock.h>

/****  Forward declarations  ****/

struct core_s;
struct thread_s;

/*********************************************************************************************
 * This structure define the scheduler associated to a given core.
 ********************************************************************************************/

typedef struct scheduler_s
{
    spinlock_t        lock;            /*! lock protecting lists of threads                 */
    uint16_t          u_threads_nr;    /*! total number of attached user threads            */
    uint16_t          k_threads_nr;    /*! total number of attached kernel threads          */
    list_entry_t      u_root;          /*! root of list of user threads                     */
    list_entry_t      k_root;          /*! root of list of kernel threads                   */
    list_entry_t    * u_last;          /*! pointer on list_entry for last executed k_thread */
    list_entry_t    * k_last;          /*! pointer on list entry for last executed u_thread */
    struct thread_s * idle;            /*! pointer on idle thread                           */
    struct thread_s * current;         /*! pointer on current running thread                */
    volatile bool_t   req_ack_pending; /*! sequencialize ack requests when true             */
    bool_t            trace;           /*! context switches trace activated if true         */
}
scheduler_t;

/*********************************************************************************************
 *  This function initialises the scheduler for a given core.
 ********************************************************************************************/  
void sched_init( struct core_s * core );

/*********************************************************************************************
 * This function atomically register a new thread in a given core scheduler.
 *********************************************************************************************
 * @ core    : local pointer on the core descriptor.
 * @ thread  : local pointer on the thread descriptor.
 ********************************************************************************************/  
void sched_register_thread( struct core_s   * core,
                            struct thread_s * thread );

/********************************************************************************************* 
 * This function is the only method to make a context switch. It is called in cas of TICK,
 * or when when a thread explicitely requires a scheduling.
 * It handles the pending signals for all threads attached to the core running the calling
 * thread, and calls the sched_select() function to select a new thread.
 * The cause argument is only used for debug by the sched_display() function, and 
 * indicates the scheduling cause.
 *********************************************************************************************
 * @ cause    : character string defining the scheduling cause.
 ********************************************************************************************/
void sched_yield( const char * cause );

/*********************************************************************************************
 * This function scan all threads attached to a given scheduler, and executes the relevant
 * actions for pending THREAD_FLAG_REQ_ACK or THREAD_FLAG_REQ_DELETE requests. 
 * It is called in by the sched_yield() function, with IRQ disabled.
 * - REQ_ACK : it checks that target thread is blocked, decrements the response counter
 *   to acknowledge the client thread, and reset the pending request. 
 * - REQ_DELETE : it detach the target thread from parent if attached, detach it from
 *   the process, remove it from scheduler, release memory allocated to thread descriptor,
 *   and destroy the process descriptor it the target thread was the last thread.
 *********************************************************************************************
 * @ core    : local pointer on the core descriptor.
 ********************************************************************************************/
void sched_handle_signals( struct core_s * core );

/*********************************************************************************************
 * This function does NOT modify the scheduler state.
 * It just select a thread in the list of attached threads, implementing the following 
 * three steps policy: 
 * 1) It scan the list of kernel threads, from the next thread after the last executed one,
 *    and returns the first runnable found : not IDLE, not blocked, client queue not empty.
 *    It can be the current thread.
 * 2) If no kernel thread found, it scan the list of user thread, from the next thread after
 *    the last executed one, and returns the first runable found : not blocked.
 *    It can be the current thread.
 * 3) If no runable thread found, it returns the idle thread.
 *********************************************************************************************
 * @ core    : local pointer on scheduler.
 * @ returns pointer on selected thread descriptor
 ********************************************************************************************/
struct thread_s * sched_select( struct scheduler_s * sched );

/********************************************************************************************* 
 * This debug function displays on TXT0 the internal state of a local scheduler, 
 * identified by the core local index <lid>.
 *********************************************************************************************
 * @ lid      : local index of target core.
 ********************************************************************************************/
void sched_display( lid_t lid );

/********************************************************************************************* 
 * This debug function displays on TXT0 the internal state of a scheduler, 
 * identified by the target cluster identifier <cxy> and the core local index <lid>.
 * It can be called by a thread running in any cluster, as it uses remote accesses,
 * to scan the scheduler local lists of threads.
 *********************************************************************************************
 * @ cxy      : target cluster identifier
 * @ lid      : local index of target core.
 ********************************************************************************************/
void sched_remote_display( cxy_t  cxy,
                           lid_t  lid );

#endif	/* _SCHEDULER_H_ */
