/*
 * remote_rwlock.c - kernel remote rwlock implementation.
 * 
 * Authors    Alain   Greiner (2016,2017,2018)
 *
 * Copyright (c) UPMC Sorbonne Universites
 *
 * This file is part of ALMOS-MKH.
 *
 * ALMOS-MKH is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2.0 of the License.
 *
 * ALMOS-MKH is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with ALMOS-MKH; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */

#include <hal_kernel_types.h>
#include <hal_remote.h>
#include <hal_irqmask.h>
#include <thread.h>
#include <printk.h>
#include <cluster.h>
#include <scheduler.h>
#include <remote_rwlock.h>

///////////////////////////////////////////
void remote_rwlock_init( xptr_t lock_xp )
{ 
    remote_rwlock_t * lock_ptr = GET_PTR( lock_xp );
    cxy_t             lock_cxy = GET_CXY( lock_xp );

    hal_remote_sw ( XPTR( lock_cxy , &lock_ptr->ticket )  , 0 );
    hal_remote_sw ( XPTR( lock_cxy , &lock_ptr->current ) , 0 );
    hal_remote_sw ( XPTR( lock_cxy , &lock_ptr->count )   , 0 );

#if DEBUG_REMOTE_RWLOCKS
hal_remote_swd( XPTR( lock_cxy , &lock_ptr->owner )   , XPTR_NULL );
xlist_entry_init( XPTR( lock_cxy , &lock_ptr->list ) );
#endif

}

//////////////////////////////////////////////
void remote_rwlock_rd_lock( xptr_t lock_xp )
{ 
	reg_t      mode;
    uint32_t   ticket;

    // get cluster and local pointer on remote_rwlock
    remote_rwlock_t * lock_ptr = GET_PTR( lock_xp );
    cxy_t             lock_cxy = GET_CXY( lock_xp );

    // get local pointer on local thread
    thread_t          * thread_ptr = CURRENT_THREAD;

    // extended pointers on ticket, current, count
    xptr_t              ticket_xp  = XPTR( lock_cxy   , &lock_ptr->ticket );
    xptr_t              current_xp = XPTR( lock_cxy   , &lock_ptr->current );
    xptr_t              count_xp   = XPTR( lock_cxy   , &lock_ptr->count );

    // disable interrupts
    hal_disable_irq( &mode );

    // get next free ticket
    ticket = hal_remote_atomic_add( ticket_xp , 1 );

    // busy waiting loop to take the lock 
	while( ticket != hal_remote_lw( current_xp ) )
	{
        hal_fixed_delay( CONFIG_RWLOCK_DELAY );
	}

    ////////// From here we have the lock  ////////////

    // increment count 
    hal_remote_atomic_add( count_xp , 1 );

    // increment thread.remote_locks
    thread_ptr->remote_locks++;

#if DEBUG_REMOTE_RWLOCKS
xlist_add_first( XPTR( local_cxy , &thread_ptr->xlocks_root ) ,
                 XPTR( lock_cxy ,  &lock_ptr->list ) );
#endif

    // sync
    hal_fence();

    // release lock to allow several simultaneous readers
    hal_remote_atomic_add( current_xp , 1 );

    // enable interrupts
	hal_restore_irq( mode );

}  // end remote_rwlock_rd_lock()

////////////////////////////////////////////////
void remote_rwlock_rd_unlock( xptr_t lock_xp )
{
	reg_t               mode;

    // get cluster and local pointer on remote_rwlock
    remote_rwlock_t * lock_ptr = GET_PTR( lock_xp );
    cxy_t             lock_cxy = GET_CXY( lock_xp );

    // get cluster and local pointer on local thread
    thread_t          * thread_ptr = CURRENT_THREAD;

    // extended pointers on lock->count 
    xptr_t              count_xp = XPTR( lock_cxy   , &lock_ptr->count );

    // disable interrupts
	hal_disable_irq( &mode );
  
    // decrement count
    hal_remote_atomic_add( count_xp , -1 );

    // decrement thread.remote_locks
	thread_ptr->remote_locks--;

#if DEBUG_REMOTE_RWLOCKS
xlist_unlink( XPTR( lock_cxy , &lock_ptr->list ) );
#endif

    // enable interrupts
	hal_restore_irq( mode );
    
    // deschedule if pending request
    thread_check_sched();

}  // end remote_rwlock_rd_unlock()

//////////////////////////////////////////////
void remote_rwlock_wr_lock( xptr_t lock_xp )
{ 
	reg_t      mode;
    uint32_t   ticket;

    // get cluster and local pointer on remote_rwlock
    remote_rwlock_t * lock_ptr = GET_PTR( lock_xp );
    cxy_t             lock_cxy = GET_CXY( lock_xp );

    // get local pointer on local thread
    thread_t          * thread_ptr = CURRENT_THREAD;

    // compute extended pointers on lock->ticket, lock->owner
    xptr_t              ticket_xp  = XPTR( lock_cxy   , &lock_ptr->ticket );
    xptr_t              count_xp   = XPTR( lock_cxy   , &lock_ptr->count );
    xptr_t              current_xp = XPTR( lock_cxy   , &lock_ptr->current );

    // disable interrupts
    hal_disable_irq( &mode );

    // get next free ticket
    ticket = hal_remote_atomic_add( ticket_xp , 1 );

    // loop to take the lock 
	while( ticket != hal_remote_lw( current_xp ) )
	{
        hal_fixed_delay( CONFIG_RWLOCK_DELAY );
	}

    ////////// From here we have the lock  ////////////

    // wait completion of read accesses
    while( hal_remote_lw( count_xp ) != 0 )
    {
        hal_fixed_delay( CONFIG_RWLOCK_DELAY );
    }

#if DEBUG_REMOTE_RWLOCKS
hal_remote_swd( XPTR( lock_cxy  , &lock_ptr->owner ) ,
                XPTR( local_cxy , thread_ptr ) );
xlist_add_first( XPTR( local_cxy , &thread_ptr->xlocks_root ) ,
                 XPTR( lock_cxy  , &lock_ptr->list ) );
#endif    

    // increment thread.remote_locks
    thread_ptr->remote_locks++;

    // enable interrupts
	hal_restore_irq( mode );

}  // end remote_rwlock_wr_lock()

//////////////////////////////////////////////
void remote_rwlock_wr_unlock( xptr_t lock_xp )
{
	reg_t               mode;

    // get cluster and local pointer on remote_rwlock
    remote_rwlock_t * lock_ptr = GET_PTR( lock_xp );
    cxy_t             lock_cxy = GET_CXY( lock_xp );

    // get cluster and local pointer on local thread
    thread_t          * thread_ptr = CURRENT_THREAD;

    // compute extended pointer on lock->ticket 
    xptr_t              current_xp = XPTR( lock_cxy   , &lock_ptr->current );

    // disable interrupts
	hal_disable_irq( &mode );
  
#if CONFIG_LOCKS_OWNER
hal_remote_swd( XPTR( lock_cxy , &lock_ptr->owner ) , XPTR_NULL );
xlist_unlink( XPTR( lock_cxy , &lock_ptr->list ) );
#endif

    // release lock
    hal_remote_atomic_add( current_xp , 1 );

    // decrement thread.remote_locks
	thread_ptr->remote_locks--;

    // enable interrupts
	hal_restore_irq( mode );
    
    // deschedule if pending request
    thread_check_sched();

}  // end remote_rwlock_wr_unlock()

///////////////////////////////////////////
void remote_rwlock_print( xptr_t   lock_xp,
                          char   * comment )
{
    uint32_t     ticket;                // first free ticket index
    uint32_t     current;               // ticket index of current owner 
    uint32_t     count;                 // current number of reader threads

    // get cluster and local pointer on remote_rwlock
    remote_rwlock_t * lock_ptr = GET_PTR( lock_xp );
    cxy_t             lock_cxy = GET_CXY( lock_xp );

    ticket  = hal_remote_lw ( XPTR( lock_cxy , &lock_ptr->ticket ) );
    current = hal_remote_lw ( XPTR( lock_cxy , &lock_ptr->current ) );
    count   = hal_remote_lw ( XPTR( lock_cxy , &lock_ptr->count ) );

    printk("\n*** rwlock <%l> %s : ticket = %d / current = %d / count = %d\n",
           lock_xp , comment , ticket , current , count ); 

}  // end remote_rwlock_print()

