/*
 * sys_read.c - read bytes from a file
 * 
 * Author     Alain Greiner (2016,2017,2018)
 *
 * Copyright (c) UPMC Sorbonne Universites
 *
 * This file is part of ALMOS-MKH.
 *
 * ALMOS-MKH is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2.0 of the License.
 *
 * ALMOS-MKH is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with ALMOS-MKH; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */

#include <kernel_config.h>
#include <hal_kernel_types.h>
#include <hal_uspace.h>
#include <hal_irqmask.h>
#include <hal_special.h>
#include <errno.h>
#include <vfs.h>
#include <vmm.h>
#include <thread.h>
#include <printk.h>
#include <process.h>

#include <syscalls.h>

// TODO: concurrent user page(s) munmap need to be handled [AG] 

extern uint32_t enter_sys_read;
extern uint32_t enter_devfs_read;
extern uint32_t enter_txt_read;
extern uint32_t enter_chdev_cmd_read;
extern uint32_t enter_chdev_server_read;
extern uint32_t enter_tty_cmd_read;
extern uint32_t enter_tty_isr_read;
extern uint32_t exit_tty_isr_read;
extern uint32_t exit_tty_cmd_read;
extern uint32_t exit_chdev_server_read;
extern uint32_t exit_chdev_cmd_read;
extern uint32_t exit_txt_read;
extern uint32_t exit_devfs_read;
extern uint32_t exit_sys_read;


/////////////////////////////////
int sys_read( uint32_t   file_id,
              void     * vaddr,
              uint32_t   count )
{
    error_t      error;
    vseg_t     * vseg;        // required for user space checking
	xptr_t       file_xp;     // remote file extended pointer
    uint32_t     nbytes;      // number of bytes actually read
    reg_t        save_sr;     // required to enable IRQs during syscall

	thread_t  *  this             = CURRENT_THREAD;
	process_t *  process          = this->process;
    xptr_t       process_owner_xp = process->owner_xp;
 
#if DEBUG_SYS_READ
uint64_t     tm_start;
uint64_t     tm_end;
tm_start = hal_get_cycles();
if( DEBUG_SYS_READ < tm_start )
printk("\n[DBG] %s : thread %x in process %x enter / vaddr %x / count %d / cycle %d\n",
__FUNCTION__, this->trdid, process->pid, vaddr, count, (uint32_t)tm_start );
#endif

#if (DEBUG_SYS_READ & 1)
enter_sys_read = (uint32_t)tm_start;
#endif

    // check file_id argument
	if( file_id >= CONFIG_PROCESS_FILE_MAX_NR )
	{

#if DEBUG_SYSCALLS_ERROR
printk("\n[ERROR] in %s : illegal file descriptor index = %d\n", __FUNCTION__ , file_id );
#endif
		this->errno = EBADFD;
		return -1;
	}

    // check user buffer in user space
    error = vmm_get_vseg( process , (intptr_t)vaddr , &vseg );

    if ( error )
    {

#if DEBUG_SYSCALLS_ERROR
printk("\n[ERROR] in %s : user buffer unmapped %x / thread %x / process %x\n",
__FUNCTION__ , (intptr_t)vaddr, this->trdid, process->pid );
vmm_display( process , false );
#endif
		this->errno = EINVAL;
		return -1;
    }

    // enable IRQs
    hal_enable_irq( &save_sr );

    // get extended pointer on remote file descriptor
    file_xp = process_fd_get_xptr( process , file_id );

    if( file_xp == XPTR_NULL )
    {

#if DEBUG_SYSCALLS_ERROR
printk("\n[ERROR] in %s : undefined fd_id %d in process %x\n",
__FUNCTION__ , file_id , process->pid );
#endif
        this->errno = EBADFD;
        return -1;
    }

    // get file descriptor cluster and local pointer
    vfs_file_t * file_ptr = GET_PTR( file_xp );
    cxy_t        file_cxy = GET_CXY( file_xp );

    // check file readable
    uint32_t attr = hal_remote_lw( XPTR( file_cxy , &file_ptr->attr ) );
    if( (attr & FD_ATTR_READ_ENABLE) == 0 )
	{

#if DEBUG_SYSCALLS_ERROR
printk("\n[ERROR] in %s : file %d not readable in process %x\n",
__FUNCTION__ , file_id , process->pid );
#endif
		this->errno = EBADFD;
		return -1;
	}
    
    // get file type
    vfs_inode_type_t type = hal_remote_lw( XPTR( file_cxy , &file_ptr->type ) );

    // action depend on file type
    if( type == INODE_TYPE_FILE )      // check file readable & read from mapper 
    {
        // check file readable
        uint32_t attr = hal_remote_lw( XPTR( file_cxy , &file_ptr->attr ) );

        if( (attr & FD_ATTR_READ_ENABLE) == 0 )
	    {

#if DEBUG_SYSCALLS_ERROR
printk("\n[ERROR] in %s : file %d not readable in process %x\n",
__FUNCTION__ , file_id , process->pid );
#endif
		    this->errno = EBADFD;
		    return -1;
	    }

        // move count bytes from mapper
        nbytes = vfs_user_move( true,               // from mapper to buffer
                                file_xp,
                                vaddr, 
                                count );
    }
    else if( type == INODE_TYPE_DEV )  // check ownership & read from device
    {
        // get cluster and pointers on TXT_RX chdev 
        xptr_t    chdev_xp  = chdev_from_file( file_xp );
        cxy_t     chdev_cxy = GET_CXY( chdev_xp );
        chdev_t * chdev_ptr = GET_PTR( chdev_xp );

        volatile xptr_t    txt_owner_xp;    
        uint32_t           iter = 0;

        while( 1 )
        {
            // extended pointer on TXT owner process
            txt_owner_xp  = hal_remote_lwd( XPTR( chdev_cxy , &chdev_ptr->ext.txt.owner_xp ) );

            // check TXT_RX ownership
            if ( process_owner_xp != txt_owner_xp )
            {
                if( (iter & 0xFFF) == 0 )
                printk("\n[WARNING] in %s : thread %x in process %x wait TXT_RX / cycle %d\n",
                __FUNCTION__, this->trdid, process->pid, (uint32_t)hal_get_cycles() );

                // deschedule without blocking
                sched_yield( "wait TXT_RX ownership" );

                iter++;
            }
            else
            {
                break;
            }
        }

        // move count bytes from device
        nbytes = devfs_user_move( true,             // from device to buffer
                                  file_xp,
                                  vaddr,
                                  count );

    }
    else
    {
        nbytes = 0;
        assert( false , "file type %d non supported yet\n", type );
    }

    if( nbytes != count )
    {

#if DEBUG_SYSCALLS_ERROR
printk("\n[ERROR] in %s cannot read data from file %d in process %x\n",
__FUNCTION__ , file_id , process->pid );
#endif
        this->errno = error;
        return -1;
    }

    // restore IRQs
    hal_restore_irq( save_sr );

    hal_fence();

#if DEBUG_SYS_READ
tm_end = hal_get_cycles();
if( DEBUG_SYS_READ < tm_end )
printk("\n[DBG] %s : thread %x in process %x exit / cycle %d / cost %d\n",
__FUNCTION__ , this->trdid, process->pid, (uint32_t)tm_start, (uint32_t)(tm_end - tm_start) );
#endif

#if (DEBUG_SYS_READ & 1) 
exit_sys_read = (uint32_t)tm_end;

printk("\n***** timing to read one character *****\n"
" - enter_sys_read          = %d / delta %d\n"
" - enter_devfs_read        = %d / delta %d\n"
" - enter_txt_read          = %d / delta %d\n"
" - enter_chdev_cmd_read    = %d / delta %d\n"
" - enter_chdev_server_read = %d / delta %d\n"
" - enter_tty_cmd_read      = %d / delta %d\n"
" - enter_tty_isr_read      = %d / delta %d\n"
" - exit_tty_isr_read       = %d / delta %d\n"
" - exit_tty_cmd_read       = %d / delta %d\n"
" - exit_chdev_server_read  = %d / delta %d\n"
" - exit_chdev_cmd_read     = %d / delta %d\n"
" - exit_txt_read           = %d / delta %d\n"
" - exit_devfs_read         = %d / delta %d\n"
" - exit_sys_read           = %d / delta %d\n",
enter_sys_read          , 0 ,
enter_devfs_read        , enter_devfs_read        - enter_sys_read          ,
enter_txt_read          , enter_txt_read          - enter_devfs_read        ,
enter_chdev_cmd_read    , enter_chdev_cmd_read    - enter_txt_read          ,
enter_chdev_server_read , enter_chdev_server_read - enter_chdev_cmd_read    ,
enter_tty_cmd_read      , enter_tty_cmd_read      - enter_chdev_server_read ,
enter_tty_isr_read      , enter_tty_isr_read      - enter_tty_cmd_read      ,
exit_tty_isr_read       , exit_tty_isr_read       - enter_tty_isr_read      ,
exit_tty_cmd_read       , exit_tty_cmd_read       - exit_tty_isr_read       ,
exit_chdev_server_read  , exit_chdev_server_read  - exit_tty_cmd_read       ,
exit_chdev_cmd_read     , exit_chdev_cmd_read     - exit_chdev_server_read  ,
exit_txt_read           , exit_txt_read           - exit_chdev_cmd_read     ,
exit_devfs_read         , exit_devfs_read         - exit_txt_read           ,
exit_sys_read           , exit_sys_read           - exit_devfs_read         );
#endif
 
	return nbytes;

}  // end sys_read()
