/*
 * hal_vmm.c - Generic Virtual Memory Manager Initialisation for TSAR
 *
 * Authors  Alain Greiner (2016,2017)
 *
 * Copyright (c) UPMC Sorbonne Universites
 *
 * This file is part of ALMOS-MKH.
 *
 * ALMOS-MKH is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2.0 of the License.
 *
 * ALMOS-MKH is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with ALMOS-MKH; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */

#include <kernel_config.h>
#include <hal_kernel_types.h>
#include <hal_vmm.h>
#include <hal_gpt.h>
#include <vseg.h>
#include <xlist.h>
#include <vmm.h>
#include <remote_rwlock.h>

//////////////////////////////////////////////////////////////////////////////////////////
// This file contains the TSAR specific code to initialize the Virtual Memory Manager.
// The "kentry" vseg contains the kernel code executed when a core enter/exit the kernel,
// in case of Interrupt, Exception, or Syscall.
// For the TSAR architecture, the kernel uses physical addresses, and this code must be
// identity mapped. The following function is called by the generic vmm_init() function
// and identity map all pages of the "kentry" vseg.
//////////////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////
error_t  hal_vmm_init( vmm_t * vmm )
{
    error_t error;

    // get pointer on GPT
    gpt_t * gpt = &vmm->gpt;

    // map all pages of "kentry" vseg
    uint32_t vpn;
    uint32_t attr;
    attr = GPT_MAPPED | GPT_SMALL | GPT_EXECUTABLE | GPT_CACHABLE | GPT_GLOBAL;
    for( vpn = CONFIG_VMM_KENTRY_BASE;
         vpn < (CONFIG_VMM_KENTRY_BASE + CONFIG_VMM_KENTRY_SIZE); vpn++ )
    {
        error = hal_gpt_set_pte( gpt,
                                 vpn,
                                 attr,
                                 (local_cxy<<20) | (vpn & 0xFFFFF) );

        if( error ) return error;
    }

    // get extended pointer on lock protecting the VSL
    xptr_t lock_xp = XPTR( local_cxy , &vmm->vsegs_lock );

    // get VSL lock 
    remote_rwlock_wr_acquire( lock_xp );

    // scan the VSL to found the "kentry" vseg
    xptr_t         root_xp = XPTR( local_cxy , &vmm->vsegs_root );
    xptr_t         iter_xp;
    xptr_t         vseg_xp;
    vseg_t       * vseg;
    bool_t         found = false;
 
    XLIST_FOREACH( root_xp , iter_xp )
    {
        vseg_xp = XLIST_ELEMENT( iter_xp , vseg_t , xlist );
        vseg    = (vseg_t *)GET_PTR( vseg_xp );

        // set the IDENT flag in "kentry" vseg descriptor
        if( vseg->vpn_base == CONFIG_VMM_KENTRY_BASE )
        {
            vseg->flags |= VSEG_IDENT;
            found = true;
            break;
        }
    }

    // release the VSL lock
    remote_rwlock_wr_release( lock_xp );

    if( found == false ) return error;

    return 0;

}  // end hal_vmm_init()


