/*
 * sys_fbf.c - Acces the frame buffer peripheral.
 * 
 * Authors     Alain Greiner (2016,2017,2018,2019)
 *
 * Copyright (c) UPMC Sorbonne Universites
 *
 * This file is part of ALMOS-MKH.
 *
 * ALMOS-MKH is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2.0 of the License.
 *
 * ALMOS-MKH is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with ALMOS-MKH; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */

#include <hal_kernel_types.h>
#include <hal_uspace.h>
#include <hal_vmm.h>
#include <shared_fbf.h>
#include <dev_fbf.h>
#include <errno.h>
#include <thread.h>
#include <printk.h>
#include <vmm.h>
#include <syscalls.h>

//////////////////////////////////
int sys_fbf( uint32_t   operation,
             void     * arg0,           
             void     * arg1,
             void     * arg2 )
{
	vseg_t         * vseg;        // for vaddr check
    error_t          error;

    thread_t       * this    = CURRENT_THREAD;
    process_t      * process = this->process;

#if (DEBUG_SYS_FBF || CONFIG_INSTRUMENTATION_SYSCALLS)
uint64_t     tm_start = hal_get_cycles();
#endif

#if DEBUG_SYS_FBF
if( DEBUG_SYS_FBF < tm_start )
printk("\n[%s] thread[%x,%x] enter for %s / cycle %d\n",
__FUNCTION__, process->pid, this->trdid, dev_fbf_cmd_str( operation ), (uint32_t)tm_start );
#endif

    // execute requested operation
	switch( operation )
	{
        ////////////////////
	    case FBF_GET_CONFIG:
        {
            uint32_t width;
            uint32_t height;
            uint32_t type;

            // check arg0 (width) in user vspace
	        error = vmm_get_vseg( process , (intptr_t)arg0 , &vseg );
	        if( error )
            {

#if DEBUG_SYSCALLS_ERROR
printk("\n[ERROR] in %s : unmapped arg0 %x for %s / thread[%x,%x]\n",
__FUNCTION__ , arg0, dev_fbf_cmd_str(operation), process->pid, this->trdid );
#endif
                this->errno = EINVAL;
                return -1;
            }

            // check arg1 (height) in user vspace
	        error = vmm_get_vseg( process , (intptr_t)arg1 , &vseg );
	        if( error )
            {

#if DEBUG_SYSCALLS_ERROR
printk("\n[ERROR] in %s : unmapped arg1 %x for %s / thread[%x,%x]\n",
__FUNCTION__ , arg1, dev_fbf_cmd_str(operation), process->pid, this->trdid );
#endif
                this->errno = EINVAL;
                return -1;
            }

            // check arg2 (type) in user vspace
	        error = vmm_get_vseg( process , (intptr_t)arg2 , &vseg );
	        if( error )
            {

#if DEBUG_SYSCALLS_ERROR
printk("\n[ERROR] in %s : unmapped arg2 %x for %s / thread[%x,%x]\n",
__FUNCTION__ , arg2, dev_fbf_cmd_str(operation), process->pid, this->trdid );
#endif
                this->errno = EINVAL;
                return -1;
            }

            // call relevant kernel function to get  config
            dev_fbf_get_config( &width , &height , &type );

            // transfer to user space
            hal_copy_to_uspace( arg0 , XPTR( local_cxy , &width )  , sizeof(uint32_t) );
            hal_copy_to_uspace( arg1 , XPTR( local_cxy , &height ) , sizeof(uint32_t) );
            hal_copy_to_uspace( arg2 , XPTR( local_cxy , &type )   , sizeof(uint32_t) );
        
            break;
        }
        //////////////
        case FBF_READ:
        case FBF_WRITE:
        {
            // check arg0 (buffer) in user space
	        error = vmm_get_vseg( process , (intptr_t)arg0 , &vseg );
	        if( error )
            {

#if DEBUG_SYSCALLS_ERROR
printk("\n[ERROR] in %s : unmapped arg0 %x for %s / thread[%x,%x]\n",
__FUNCTION__ , arg0, dev_fbf_cmd_str(operation), process->pid, this->trdid );
#endif
                this->errno = EINVAL;
                return -1;
            }

            // get length and offset values

            uint32_t length = (uint32_t)(intptr_t)arg1;
            uint32_t offset = (uint32_t)(intptr_t)arg2;

            // call relevant kernel function to move data
            error = dev_fbf_move_data( operation , arg0 , length , offset ); 

            if( error )
            {

#if DEBUG_SYSCALLS_ERROR
printk("\n[ERROR] in %s : cannot move data for %s / buffer %x / thread[%x,%x]\n",
__FUNCTION__ , dev_fbf_cmd_str(operation), arg0, process->pid, this->trdid );
#endif
                this->errno = EINVAL;
                return -1;
            }
          
            break;
        }
        ///////
	    default:  // undefined operation                       
        {

#if DEBUG_SYSCALLS_ERROR
printk("\n[ERROR] in %s : undefined operation type %d / thread %x in process %x / cycle %d\n",
__FUNCTION__ , operation, this->trdid, process->pid, (uint32_t)hal_get_cycles() );
#endif
            this->errno = EINVAL;
            return -1;
        }
	}  // end switch on operation

    hal_fence();

#if (DEBUG_SYS_FBF || CONFIG_INSTRUMENTATION_SYSCALLS)
uint64_t     tm_end = hal_get_cycles();
#endif

#if DEBUG_SYS_FBF
if( DEBUG_SYS_FBF < tm_end )
printk("\n[DBG] %s : thread %x in process %x exit for %s / cost = %d / cycle %d\n",
__FUNCTION__, this->trdid, process->pid, dev_fbf_cmd_str( operation ),
(uint32_t)(tm_end - tm_start), (uint32_t)tm_end );
#endif

#if CONFIG_INSTRUMENTATION_SYSCALLS
hal_atomic_add( &syscalls_cumul_cost[SYS_FBF] , tm_end - tm_start );
hal_atomic_add( &syscalls_occurences[SYS_FBF] , 1 );
#endif

    return 0;

}  // end sys_fbf()
