/*
 * dev_ioc.c - IOC (Block Device Controler) generic device API implementation.
 *
 * Author  Alain Greiner    (2016,2017,2018,2019)
 *
 * Copyright (c) UPMC Sorbonne Universites
 *
 * This file is part of ALMOS-MK
 *
 * ALMOS-MKH is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2.0 of the License.
 *
 * ALMOS-MKH is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with ALMOS-MKH; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */

#include <kernel_config.h>
#include <hal_kernel_types.h>
#include <hal_gpt.h>
#include <hal_drivers.h>
#include <thread.h>
#include <printk.h>
#include <chdev.h>
#include <dev_ioc.h>

/////////////////////////////////////////////////////////////////////////////////////////
// Extern global variables
/////////////////////////////////////////////////////////////////////////////////////////

extern chdev_directory_t  chdev_dir;     // allocated in kernel_init.c

////////////////////////////////////////
char * dev_ioc_cmd_str( cmd_type_t cmd )
{
    if     ( cmd == IOC_READ       )  return "READ";
    else if( cmd == IOC_WRITE      )  return "WRITE";
    else if( cmd == IOC_SYNC_READ  )  return "SYNC_READ";
    else if( cmd == IOC_SYNC_WRITE )  return "SYNC_WRITE";
    else                              return "undefined";
}

//////////////////////////////////
void dev_ioc_init( chdev_t * ioc )
{
    // get channel from chdev descriptor
    uint32_t  channel = ioc->channel;

    // set chdev name
    snprintf( ioc->name , 16 , "ioc%d" , channel );

    // call driver init function
    hal_drivers_ioc_init( ioc );

    // select a core to execute the IOC server thread
    lid_t lid = cluster_select_local_core( local_cxy );

    // bind the IOC IRQ to the selected core
    dev_pic_bind_irq( lid , ioc );

    // enable IRQ
    dev_pic_enable_irq( lid , XPTR( local_cxy , ioc ) );

    // create server thread
    thread_t * new_thread;
    error_t    error;

    error = thread_kernel_create( &new_thread,
                                  THREAD_DEV,
                                  &chdev_server_func,
                                  ioc,
                                  lid );

    assert( (error == 0) , "cannot create server thread" );

    // set "server" field in chdev descriptor
    ioc->server = new_thread;

    // set "chdev" field in thread descriptor
    new_thread->chdev = ioc;

    // unblock server thread
    thread_unblock( XPTR( local_cxy , new_thread ) , THREAD_BLOCKED_GLOBAL );

}  // end dev_ioc_init()

///////////////////////////////////////////////
error_t dev_ioc_move_data( uint32_t   cmd_type,
                           xptr_t     buffer_xp,
                           uint32_t   lba,
                           uint32_t   count )
{
    thread_t * this = CURRENT_THREAD;              // pointer on client thread

#if ( DEBUG_DEV_IOC_RX  || DEBUG_DEV_IOC_TX )
uint32_t   cycle = (uint32_t)hal_get_cycles();
#endif

    // software L2/L3 cache coherence for memory buffer
    if( chdev_dir.iob )
    {
        if( (cmd_type == IOC_SYNC_READ) || (cmd_type == IOC_READ) )
        {
            dev_mmc_inval( buffer_xp , count<<9 );
        }
        else  // (cmd_type == IOC_SYNC_WRITE) or (cmd_type == IOC_WRITE)
        {
            dev_mmc_sync ( buffer_xp , count<<9 );
        }
    }

    // get extended pointer on IOC chdev descriptor
    xptr_t      ioc_xp  = chdev_dir.ioc[0];

// check dev_xp
assert( (ioc_xp != XPTR_NULL) , "undefined IOC chdev descriptor" );

    // register command in calling thread descriptor
    this->ioc_cmd.dev_xp    = ioc_xp;
    this->ioc_cmd.type      = cmd_type;
    this->ioc_cmd.buf_xp    = buffer_xp;
    this->ioc_cmd.lba       = lba;
    this->ioc_cmd.count     = count;

    // for a synchronous acces, the driver is directly called by the client thread 
    if( (cmd_type == IOC_SYNC_READ) || (cmd_type == IOC_SYNC_WRITE) )
    {

#if DEBUG_DEV_IOC_RX
uint32_t   cycle = (uint32_t)hal_get_cycles();
if( (DEBUG_DEV_IOC_RX < cycle) && (cmd_type == IOC_SYNC_READ) )
printk("\n[%s] thread[%x,%x] enters for SYNC_READ / lba  %x / buffer[%x,%x] / cycle %d\n",
__FUNCTION__ , this->process->pid, this->trdid, lba,
GET_CXY(buffer_xp), GET_PTR(buffer_xp), cycle );
#endif

#if DEBUG_DEV_IOC_TX
uint32_t   cycle = (uint32_t)hal_get_cycles();
if( (DEBUG_DEV_IOC_TX < cycle) && (cmd_type == IOC_SYNC_WRITE) )
printk("\n[%s] thread[%x,%x] enters for SYNC_WRITE / lba  %x / buffer[%x,%x] / cycle %d\n",
__FUNCTION__ , this->process->pid, this->trdid, lba,
GET_CXY(buffer_xp), GET_PTR(buffer_xp), cycle );
#endif
        // get driver command function
        cxy_t       ioc_cxy = GET_CXY( ioc_xp );
        chdev_t   * ioc_ptr = GET_PTR( ioc_xp );
        dev_cmd_t * cmd = (dev_cmd_t *)hal_remote_lpt( XPTR( ioc_cxy , &ioc_ptr->cmd ) );

        // call driver function
        cmd( XPTR( local_cxy , this ) );

#if DEBUG_DEV_IOC_RX
if( (DEBUG_DEV_IOC_RX < cycle) && (cmd_type == IOC_SYNC_READ) )
printk("\n[%s] thread[%x,%x] resumes for IOC_SYNC_READ\n",
__FUNCTION__, this->process->pid , this->trdid )
#endif

#if DEBUG_DEV_IOC_TX
if( (DEBUG_DEV_IOC_RX < cycle) && (cmd_type == IOC_SYNC_WRITE) )
printk("\n[%s] thread[%x,%x] resumes for IOC_SYNC_WRITE\n",
__FUNCTION__, this->process->pid , this->trdid )
#endif

    }
    // for an asynchronous access, the client thread registers in the chdev waiting queue,
    // activates server thread, blocks on THREAD_BLOCKED_IO and deschedules.
    // It is re-activated by the server thread after IO operation completion.
    else  // (cmd_type == IOC_READ) || (cmd_type == IOC_WRITE)
    {

#if DEBUG_DEV_IOC_RX
uint32_t   cycle = (uint32_t)hal_get_cycles();
if( (DEBUG_DEV_IOC_RX < cycle) && (cmd_type == IOC_READ) )
printk("\n[%s] thread[%x,%x] enters for READ / lba  %x / buffer[%x,%x] / cycle %d\n",
__FUNCTION__ , this->process->pid, this->trdid, lba,
GET_CXY(buffer_xp), GET_PTR(buffer_xp), cycle );
#endif

#if DEBUG_DEV_IOC_TX
uint32_t   cycle = (uint32_t)hal_get_cycles();
if( (DEBUG_DEV_IOC_TX < cycle) && (cmd_type == IOC_WRITE) )
printk("\n[%s] thread[%x,%x] enters for WRITE / lba  %x / buffer[%x,%x] / cycle %d\n",
__FUNCTION__ , this->process->pid, this->trdid, lba,
GET_CXY(buffer_xp), GET_PTR(buffer_xp), cycle );
#endif
        chdev_register_command( ioc_xp );

#if(DEBUG_DEV_IOC_RX )
if( (DEBUG_DEV_IOC_RX < cycle) && (cmd_type == IOC_READ) )
printk("\n[%s] thread[%x,%x] resumes for IOC_READ\n",
__FUNCTION__, this->process->pid , this->trdid )
#endif

#if(DEBUG_DEV_IOC_TX & 1)
if( (DEBUG_DEV_IOC_RX < cycle) && (cmd_type == IOC_WRITE) )
printk("\n[%s] thread[%x,%x] resumes for IOC_WRITE\n",
__FUNCTION__, this->process->pid , this->trdid )
#endif

    }

    // return I/O operation status
    return this->ioc_cmd.error;

}  // end dev_ioc_move_data()


