/*
 * sys_get_best_core.c - Kernel function implementing the "get_best_core" syscall.
 * 
 * Author    Alain Greiner (2016,2017,2018,2019)
 *  
 * Copyright (c) UPMC Sorbonne Universites
 *
 * This file is part of ALMOS-MKH.
 *
 * ALMOS-MKH is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2.0 of the License.
 *
 * ALMOS-MKH is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with ALMOS-MKH; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */

#include <hal_kernel_types.h>
#include <hal_uspace.h>
#include <hal_vmm.h>
#include <hal_special.h>
#include <errno.h>
#include <core.h>
#include <thread.h>
#include <process.h>
#include <vmm.h>
#include <printk.h>

#include <syscalls.h>

///////////////////////////////////////////
int sys_get_best_core( uint32_t   base_cxy,
                       uint32_t   level,
                       uint32_t * cxy,
                       uint32_t * lid )
{
    int       ret;
    vseg_t  * vseg;
    uint32_t  k_cxy;
    uint32_t  k_lid;

    thread_t  * this    = CURRENT_THREAD;
    process_t * process = this->process;

#if (DEBUG_SYS_GET_BEST_CORE || CONFIG_INSTRUMENTATION_SYSCALLS)
uint64_t     tm_start = hal_get_cycles();
#endif

#if DEBUG_SYS_GET_BEST_CORE
if( DEBUG_SYS_GET_BEST_CORE < tm_start )
printk("\n[%s] thread[%x,%x] enter / base_cxy %x / level %d / cycle %d\n",
__FUNCTION__, process->pid, this->trdid, base_cxy, level, (uint32_t)tm_start );
#endif

   // check <cxy> buffer in user space
    if( vmm_get_vseg( process , (intptr_t)cxy , &vseg ) )
	{

#if DEBUG_SYSCALLS_ERROR
printk("\n[ERROR] in %s : cxy buffer %x unmapped / thread %x / process %x\n",
__FUNCTION__ , (intptr_t)cxy , this->trdid , process->pid );
#endif
        this->errno = EINVAL;
		return -1;
	}

    // check <lid> buffer in user space
    if( vmm_get_vseg( process , (intptr_t)lid , &vseg ) )
	{

#if DEBUG_SYSCALLS_ERROR
printk("\n[ERROR] in %s : lid buffer unmapped %x / thread %x / process %x\n",
__FUNCTION__ , (intptr_t)lid , this->trdid , process->pid );
#endif
        this->errno = EINVAL;
		return -1;
	}

    // check level argument
	if( level > 5 ) 
	{

#if DEBUG_SYSCALLS_ERROR
printk("\n[ERROR] in %s : level argument %d too large / thread %x / process %x\n",
__FUNCTION__ , level , this->trdid , process->pid );
#endif
        this->errno = EINVAL;
		return -1;
	}

    // get extended pointer on the macro-cluster root node
    xptr_t root_xp = dqdt_get_root( base_cxy , level );

    if( root_xp == XPTR_NULL )  // macro-cluster undefined
    {

#if DEBUG_SYS_GET_BEST_CORE 
printk("\n[%s] thread[%x,%x] do nothing : macro cluster undefined\n",
__FUNCTION__, process->pid, this->trdid );
#endif
        k_lid = -1;
        k_cxy = -1;
        ret   = 1;
    }
    else                        // macro-cluster exist
    {
	    k_cxy = (uint32_t)dqdt_get_cluster_for_thread( root_xp );
	    k_lid = (uint32_t)cluster_select_local_core( k_cxy );
        ret   = 0;

#if DEBUG_SYS_GET_BEST_CORE
printk("\n[%s] thread[%x,%x] succeed : root_cxy %x / best_cluster %x / best_lid %d\n",
__FUNCTION__, process->pid, this->trdid, GET_CXY( root_xp ), k_cxy, k_lid );
#endif

    }

    // copy to user space
	hal_copy_to_uspace( cxy,
                        XPTR( local_cxy , &k_cxy ),
                        sizeof(uint32_t) );

	hal_copy_to_uspace( lid,
                        XPTR( local_cxy , &k_lid ),
                        sizeof(uint32_t) );

#if (DEBUG_SYS_GET_BEST_CORE || CONFIG_INSTRUMENTATION_SYSCALLS)
uint64_t     tm_end = hal_get_cycles();
#endif

#if DEBUG_SYS_GET_BEST_CORE
if( DEBUG_SYS_GET_BEST_CORE < tm_end )
printk("\n[%s] thread[%x,%x] exit / cycle %d\n",
__FUNCTION__ , process->pid, this->trdid, (uint32_t)tm_end );
#endif

#if CONFIG_INSTRUMENTATION_SYSCALLS
hal_atomic_add( &syscalls_cumul_cost[SYS_GET_BEST_CORE] , tm_end - tm_start );
hal_atomic_add( &syscalls_occurences[SYS_GET_BEST_CORE] , 1 );
#endif

	return ret; 

}  // end sys_get_best_core()
