/*
 * dev_ioc.c - IOC (Block Device Controler) generic device API implementation.
 *
 * Author  Alain Greiner    (2016,2017,2018,2019,2020)
 *
 * Copyright (c) UPMC Sorbonne Universites
 *
 * This file is part of ALMOS-MK
 *
 * ALMOS-MKH is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2.0 of the License.
 *
 * ALMOS-MKH is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with ALMOS-MKH; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */

#include <kernel_config.h>
#include <hal_kernel_types.h>
#include <hal_gpt.h>
#include <hal_drivers.h>
#include <thread.h>
#include <printk.h>
#include <chdev.h>
#include <dev_ioc.h>

/////////////////////////////////////////////////////////////////////////////////////////
// Extern global variables
/////////////////////////////////////////////////////////////////////////////////////////

extern chdev_directory_t  chdev_dir;     // allocated in kernel_init.c

////////////////////////////////////////////
char * dev_ioc_cmd_str( ioc_cmd_type_t cmd )
{
    if     ( cmd == IOC_READ       )  return "READ";
    else if( cmd == IOC_WRITE      )  return "WRITE";
    else if( cmd == IOC_SYNC_READ  )  return "SYNC_READ";
    else if( cmd == IOC_SYNC_WRITE )  return "SYNC_WRITE";
    else                              return "undefined";
}

//////////////////////////////////
void dev_ioc_init( chdev_t * ioc )
{
    // get channel from chdev descriptor
    uint32_t  channel = ioc->channel;

    // set chdev name
    snprintf( ioc->name , 16 , "ioc%d" , channel );

    // call driver init function
    hal_drivers_ioc_init( ioc );

    // select a core to execute the IOC server thread
    lid_t lid = cluster_select_local_core( local_cxy );

    // bind the IOC IRQ to the selected core
    dev_pic_bind_irq( lid , ioc );

    // enable IRQ
    dev_pic_enable_irq( lid , XPTR( local_cxy , ioc ) );

    // create server thread
    thread_t * new_thread;
    error_t    error;

    error = thread_kernel_create( &new_thread,
                                  THREAD_DEV,
                                  &chdev_server_func,
                                  ioc,
                                  lid );

    assert( (error == 0) , "cannot create server thread" );

    // set "server" field in chdev descriptor
    ioc->server = new_thread;

    // set "chdev" field in thread descriptor
    new_thread->chdev = ioc;

    // unblock server thread
    thread_unblock( XPTR( local_cxy , new_thread ) , THREAD_BLOCKED_GLOBAL );

}  // end dev_ioc_init()

////////////////////////////////////////////////////////////////////////////////////
// This static function executes an asynchronous SYNC_READ or SYNC_WRITE request.
// thread in the IOC device waiting queue, activates the server thread, blocks on 
// the THREAD_BLOCKED_IO condition and deschedules.
// The clent is re-activated by the server thread after IO operation completion.
////////////////////////////////////////////////////////////////////////////////////
static error_t dev_ioc_move( uint32_t   cmd_type,
                             xptr_t     buffer_xp,
                             uint32_t   lba,
                             uint32_t   count )
{
    thread_t * this = CURRENT_THREAD;              // pointer on client thread

    // get extended pointer on IOC chdev descriptor
    xptr_t      ioc_xp  = chdev_dir.ioc[0];

// check dev_xp
assert( (ioc_xp != XPTR_NULL) , "undefined IOC chdev descriptor" );

    // register command in client thread  
    this->ioc_cmd.dev_xp    = ioc_xp;
    this->ioc_cmd.type      = cmd_type;
    this->ioc_cmd.buf_xp    = buffer_xp;
    this->ioc_cmd.lba       = lba;
    this->ioc_cmd.count     = count;

    // register client thread in IOC queue, blocks and deschedules 
    chdev_register_command( ioc_xp );

    // return I/O operation status
    return this->ioc_cmd.error;

}  // end dev_ioc_move()

////////////////////////////////////////////////////////////////////////////////////
// This static function executes a synchronous READ or WRITE request.
// It register the command in the client thread descriptor, and calls directly
// the driver cmd function.
////////////////////////////////////////////////////////////////////////////////////
error_t dev_ioc_sync_move( uint32_t   cmd_type,
                           xptr_t     buffer_xp,
                           uint32_t   lba,
                           uint32_t   count )
{
    thread_t * this = CURRENT_THREAD;              // pointer on client thread

    // get extended pointer on IOC chdev descriptor
    xptr_t      ioc_xp  = chdev_dir.ioc[0];

// check dev_xp
assert( (ioc_xp != XPTR_NULL) , "undefined IOC chdev descriptor" );

    // register command in calling thread descriptor
    this->ioc_cmd.dev_xp    = ioc_xp;
    this->ioc_cmd.type      = cmd_type;
    this->ioc_cmd.buf_xp    = buffer_xp;
    this->ioc_cmd.lba       = lba;
    this->ioc_cmd.count     = count;

    // get driver command function
    cxy_t       ioc_cxy = GET_CXY( ioc_xp );
    chdev_t   * ioc_ptr = GET_PTR( ioc_xp );
    dev_cmd_t * cmd = (dev_cmd_t *)hal_remote_lpt( XPTR( ioc_cxy , &ioc_ptr->cmd ) );

    // call driver function whithout blocking & descheduling
    cmd( XPTR( local_cxy , this ) );

    // return I/O operation status
    return this->ioc_cmd.error;

}  // end dev_ioc_sync_move()

///////////////////////////////////////////
error_t dev_ioc_read( xptr_t     buffer_xp,
                      uint32_t   lba,
                      uint32_t   count )
{

#if DEBUG_DEV_IOC_RX
thread_t * this  = CURRENT_THREAD;
uint32_t   cycle = (uint32_t)hal_get_cycles();
if( DEBUG_DEV_IOC_RX < cycle )
printk("\n[%s] thread[%x,%x] enters IOC_READ / lba  %x / buffer[%x,%x] / cycle %d\n",
__FUNCTION__, this->process->pid, this->trdid, lba,
GET_CXY(buffer_xp), GET_PTR(buffer_xp), cycle );
#endif

    // software L2/L3 cache coherence for memory buffer
    if( chdev_dir.iob ) dev_mmc_inval( buffer_xp , count<<9 );

    // request an asynchronous transfer
    error_t error = dev_ioc_move( IOC_READ,
                                  buffer_xp,
                                  lba,
                                  count );
#if(DEBUG_DEV_IOC_RX & 1)
cycle = (uint32_t)hal_get_cycles();
if( DEBUG_DEV_IOC_RX < cycle )
printk("\n[%s] thread[%x,%x] exit IOC_READ / cycle %d\n",
__FUNCTION__, this->process->pid , this->trdid , cycle );
#endif

    return error;

}  // end dev_ioc_read()

///////////////////////////////////////////
error_t dev_ioc_write( xptr_t     buffer_xp,
                       uint32_t   lba,
                       uint32_t   count )
{

#if DEBUG_DEV_IOC_TX
thread_t * this  = CURRENT_THREAD;
uint32_t   cycle = (uint32_t)hal_get_cycles();
if( DEBUG_DEV_IOC_TX < cycle )
printk("\n[%s] thread[%x,%x] enters IOC_WRITE / lba  %x / buffer[%x,%x] / cycle %d\n",
__FUNCTION__, this->process->pid, this->trdid, lba,
GET_CXY(buffer_xp), GET_PTR(buffer_xp), cycle );
#endif

    // software L2/L3 cache coherence for memory buffer
    if( chdev_dir.iob ) dev_mmc_sync ( buffer_xp , count<<9 );

    // request a blocking, but asynchronous, transfer
    error_t error = dev_ioc_move( IOC_WRITE,
                          buffer_xp,
                          lba,
                          count );
#if(DEBUG_DEV_IOC_TX & 1)
cycle = (uint32_t)hal_get_cycles();
if( DEBUG_DEV_IOC_TX < cycle )
printk("\n[%s] thread[%x,%x] exit IOC_WRITE / cycle %d\n",
__FUNCTION__, this->process->pid , this->trdid , cycle );
#endif

    return error;

}  // end dev_ioc_write()


///////////////////////////////////////////
error_t dev_ioc_sync_read( xptr_t     buffer_xp,
                           uint32_t   lba,
                           uint32_t   count )
{

#if DEBUG_DEV_IOC_RX
thread_t * this  = CURRENT_THREAD;
uint32_t   cycle = (uint32_t)hal_get_cycles();
if( DEBUG_DEV_IOC_RX < cycle )
printk("\n[%s] thread[%x,%x] enters IOC_SYNC_READ / lba  %x / buffer[%x,%x] / cycle %d\n",
__FUNCTION__, this->process->pid, this->trdid, lba,
GET_CXY(buffer_xp), GET_PTR(buffer_xp), cycle );
#endif

    // software L2/L3 cache coherence for memory buffer
    if( chdev_dir.iob ) dev_mmc_inval( buffer_xp , count<<9 );

    // request an asynchronous transfer
    error_t error = dev_ioc_sync_move( IOC_SYNC_READ,
                                       buffer_xp,
                                       lba,
                                       count );
#if(DEBUG_DEV_IOC_RX & 1)
cycle = (uint32_t)hal_get_cycles();
if( DEBUG_DEV_IOC_RX < cycle )
printk("\n[%s] thread[%x,%x] exit IOC_SYNC_READ / cycle %d\n",
__FUNCTION__, this->process->pid , this->trdid , cycle );
#endif

    return error;

}  // end dev_ioc_sync_read()  

/////////////////////////////////////////////////
error_t dev_ioc_sync_write( xptr_t     buffer_xp,
                            uint32_t   lba,
                            uint32_t   count )
{

#if DEBUG_DEV_IOC_TX
thread_t * this  = CURRENT_THREAD;
uint32_t   cycle = (uint32_t)hal_get_cycles();
if( DEBUG_DEV_IOC_TX < cycle )
printk("\n[%s] thread[%x,%x] enters IOC_SYNC_WRITE / lba  %x / buffer[%x,%x] / cycle %d\n",
__FUNCTION__, this->process->pid, this->trdid, lba,
GET_CXY(buffer_xp), GET_PTR(buffer_xp), cycle );
#endif

    // software L2/L3 cache coherence for memory buffer
    if( chdev_dir.iob ) dev_mmc_sync ( buffer_xp , count<<9 );

    // request a blocking, but asynchronous, transfer
    error_t error = dev_ioc_sync_move( IOC_SYNC_WRITE,
                                       buffer_xp,
                                       lba,
                                       count );
#if(DEBUG_DEV_IOC_TX & 1)
cycle = (uint32_t)hal_get_cycles();
if( DEBUG_DEV_IOC_TX < cycle )
printk("\n[%s] thread[%x,%x] exit IOC_SYNC_WRITE / cycle %d\n",
__FUNCTION__, this->process->pid , this->trdid , cycle );
#endif

    return error;

}  // end dev_ioc_sync_write() 


