///////////////////////////////////////////////////////////////////////////////////////
//  file   : display.c  
//  date   : may 2014
//  author : Alain Greiner
///////////////////////////////////////////////////////////////////////////////////////
//  This file describes the single thread "display" application.
//  It uses the external chained buffer DMA to display a stream
//  of images on the frame buffer.  
///////////////////////////////////////////////////////////////////////////////////////

#include <stdio.h>
#include <stdlib.h>
#include <fcntl.h>
#include <unistd.h>
#include <almosmkh.h>

#define PATH_MAX_LENGHT    128

#define FBF_TYPE    420         
#define NPIXELS     128
#define NLINES      128

/////////// Global variables  ///////////////////////////////////////

char    buffer[NPIXELS * NLINES] __attribute__((aligned(512)));

/////////////////////////
void build( char * base )
{
    int    line;
    int    pixel;
    char * buf;

    for( line = 0 ; line < NLINES ; line++ )
    {
        for( pixel = 0 ;  pixel < NPIXELS ; pixel++ )
        {
            buf = base + (line * NPIXELS) + pixel;

            *buf = ( ((pixel <  (NPIXELS>>1)) && (line <  (NLINES>>1))) ||
                     ((pixel >= (NPIXELS>>1)) && (line >= (NLINES>>1))) ) ? 0xFF : 0;
        }
    }
}

//////////////////////////////////////     
void print_block( unsigned int * buf )
{
    int offset;    // word index in array of words
    
    printf("\n***** content of buffer %x *****\n", buf );
    for( offset = 0 ; offset < 128 ; offset += 8 )
    {
        printf(" - %d\t: %x | %x | %x | %x | %x | %x | %x | %x\n",
        offset<<2, buf[offset+0], buf[offset+1], buf[offset+2], buf[offset+3], 
                   buf[offset+4], buf[offset+5], buf[offset+6], buf[offset+7] );
    }
}        

////////////////
int main( void )
{

    unsigned int    fbf_width;
    unsigned int    fbf_height;
    unsigned int    fbf_type;
    unsigned int    nbytes;
    int             val;
    int             error;
    char            pathname[PATH_MAX_LENGHT];

    // check frame buffer size
    fbf_get_config( &fbf_width , &fbf_height , &fbf_type );

    if( (NPIXELS != fbf_width) || (NLINES != fbf_height) )
    {
        printf("\n[display error] FBF size [%d,%d] does not fit image size[%d,%d]\n",
        fbf_width, fbf_height, NPIXELS, NLINES );
        exit( 0 );
    }

    if( fbf_type != FBF_TYPE )
    {
        printf("\n[display error] FBF type [%d] does not fit image type [%d]\n",
        fbf_type, FBF_TYPE);
        exit( 0 );
    }

    // get pathname for input file 
    while( 1 )
    {
        printf("\n[display] path = ");

        error = get_string( pathname , PATH_MAX_LENGHT );

        if ( error )  printf("\n[display error] cannot get path for input file\n" );
        else  break;
    }

    // open file
    int fd = open( pathname , O_RDONLY , 0 );

    if( fd < 0 )
    {
        printf("\n[display error] Cannot open file <%s>\n", pathname );
        exit( 0 );
    }

    printf("\n[display] open file <%s>\n", pathname );

    // load buffer from file
    nbytes = read( fd , buffer , NPIXELS * NLINES );

    if( nbytes != NPIXELS * NLINES )
    {
        printf("\n[display error] Cannot load image \n" );
        exit( 0 );
    }

    printf("\n[display] load image in buffer %x\n", buffer );
 
    // display image
    error = fbf_write( buffer , NPIXELS * NLINES , 0 );

    if( error )
    {
        printf("\n[display error] Cannot access frame buffer\n");
        exit( 0 );
    }

    printf("\n[display] display image\n");

    // close file
    close( fd );
   
    exit(0);

    return 0;
}
