/*
 * hal_uspace.h - Generic User Space Access API definition
 *
 * Authors    Alain Greiner (2016,2017,2018,2019,2020)
 *
 * Copyright (c)  UPMC Sorbonne Universites
 *
 * This file is part of ALMOS-MKH.
 *
 * ALMOS-MKH is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2.0 of the License.
 *
 * ALMOS-MKH is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with ALMOS-MKH; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */

#ifndef  _HAL_USPACE_H_
#define  _HAL_USPACE_H_

#include <hal_kernel_types.h>

//////////////////////////////////////////////////////////////////////////////////////////
//           User space access API (implementation in hal_uspace.c)
//
// For sake of portability, user/kernel data transfers must use the following API.
//
// When moving data between user space and kernel space, the user address is always
// a virtual address, and the kernel address is an extended pointer.
// Therefore, each of these buffers can be located in any cluster.
//
// WARNING: these function must be executed by an user thread (i.e. PID > 0),
//          to handle a possible page fault when accessing the user buffer.
//////////////////////////////////////////////////////////////////////////////////////////


/*****************************************************************************************
 * This function moves <size> bytes from a source buffer in user virtual space,
 * defined by the <u_src_ptr> argument, to a destination kernel buffer, defined by the
 * <k_dst_xp> argument.
 *****************************************************************************************
 * @ k_dst_xp  : extended pointer on kernel destination buffer.
 * @ u_src_ptr : source buffer address in user space.
 * @ size      : size (number of bytes).
 ****************************************************************************************/
extern void hal_copy_from_uspace( xptr_t     k_dst_xp,
                                  void     * u_src_ptr,
                                  uint32_t   size );

/*****************************************************************************************
 * This function moves <size> bytes from a source kernel buffer, defined by the
 * <k_src_xp> argument, to a destination buffer in user virtual space, defined by 
 * the <u_dst_ptr> argument.
 *****************************************************************************************
 * @ u_dst_ptr : destination buffer address in user space.
 * @ k_src_xp  : extended pointer on kernel source buffer.
 * @ size      : size (number of bytes).
 ****************************************************************************************/
extern void hal_copy_to_uspace( void     * u_dst_ptr,
                                xptr_t     k_src_xp, 
                                uint32_t   size );

/*****************************************************************************************
 * This function tranfers a string from the user space to the kernel space.
 * The transfer stops after the first encountered NUL character, and no more than
 * <max_size> characters are actually copied to target buffer.
 *****************************************************************************************
 * @ k_dst_xp  : extended pointer on kernel destination buffer.
 * @ u_src_ptr : source address in user space.
 * @ max_size  : max number of characters to be copied.
 ****************************************************************************************/
extern void hal_strcpy_from_uspace( xptr_t     k_dst_xp,
                                    char     * u_src_ptr,
                                    uint32_t   max_size );

/*****************************************************************************************
 * This function tranfers a string from the kernel space to the user space.
 * The transfer stops after the first encountered NUL character, and no more than
 * <max_size> characters are actually copied to target buffer.
 *****************************************************************************************
 * @ u_dst_ptr : destination buffer address in user space.
 * @ k_src_xp  : extended pointer on kernel source buffer.
 * @ max_size  : max number of characters to be copied.
 ****************************************************************************************/
extern void hal_strcpy_to_uspace( char     * u_dst_ptr,
                                  xptr_t     k_src_xp,
                                  uint32_t   max_size );

/*****************************************************************************************
 * This function returns the length of a string located in user space.
 *****************************************************************************************
 * @ string     : string in user space.
 * @ return length of the string.
 ****************************************************************************************/
uint32_t hal_strlen_from_uspace( char * string );


#endif	/* _HAL_USPACE_H_ */
