/*
 * sys_get_config.c - get hardware platform parameters.
 * 
 * Author    Alain Greiner (2016,2017,2018,2019,2020)
 *  
 * Copyright (c) UPMC Sorbonne Universites
 *
 * This file is part of ALMOS-MKH.
 *
 * ALMOS-MKH is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2.0 of the License.
 *
 * ALMOS-MKH is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with ALMOS-MKH; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */

#include <hal_kernel_types.h>
#include <hal_uspace.h>
#include <hal_vmm.h>
#include <hal_special.h>
#include <errno.h>
#include <core.h>
#include <thread.h>
#include <process.h>
#include <vmm.h>
#include <printk.h>

#include <syscalls.h>
#include <shared_syscalls.h>

//////////////////////////////////////////////
int sys_get_config( hard_config_t * u_config )
{
    vseg_t        * vseg;        
    hard_config_t   k_config;    // hard_config structure in kernel space

    thread_t  * this    = CURRENT_THREAD;
    process_t * process = this->process;

#if (DEBUG_SYS_GET_CONFIG || CONFIG_INSTRUMENTATION_SYSCALLS)
uint64_t     tm_start = hal_get_cycles();
#endif

#if DEBUG_SYS_GET_CONFIG
tm_start = hal_get_cycles();
if( DEBUG_SYS_GET_CONFIG < tm_start )
printk("\n[DBG] %s : thread %x enter / process %x / cycle %d\n",
__FUNCTION__, this, process->pid, (uint32_t)tm_start );
#endif

    // check u_config mapped in user space
    if( vmm_get_vseg( process , (intptr_t)u_config  , &vseg ) )
	{

#if DEBUG_SYSCALLS_ERROR
printk("\n[ERROR] in %s : config pointer %x unmapped / thread %x / process %x\n",
__FUNCTION__ , (intptr_t)u_config , this->trdid , process->pid );
#endif
        this->errno = EINVAL;
		return -1;
	}

    // copy config parameters from cluster descriptor to kernel structure
	k_config.x_size       = LOCAL_CLUSTER->x_size;
	k_config.y_size       = LOCAL_CLUSTER->y_size;
	k_config.ncores       = LOCAL_CLUSTER->cores_nr;
    k_config.txt_channels = LOCAL_CLUSTER->nb_txt_channels;
    k_config.nic_channels = LOCAL_CLUSTER->nb_nic_channels;
    k_config.ioc_channels = LOCAL_CLUSTER->nb_ioc_channels;
    k_config.fbf_channels = LOCAL_CLUSTER->nb_fbf_channels;

    // copy k_config structure to user space
	hal_copy_to_uspace( u_config , XPTR(local_cxy, &k_config ), sizeof(hard_config_t) );

    hal_fence();

#if (DEBUG_SYS_GET_CONFIG || CONFIG_INSTRUMENTATION_SYSCALLS)
uint64_t     tm_end = hal_get_cycles();
#endif

#if DEBUG_SYS_GET_CONFIG
if( DEBUG_SYS_GET_CONFIG < tm_end )
printk("\n[DBG] %s : thread %x exit / process %x / cost %d / cycle %d\n",
__FUNCTION__, this, process->pid, (uint32_t)(tm_end-tm_start), (uint32_t)tm_end );
#endif

#if CONFIG_INSTRUMENTATION_SYSCALLS
hal_atomic_add( &syscalls_cumul_cost[SYS_GET_CONFIG] , tm_end - tm_start );
hal_atomic_add( &syscalls_occurences[SYS_GET_CONFIG] , 1 );
#endif

	return 0; 

}  // end sys_get_config()
