/*
 * almosmkh.h - User level ALMOS-MKH specific library definition.
 * 
 * Author     Alain Greiner (2016,2017,2018,2019,2020)
 *
 * Copyright (c) UPMC Sorbonne Universites
 *
 * This file is part of ALMOS-MKH.
 *
 * ALMOS-MKH is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2.0 of the License.
 *
 * ALMOS-MKH is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with ALMOS-MKH; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */

#ifndef _LIBALMOSMKH_H_
#define _LIBALMOSMKH_H_

/*************************************************************************************** 
 * This file defines an user level, ALMOS-MKH specific library, containing: 
 * - non standard system calls.
 * - debug functions.
 * - remote malloc extensions.
 * - a parallel_pthread_create function.
 * - Frame Buffer access syscalls.
 **************************************************************************************/

#include <pthread.h>
#include <shared_almos.h>


/****************** Non standard (ALMOS_MKH specific) system calls ********************/


/*************************************************************************************** 
 * This syscall gives the process identified by the <pid> argument the exclusive 
 * ownership of its TXT terminal.
 *************************************************************************************** 
 * @ pid        : process identifier.
 * @ returns O if success / returns -1 if process not found.
 **************************************************************************************/
int fg( unsigned int pid );

/*************************************************************************************** 
 * This syscall stores in the buffer identified by the <owner> argument a non zero
 * value when the process identified by the <pid> argument is currently the exclusive
 * owner of its TXT terminal.
 *************************************************************************************** 
 * @ pid        : [in]  process identifier.
 * @ owner      : [out] pointer on buffer to store the
 * @ returns O if success / returns -1 if process not found.
 **************************************************************************************/
int is_fg( unsigned int pid,
           unsigned int * owner );

/*************************************************************************************** 
 * This syscall returns the hardware platform parameters in the hard_config_t pointed
 * by the <config> argument. This structure is defined in the shared_almos.h file.
 * The available configuration parameters are :
 * - cores       : x_size / y_size / ncores
 * - peripherals : nb_txt_channels / nb_nic_channels / nb_ioc_channels / nb_fbf_channels
 ***************************************************************************************
 * @ config   : [out] pointer on hard_config_t structure in user space.
 * @ return 0 if success / return -1 if illegal argument.
 **************************************************************************************/
int get_config( struct hard_config_s * config );

/*************************************************************************************** 
 * This syscall returns the cluster identifier and the local index
 * for the calling core.
 ***************************************************************************************
 * @ cxy      : [out] cluster identifier.
 * @ lid      : [out] core local index in cluster.
 * @ return always 0.
 **************************************************************************************/
int get_core_id( unsigned int * cxy,
                 unsigned int * lid );

/*************************************************************************************** 
 * This syscall returns the number of cores in a given cluster.
 ***************************************************************************************
 * @ cxy      : [in]  target cluster identifier.
 * @ ncores   : [out] number of cores in target cluster.
 * @ return always 0.
 **************************************************************************************/
int get_nb_cores( unsigned int   cxy,
                  unsigned int * ncores );

/*************************************************************************************** 
 * This syscall uses the DQDT to search, in a macro-cluster specified by the 
 * <cxy_base> and <level> arguments arguments, the core with the lowest load.
 * it writes in the <cxy> and <lid> buffers the selected core cluster identifier
 * and the local core index.
 ***************************************************************************************
 * @ cxy_base : [in]  any cluster identifier in macro-cluster.in clusters array.
 * @ level    : [in]  macro-cluster level in [1,2,3,4,5].
 * @ cxy      : [out] selected core cluster identifier.
 * @ lid      : [out] selected core local index.
 * @ return 0 if success / 1 if no core in macro-cluster / -1 if illegal arguments.
 **************************************************************************************/
int get_best_core( unsigned int   cxy_base,
                   unsigned int   level,
                   unsigned int * cxy,
                   unsigned int * lid );

/***************************************************************************************
 * This function returns the value contained in the calling core cycles counter,
 * taking into account a possible overflow on 32 bits architectures.
 ***************************************************************************************
 * @ cycle    : [out] current cycle value.
 * @ return always 0.
 **************************************************************************************/
int get_cycle( unsigned long long * cycle );

/***************************************************************************************
 * This syscall allows the calling thread to specify the target cluster for
 * a subsequent fork(). It must be called for each fork().
 ***************************************************************************************
 * @ cxy      : [in] target cluster identifier.
 * @ return 0 if success / returns -1 if illegal cxy argument.
 **************************************************************************************/
int place_fork( unsigned int cxy );

/*************************************************************************************** 
 * This syscall implements the operations related to User Thread Local Storage.
 ***************************************************************************************
 * @ operation  : UTLS operation type as defined in "shared_sycalls.h" file.
 * @ value      : argument value for the UTLS_SET operation.
 * @ return value for the UTLS_GET and UTLS_GET_ERRNO / return -1 if failure.
 **************************************************************************************/
int utls( unsigned int operation,
          unsigned int value );

/***************************************************************************************
 * This syscall returns an 32 bits integer from the standard "stdin" stream.
 * The value is copied in buffer pointed by the <buf> argument
 * Both decimal numbers and hexadecimal numbers (prefixed by 0x) are supported.
 ***************************************************************************************
 * @ buffer    : pointer on buffer.
 * @ returns 0 if success / returns -1 if failure.
 **************************************************************************************/
int get_uint32( unsigned int * buffer );

/***************************************************************************************
 * This syscall returns a character string from the standard "stdin" stream.
 * The string is copied in buffer pointed by the <string> argument. 
 * If the number of characters exceed the buffer length, all character are canceled
 * in both the buffer & the echoed string, and a new string capture start. 
 * - Only printable characters c such as (0x20 <= c) and (c < 0x7f) are echoed
 *   and copied in the target buffer. 
 * - The LF (0x0A) character signaling the end of string is replaced by an extra
 *   NUL (0x00) character in the target buffer.
 * - the DEL (0x7F), BS (0x08) control characters are taken into account, and can be
 *   used to correct both the buffer content and the echoed string.
 ***************************************************************************************
 * @ string   : pointer on the string buffer.
 * @ maxlen   : max number of bytes in string buffer, including the terminating NUL.
 * @ returns string length (including NUL) if success / returns -1 if failure.
 **************************************************************************************/
int get_string( char * string,
                int    maxlen );


/***************** Non standard (ALMOS-MKH specific) debug functions ******************/

/*************************************************************************************** 
 * This syscall copies in the user structure defined by the <info> argument the values
 * registered in the calling thread "thread_info_t" kernel structure.
 ******************************************************************************************
 * @ info    : [out] pointer on thread_info_t structure in user space.
 * @ return 0 if success / return -1 if illegal argument.
 *****************************************************************************************/
int get_thread_info( thread_info_t * info );

/*************************************************************************************** 
 * This debug syscall displays on the kernel terminal TXT0
 * the thread / process / core identifiers, the current cycle, plus a user defined
 * message as specified by the <string> argument. 
 ***************************************************************************************
 * @ string    : [in] user defined message.
 **************************************************************************************/
void display_string( char * string );

/*************************************************************************************** 
 * This debug function displays on the kernel terminal TXT0 
 * the state of the  VMM for the process <pid> in cluster <cxy>.
 * It can be called by any thread running in any cluster.
 ***************************************************************************************
 * @ cxy      : [in] target cluster identifier.
 * @ pid      : [in] process identifier.
 * @ mapping  : [in] detailed mapping if non zero.
 * @ return 0 if success / return -1 if illegal argument.
 **************************************************************************************/
int display_vmm( unsigned int cxy,
                 unsigned int pid,
                 unsigned int mapping );

/*************************************************************************************** 
 * This debug syscall displays on the kernel terminal TXT0 
 * the state of the core scheduler identified by the <cxy> and <lid> arguments.
 * It can be called by any thread running in any cluster.
 ***************************************************************************************
 * @ cxy      : [in] target cluster identifier.
 * @ lid      : [in] target core local index.
 * @ return 0 if success / return -1 if illegal arguments.
 **************************************************************************************/
int display_sched( unsigned int  cxy,
                   unsigned int  lid );

/*************************************************************************************** 
 * This debug syscall displays on the kernel terminal TXT0 
 * the list of process registered in a given cluster identified by the <cxy> argument.
 * Only the owned processes are displayed when the <owned> argument is non zero.
 * It can be called by any thread running in any cluster.
 ***************************************************************************************
 * @ cxy      : [in] target cluster identifier.
 * @ owned    : [in] only owned processes if non zero.
 * @ return 0 if success / return -1 if illegal argument.
 **************************************************************************************/
int display_cluster_processes( unsigned int  cxy,
                               unsigned int  owned );

/*************************************************************************************** 
 * This debug syscall displays on the kernel terminal TXT0 
 * the list of processes attached to a given TXT channel.
 * It can be called by any thread running in any cluster.
 ***************************************************************************************
 * @ txt_id   : [in] TXT terminal indes.
 * @ return 0 if success / return -1 if illegal argument.
 **************************************************************************************/
int display_txt_processes( unsigned int txt_id );

/*************************************************************************************** 
 * This debug syscall displays on the kernel terminal TXT0 
 * the set of busylocks hold by a thread identified by the <pid> and <trdid> arguments.
 * It can be called by any thread running in any cluster.
 ***************************************************************************************
 * @ pid      : [in] process identifier.
 * @ trdid    : [in] thread identifier.
 * @ return 0 if success / return -1 if illegal arguments.
 **************************************************************************************/
int display_busylocks( unsigned int pid,
                       unsigned int trdid );

/*************************************************************************************** 
 * This debug syscall displays on the kernel terminal TXT0 
 * the list of channel devices available in the architecture.
 * It can be called by any thread running in any cluster.
 ***************************************************************************************
 * @ return always 0.
 **************************************************************************************/
int display_chdev( void );

/*************************************************************************************** 
 * This debug syscall displays on the kernel terminal TXT0 
 * the list of channel device or pseudo-files registered in the VFS cache.
 * It can be called by any thread running in any cluster.
 ***************************************************************************************
 * @ return always 0.
 **************************************************************************************/
int display_vfs( void );

/*************************************************************************************** 
 * This debug syscall displays on the kernel terminal TXT0 the current DQDT state.
 * It can be called by any thread running in any cluster.
 ***************************************************************************************
 * @ return always 0.
 **************************************************************************************/
int display_dqdt( void );

/*************************************************************************************** 
 * This debug syscall displays on the kernel terminal TXT0 the content of a given
 * page of a given VFS mapper.
 * It can be called by any thread running in any cluster.
 ***************************************************************************************
 * @ path      : pathname identifying the file/directory in VFS.
 * @ page_id   : page index in file.
 * @ nbytes    : number of bytes to display.
 * @ return 0 if success / return -1 if file or page not found.
 **************************************************************************************/
int display_mapper( char        * path,
                    unsigned int  page_id,
                    unsigned int  nbytes);

/*************************************************************************************** 
 * This debug syscall displays on the kernel terminal TXT0 
 * the state of the barrier used by the process identified by the <pid> argument.
 * It can be called by any thread running in any cluster.
 ***************************************************************************************
 * @ pid      : [in] process identifier.
 * @ return 0 if success / return -1 if illegal arguments.
 **************************************************************************************/
int display_barrier( unsigned int pid );

/*************************************************************************************** 
 * This debug syscall displays on the kernel terminal TXT0 the content of <nb_slots>
 * of the FAT mapper, starting from <min_slot>.
 * If <nb_slots> is zero, it displays the state of the FATFS context in cluster
 * defined by the <min_slot> argument interpreted as a cluster identifier.
 * It can be called by any thread running in any cluster.
 ***************************************************************************************
 * @ min_slot   : first slot in page
 * @ nb_slots   : number of slots to display.
 * @ return 0 if success / return -1 if page not found.
 **************************************************************************************/
int display_fat( unsigned int min_slot,
                 unsigned int nb_slots );

/*************************************************************************************** 
 * This debug syscall displays on the kernel terminal the current state of
 * a socket identified by the <pid> and <fdid> arguments.
 * It can be called by any thread running in any cluster.
 ***************************************************************************************
 * @ pid        : target process identifier.
 * @ fdid       : file descriptor identifying the target socket.
 * @ return 0 if success / return -1 if socket not found.
 **************************************************************************************/
int display_socket( unsigned int pid,
                    unsigned int fdid );

/***************************************************************************************** 
* This debug syscall is used to activate / desactivate the context switches trace
* for a core identified by the <cxy> and <lid> arguments.
* It can be called by any thread running in any cluster.
***************************************************************************************** 
* @ active     : activate trace if non zero / desactivate if zero.
* @ cxy        : cluster identifier.
* @ lid        : core local index.
* @ returns O if success / returns -1 if illegal arguments.
****************************************************************************************/
int trace( unsigned int active,
           unsigned int cxy, 
           unsigned int lid );

/****************************************************************************************
 * This syscall implements an user-level interactive debugger that can be
 * introduced in any user application to display various kernel distributed structures.
 ***************************************************************************************/
void idbg( void );


/****************** Non standard (ALMOS-MKH specific) malloc operations  ***************/

/////////////////////////////////////////////////////////////////////////////////////////
// General principles:
// - In user space the HEAP zone spread between the ELF zone and the STACK zone,
//   as defined in the kernel_config.h file.
// - The malloc library uses the mmap() syscall to create - on demand -
//   one vseg in a given cluster. The size of this vseg is defined below
//   by the MALLOC_LOCAL_STORE_SIZE parameter.
// - For a standard malloc(), the target cluster is the cluster containing
//   the core running the client thread.
// - For a remote_malloc(), the target cluster is explicitely defined
//   by the argument.
// - In each cluster, the available storage in virtual space is handled by a
//   local allocator using the buddy algorithm.
//
// TODO : In this first implementation one single - fixed size - vseg 
//        is allocated on demand in each cluster.
//        We should introduce the possibility to dynamically allocate
//        several vsegs in each cluster, using several mmap when required. 
/////////////////////////////////////////////////////////////////////////////////////////
// Free blocks organisation in each cluster :
// - All free blocks have a size that is a power of 2, larger or equal
//   to MALLOC_MIN_BLOCK_SIZE (typically 64 bytes).
// - All free blocks are aligned.
// - They are pre-classed in an array of linked lists, where all blocks in a
//   given list have the same size. 
// - The NEXT pointer implementing those linked lists is written 
//   in the first bytes of the block itself, using the unsigned int type.
// - The pointers on the first free block for each size are stored in an
//   array of pointers free[32] in the storage(x,y) descriptor.
/////////////////////////////////////////////////////////////////////////////////////////
// Allocation policy:
// - The block size required by the user can be any value, but the allocated
//   block size can be larger than the requested size:
// - The allocator computes actual_size, that is the smallest power of 2 
//   value larger or equal to the requested size AND larger or equal to
//   MALLOC_MIN_BLOCK_SIZE.
// - It pop the linked list of free blocks corresponding to actual_size,
//   and returns the block B if the list[actual_size] is not empty.
// - If the list[actual_size] is empty, it pop the list[actual_size * 2].
//   If a block B' is found, it breaks this block in 2 B/2 blocks, returns 
//   the first B/2 block and push the other B/2 block into list[actual_size].
// - If the list[actual_size * 2] is empty, it pop the list[actual_size * 4].
//   If a block B is found, it break this block in 3 blocks B/4, B/4 and B/2,
//   returns the first B/4 block, push the other blocks B/4 and B/2 into
//   the proper lists. etc... 
// - If no block satisfying the request is available it returns a failure
//   (NULL pointer).
// - This allocation policy has the nice following property:
//   If the vseg is aligned (the vseg base is a multiple of the
//   vseg size), all allocated blocks are aligned on the actual_size.
/////////////////////////////////////////////////////////////////////////////////////////
// Free policy:
// - Each allocated block is registered in an alloc[] array of unsigned char.
// - This registration is required by the free() operation, because the size
//   of the allocated block must be obtained from the base address of the block.  
// - The number of entries in this array is equal to the max number
//   of allocated block : MALLOC_LOCAL_STORE_SIZE / MALLOC_MIN_BLOCK_SIZE.
// - For each allocated block, the value registered in the alloc[] array
//   is log2( size_of_allocated_block ).
// - The index in this array is computed from the allocated block base address:
//      index = (block_base - vseg_base) / MALLOC_MIN_BLOCK_SIZE
// - The alloc[] array is stored at the end of heap segment. This consume
//   (1 / MALLOC_MIN_BLOCK_SIZE) of the total storage capacity.
/////////////////////////////////////////////////////////////////////////////////////////


#define MALLOC_INITIALIZED         0xBABEF00D   // magic number when initialised
#define MALLOC_MIN_BLOCK_SIZE      0x40         // 64 bytes
#define MALLOC_LOCAL_STORE_SIZE    0x2000000    // 32 Mbytes     
#define MALLOC_MAX_CLUSTERS        0x100        // 256 clusters

/////////////////////////////////////////////////////////////////////////////////////////
//               store(x,y) descriptor (one per cluster)
/////////////////////////////////////////////////////////////////////////////////////////

typedef struct malloc_store_s
{
    pthread_mutex_t mutex;           // lock protecting exclusive access to local heap
    unsigned int    initialized;     // initialised when value == MALLOC_INITIALIZED
    unsigned int    cxy;             // cluster identifier  
    unsigned int    store_base;      // store base address
    unsigned int    store_size;      // store size (bytes)
    unsigned int    alloc_base;      // alloc[] array base address
    unsigned int    alloc_size;      // alloc[] array size (bytes)
    unsigned int    free[32];        // array of addresses of first free block 
} 
malloc_store_t;

/*****************************************************************************************
 * This function allocates <size> bytes of memory in user space, and returns a pointer
 * to the allocated buffer. The pysical memory is allocated from store located in 
 * cluster identified by the <cxy> argument.
 *****************************************************************************************
 * @ size    : number of requested bytes.
 * @ cxy     : target cluster identifier.
 * @ returns a pointer on the allocated buffer if success / returns NULL if failure
 ****************************************************************************************/
void * remote_malloc( unsigned int size, 
                      unsigned int cxy );

/*****************************************************************************************
 * This function releases the memory buffer identified by the <ptr> argument,
 * to the store identified by the <cxy> argument.
 * It  does nothing, but displays an error message, if the ptr is illegal.
 *****************************************************************************************
 * @ ptr   : pointer on the released buffer.
 * @ cxy   : target cluster identifier.
 ****************************************************************************************/
void remote_free( void        * ptr,
                  unsigned int  cxy );

/*****************************************************************************************
 * This function releases the memory buffer identified by the <ptr> argument,
 * to the store located in cluster identified by the <cxy> argument, and allocates
 * a new buffer containing <size> bytes from this store. 
 * The content of the old buffer is copied to the new buffer, up to <size> bytes.
 * It displays an error message, but does nothing if the ptr is illegal.
 *****************************************************************************************
 * @ ptr     : pointer on the released buffer.
 * @ size    : new buffer requested size (bytes).
 * @ cxy     : target cluster identifier.
 * @ return a pointer on allocated buffer if success / return NULL if failure
 ****************************************************************************************/
void * remote_realloc( void        * ptr,
                       unsigned int  size,
                       unsigned int  cxy );

/*****************************************************************************************
 * This function allocates enough space for <count> objects that are <size> bytes
 * of memory each from the store located in cluster identied by the <cxy> argument.
 * The allocated memory is filled with bytes of value zero.
 *****************************************************************************************
 * @ count   : number of requested objects.
 * @ size    : number of bytes per object.
 * @ cxy     : target cluster identifier.
 * @ returns a pointer on allocated buffer if success / returns NULL if failure
 ****************************************************************************************/
void * remote_calloc( unsigned int count,
                      unsigned int size,
                      unsigned int cxy );

/********* Non standard (ALMOS-MKH specific) pthread_parallel_create() syscall  *********/

//////////////////////////////////////////////////////////////////////////////////////////
// This syscall can be used to parallelize the creation, and the termination 
// of a parallel multi-threaded application. 
// It removes in the main thread the sequencial loop that creates the N working threads 
// (N pthread_create() ), and removes also the sequencial loop that waits completion 
// of these N working threads (N pthread_join() ).
// It creates one <work> thread (in detached mode) per core in the target architecture.
// Each <work> thread is identified by a continuous [tid] index. 
// For a regular architecture, defined by the [x_size , y_size , ncores] parameters,
// the number of working threads can be the simply computed as (x_size * y_size * ncores), 
// and the coordinates[x,y,lid] of the core running the thread[tid] cand be directly 
// derived from the [tid] value with the following relations:
//     . cid = (x * y_size) + y
//     . tid = (cid * ncores ) + lid
//     . lid = tid % ncores
//     . cid = tid / ncores
//     . y   = cid % y_size
//     . x   = cid / y_size
// The pthread_parallel_create() function returns only when all <work> threads completed
// (successfully or not).
//
// WARNING : The function executed by the working thread is application specific,
// but the structure defining the arguments passed to this function is imposed.
// The "pthread_parallel_work_args_t" structure is defined below, and contains
// two fields: the tid value, and a pointer on a pthread_barrier_t.
// This barrier must be used by each working thread to signal completion before exit.
// The global variables implementing these stuctures for each working thread
// are allocated and initialised by the pthread_parallel_create() function.
//
// Implementation note: the pthread_parallel_create()a function creates a distributed
// quad-tree (DQT) of <build> threads covering all cores required to execute the parallel
// application. This quad tree is entirely defined by the root_level parameter. 
// Depending on the hardware topology, this DQT can be truncated, (i.e. some
// parent nodes can have less than 4 chidren), if (x_size != y_size), or if one size
// is not a power of 2. Each <build> thread is identified by two indexes [cid][level].
// Each <build> thread makes the following tasks:
// 1) It calls the pthread_create() function to create up to 4 children threads, that
//    are <work> threads when (level == 0), or <build> threads, when (level > 0).
// 2) It allocates and initializes the barrier, used to block the parent thread until
//    children completion.
// 3) It calls the pthread_barrier_wait( self ) to wait until all children threads
//    completed (successfully or not).
// 4) It calls the pthread_barrier_wait( parent ) to unblock the parent thread.
//////////////////////////////////////////////////////////////////////////////////////////

/*****************************************************************************************
 *    structure defining the arguments for the <build> thread function
 ****************************************************************************************/
typedef struct pthread_parallel_build_args_s           
{
    unsigned char        cid;                    // this <build> thread cluster index
    unsigned char        level;                  // this <build> thread level in quad-tree
    unsigned char        parent_cid;             // parent <build> thread cluster index
    pthread_barrier_t  * parent_barrier;         // pointer on parent <build> thread barrier
    unsigned char        root_level;             // quad-tree root level 
    void               * work_func;              // pointer on working thread function
    unsigned int         x_size;                 // platform global parameter
    unsigned int         y_size;                 // platform global parameter
    unsigned int         ncores;                 // platform global parameter
    unsigned int         error;                  // return value : 0 if success 
}
pthread_parallel_build_args_t;

/*****************************************************************************************
 *    structure defining the arguments for the <work> thread function
 ****************************************************************************************/
typedef struct pthread_parallel_work_args_s           
{
    unsigned int         tid;                    // thread identifier
    pthread_barrier_t  * barrier;                // to signal completion
}
pthread_parallel_work_args_t;           

/*****************************************************************************************
 * This blocking function creates N working threads identified by the [tid] continuous
 * index, that execute the code defined by the <work_func> argument, and returns only 
 * when all working threads completed.
 * The number N of created threads is entirely defined by the <root_level> argument,
 * that defines an abstract quad-tree, with a square base : root_level in [0,1,2,3,4],
 * side in [1,2,4,8,16], nclusters in [1,4,16,64,256]. This base is called  macro_cluster.
 * A working thread is created on all cores contained in this abstract macro-cluster.
 * The actual number of physical clusters containing cores can be smaller than the number
 * of clusters covered by the abstract quad tree.
 * All threads execute the same <work_func> function, on different arguments, that are 
 * specified as an array of structures pthread_parallel_work_args_t, allocated and
 * initialised by this function.
 *****************************************************************************************
 * @ root_level            : [in]  DQT root level in [0,1,2,3,4].
 * @ work_func             : [in]  pointer on start function.
 * @ return 0 if success / return -1 if failure.
 ****************************************************************************************/
int pthread_parallel_create( unsigned int         root_level,
                             void               * work_func );



/********* Non standard (ALMOS-MKH specific) Frame Buffer access syscalls   *************/

//////////////////////////////////////////////////////////////////////////////////////////
// The following system calls can be used to access the SoCLib Frame Buffer, that
// is a very simple graphic controller, that is seen by the software as a single
// buffer of <height> lines of <width> pixels. 
//////////////////////////////////////////////////////////////////////////////////////////

/*****************************************************************************************
 * This function returns in the <width>, <height> and <type> arguments 
 * the implementation specific frame buffer features.
 *****************************************************************************************
 * @ width   : [out] number of pixels per line.
 * @ height  : [out] number of lines.
 * @ type    : [out] pixel encoding type.
 * @ returns 0 if success / returns -1 if not found.
 ****************************************************************************************/
int fbf_get_config( unsigned int * width,
                    unsigned int * height,
                    unsigned int * type );

/*****************************************************************************************
 * This function creates a new window defined by the <l_zero>, <p_zero>, <nlines>,
 * and <npixels> arguments for the calling process, and register the process PID, the
 * allocated <wid>, and the window size and coordinates in the FBF device descriptor.
 * It returns in the output argument <buffer> the pointer on the buffer associated to
 * the window, mapped in user space.
 * Warning : Both pixel [p_zero,l_zero] and pixel [p_zero+npixels-1,l_zero+nlines-1]
 * must be contained in the frame buffer.
 *****************************************************************************************
 * @ l_zero  : [in]  first line index in FBF coordinates system.
 * @ p_zero  : [in]  first pixel index in FBF coordinates system.
 * @ nlines  : [in]  number of lines in the window.
 * @ npixels : [in]  number of pixels per line in the window.
 * @ buffer  : [out] pointer on 
 * @ returns <wid> if success / returns -1 if illegal size or coordinates.
 ****************************************************************************************/
int fbf_create_window( unsigned int   l_zero,
                       unsigned int   p_zero,
                       unsigned int   nlines,
                       unsigned int   npixels,
                       void        ** buffer );

/*****************************************************************************************
 * This function deletes an existing window, identified by the <wid> argument.
 * The calling process must be the window owner.
 *****************************************************************************************
 * @ wid     : window identifier.
 * @ returns 0 if success / returns -1 if not found or process not owner.
 ****************************************************************************************/
int fbf_delete_window( unsigned int  wid );

/*****************************************************************************************
 * This function refreshes in FBF all lines of a window identified by the <wid> argument,
 * when the line index is in the interval [line_first,line_last[.
 * It scans all registered windows to take into account the windows overlap.
 *****************************************************************************************
 * @ wid        : window identifier.
 * @ line_first : first line index.
 * @ line_last  : last line index (excluded).
 * @ returns 0 if success / returns -1 if illegal argument.
 ****************************************************************************************/
int fbf_refresh_window( unsigned int   wid,
                        unsigned int   line_first,
                        unsigned int   line_last );

/*****************************************************************************************
 * This function changes the size of a window identified by the <wid> argument as defined
 * by the <width> and height> arguments. The calling process must be the window owner.
 * WARNING : Both pixel [p_zero,l_zero] and pixel [p_zero+width-1,l_zero+height-1]
 * must be contained in the frame buffer.
 *****************************************************************************************
 * @ wid     : window identifier.
 * @ width   : number of pixels in line (one byte per pixel).
 * @ height  : number of lines in window.
 * @ returns 0 if success / returns -1 if illegal arguments.
 ****************************************************************************************/
int fbf_resize_window( unsigned int   wid,
                       unsigned int   width,
                       unsigned int   height );

/*****************************************************************************************
 * This function changes the <l_zero> & <p_zero> coordinates of a window identified
 * by the <wid> argument. The calling process must be the window owner.
 * WARNING : Both pixel [p_zero,l_zero] and pixel [p_zero+width-1,l_zero+height-1]
 * must be contained in the frame buffer.
 *****************************************************************************************
 * @ wid     : window identifier.
 * @ l_zero  : new first line index in FBF.
 * @ p_zero  : new first pixel index in FBF.
 * @ returns 0 if success / returns -1 if illegal arguments.
 ****************************************************************************************/
int fbf_move_window( unsigned int  wid,
                     unsigned int  l_zero,
                     unsigned int  p_zero );

/*****************************************************************************************
 * This syscall is deprecated (january 2020 AG)
 * This blocking function moves <length> pixels from the user buffer defined by <buffer>
 * argument to the frame buffer, starting at <offset> in FBF.
 *****************************************************************************************
 * @ buffer  : pointer on buffer in user space.
 * @ length  : number of pixels (one byte per pixel).
 * @ offset  : first pixel index in window.
 * @ returns 0 if success / returns -1 if illegal arguments.
 ****************************************************************************************/
int fbf_write( void         * buffer,
               unsigned int   length,
               unsigned int   offset );

/*****************************************************************************************
 * This syscall is deprecated (january 2020 AG)
 * This blocking function moves <length> pixels from the frame buffer, starting at 
 * <offset> in FBF, to the user buffer defined by <buffer>  argument.
 *****************************************************************************************
 * @ buffer  : pointer on buffer in user space.
 * @ length  : number of pixels (one byte per pixel).
 * @ offset  : first pixel index in window.
 * @ returns 0 if success / returns -1 if illegal arguments.
 ****************************************************************************************/
int fbf_read( void         * buffer,
              unsigned int   length,
              unsigned int   offset );


#endif /* _LIBALMOSMKH_H_ */

