/*
 * sys_chmod.c - Change file access rights.
 *
 * Author       Alain Greiner  (2016,2017,2018,2019,2020)
 *
 * Copyright (c) 2015 UPMC Sorbonne Universites
 *
 * This file is part of ALMOS-MKH.
 *
 * ALMOS-MKH is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2.0 of the License.
 *
 * ALMOS-MKH is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with ALMOS-MKH; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */

#include <hal_kernel_types.h>
#include <hal_uspace.h>
#include <vfs.h>
#include <vmm.h>
#include <printk.h>
#include <thread.h>
#include <process.h>
#include <syscalls.h>

///////////////////////////////////
int sys_chmod( char     * pathname,
               uint32_t   rights __attribute__((unused)) )
{
    char        kbuf[CONFIG_VFS_MAX_PATH_LENGTH];

    thread_t  * this    = CURRENT_THREAD;

#if DEBUG_SYS_CHMOD || DEBUG_SYSCALLS_ERROR || CONFIG_INSTRUMENTATION_SYSCALLS
process_t * process  = this->process;
uint64_t    tm_start = hal_get_cycles();
#endif

    // check pathname length
    if( hal_strlen_from_uspace( pathname ) >= CONFIG_VFS_MAX_PATH_LENGTH )
    {

#if DEBUG_SYSCALLS_ERROR
if( DEBUG_SYSCALLS_ERROR < (uint32_t)tm_start )
printk("\n[ERROR] in %s : thread[%x,%x] / pathname too long\n",
__FUNCTION__, this->trdid, process->pid );
#endif
        this->errno = ENFILE;
        return -1;
    }

    // copy pathname in kernel space
    hal_strcpy_from_uspace( XPTR( local_cxy , kbuf ),
                            pathname,
                            CONFIG_VFS_MAX_PATH_LENGTH );




    printk("\n[ERROR] in %s : not implemented yet\n", __FUNCTION__ );
    return -1;




    hal_fence();

#if (DEBUG_SYS_CHMOD || CONFIG_INSTRUMENTATION_SYSCALLS)
uint64_t     tm_end = hal_get_cycles();
#endif

#if DEBUG_SYS_CHMOD
if( DEBUG_SYS_CHMOD < tm_end )
printk("\n[%s] thread[%x,%x] exit for / cycle %d\n",
__FUNCTION__, process->pid, this->trdid, (uint32_t)tm_end );
#endif
 
#if CONFIG_INSTRUMENTATION_SYSCALLS
hal_atomic_add( &syscalls_cumul_cost[SYS_CHMOD] , tm_end - tm_start );
hal_atomic_add( &syscalls_occurences[SYS_CHMOD] , 1 );
#endif

    return 0;

}  // end sys_chmod()
