/*
 * sys_thread_exit.c - terminates the execution of calling thread
 * 
 * Authors   Alain Greiner (2016,2017,2018,2019)
 *
 * Copyright (c) UPMC Sorbonne Universites
 *
 * This file is part of ALMOS-MKH.
 *
 * ALMOS-MKH is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2.0 of the License.
 *
 * ALMOS-MKH is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with ALMOS-MKH; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */

#include <hal_kernel_types.h>
#include <hal_irqmask.h>
#include <thread.h>
#include <process.h>
#include <core.h>
#include <vmm.h>
#include <scheduler.h>
#include <printk.h>

#include <syscalls.h>

/////////////////////////////////////////
int sys_thread_exit( void * exit_status )
{
    error_t     error;
    vseg_t    * vseg;

	thread_t  * this      = CURRENT_THREAD;
    trdid_t     trdid     = this->trdid;
    process_t * process   = this->process;
    pid_t       pid       = process->pid;
    
    // check exit_value pointer in user space if required
    if( exit_status != NULL )
    {
        error = vmm_get_vseg( process , (intptr_t)exit_status  , &vseg );

    	if( error )
	    {

#if DEBUG_SYSCALLS_ERROR
printk("\n[ERROR] in %s : exit_status buffer %x unmapped / thread[%x,%x]\n",
__FUNCTION__, (intptr_t)exit_status, process->pid, this->trdid );
#endif
            this->errno = EINVAL;
	        return -1;
        }
    }

	// check busylocks
    uint32_t count = this->busylocks;
	if( count )
	{

#if DEBUG_SYSCALLS_ERROR
printk("\n[ERROR] in %s : busylocks count = %d  / thread[%x,%x]\n",
__FUNCTION__ , count, process->pid, this->trdid );
#endif
            this->errno = EINVAL;
	        return -1;
    }

    // If calling thread is the main thread, the process must be deleted. 
    // => delete all process threads and synchronize with parent process.
    // If calling thread is not the main thread, only this thread must be deleted.
    // => register exit_status in thread descriptor, block the thread, 
    //    mark it for delete, and synchronize with joining thread. 

    if( (CXY_FROM_PID( pid ) == local_cxy) && (LTID_FROM_TRDID(trdid) == 0) )  // main
    {

#if DEBUG_SYS_THREAD_EXIT
uint64_t     tm_start = hal_get_cycles();
if( DEBUG_SYS_THREAD_EXIT < tm_start )
printk("\n[%s] thread[%x,%x] is main => delete process / cycle %d\n",
__FUNCTION__ , pid , trdid , (uint32_t)tm_start );
#endif
        // delete process
        sys_exit( 0 );
    }
    else                                                             // not the main
    {

#if DEBUG_SYS_THREAD_EXIT
uint64_t     tm_start = hal_get_cycles();
if( DEBUG_SYS_THREAD_EXIT < tm_start )
printk("\n[%s] thread[%x,%x] is not main => delete thread / cycle %d\n",
__FUNCTION__ , pid , trdid , (uint32_t)tm_start );
#endif
        // register exit_status in thread descriptor
        this->exit_status = exit_status;

        // block calling thread and mark it for delete,
        thread_delete_request( XPTR( local_cxy , this ) , false );   // not forced

        // deschedule
        sched_yield( "suicide after thread_exit" );
    }

    return 0;   // never executed but required by compiler

}  // end sys_thread exit
