/*
 * remote_buf.h -  Remotely accessible, circular buffer definition.
 *
 * Authors : 
 *           Alain Greiner (2016,2017,2018,2019,2020)
 *
 * Copyright (c) UPMC Sorbonne Universites
 *
 * This file is part of ALMOS-MKH.
 *
 * ALMOS-MHH is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2.0 of the License.
 *
 * ALMOS-MKH is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with ALMOS-MKH; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */

#ifndef _REMOTE_BUFFER_H_
#define _REMOTE_BUFFER_H_

#include <kernel_config.h>
#include <hal_kernel_types.h>

/************************************************************************************
 * This structure and the associated access functions define a remotely accessible,
 * kernel buffer, handled as a single-reader & single-writer FIFO. 
 * This kernel buffer is implementes as an array of bytes. The size must be a power
 * of 2, dynamically defined by an argument of the "remote_buf_init()" function. 
 * The "put()" and "get()" functions defined below move the content of another
 * buffer (can be in kernel or in user space) to/from this circular kernel buffer.
 * This structure is NOT protected by a lock, as the only shared variable is the 
 * "sts" variable, that is updated by the hal_remote_atomic_add() primitive.
 *
 * - It is used to implement the three socket buffers (rx_buf / r2tq / crqq).
 * - It is also used to implement the inter-process communications channels
 *   based on named "fifos", or unamed "pipes".
 ***********************************************************************************/

typedef struct remote_buf_s
{
    uint32_t   order;   /*! ln2( size of data buffer in bytes)                     */
	uint32_t   rid;     /*! first empty slot index                                 */
	uint32_t   wid;     /*! first non-empty slot index                             */
    uint32_t   sts;     /*! current number of non empty slots                      */
	uint8_t  * data;    /*! local pointer on local data buffer                     */
}
remote_buf_t;

/************************************************************************************
 * This function allocates memory for a remote_buf descriptor in cluster 
 * defined by the <cxy> argument, and return a local pointer on this descriptor.
 ************************************************************************************
 * @ cxy    : target cluster identifier.
 * @ return local pointer on remote_buf descriptor.
 ***********************************************************************************/
remote_buf_t * remote_buf_alloc( cxy_t  cxy );

/************************************************************************************
 * This function initializes a remote_buf descriptor identified by the <buf_xp>
 * argument. It allocates memory for the data, as defined by the <order> argument,
 * and initializes the buffer as empty. It must be called by a local thread.
 * The <order> value cannot be larger than 31.
 ************************************************************************************
 * @ buf_xp    : [in] extended pointer on buffer descriptor.
 * @ order     : [in] ln2( number of bytes in buffer) / must be in [0,31].
 * @ return  0 if success / return -1 if memory failure or illegal order.
 ***********************************************************************************/
error_t remote_buf_init( xptr_t   buf_xp,
                         uint32_t order );

/************************************************************************************
 * This function releases memory allocated for the data buffer of an embedded
 * remote-buf descriptor identified by the <buf_xp> argument, when the "data"
 * pointer is not NULL. It does nothing if the data buffer was not allocated. 
 * WARNING : do NOT use this function for a dynamically allocated remote_buf.
 ************************************************************************************
 * @ buf_xp      : [in] extended pointer on buffer descriptor.
 ***********************************************************************************/
void remote_buf_release_data( xptr_t buf_xp );

/************************************************************************************
 * This function releases both the memory allocated for the data buffer, and the
 * memory allocated for the remote_buf descriptor, for a remote-buf identified 
 * by the <buf_xp> argument. 
 * WARNING : do NOT use this function an embedded remote_buf.
 ************************************************************************************
 * @ buf_xp      : [in] extended pointer on buffer descriptor.
 ***********************************************************************************/
void remote_buf_destroy( xptr_t buf_xp );

/************************************************************************************
 * This function initialises as empty a buffer identified by the <buf_xp> argument.
 * It can be called by a thread runing in any cluster.
 ************************************************************************************
 * @ buf_xp    : [in] extended pointer on remote buffer descriptor.
 ***********************************************************************************/
void remote_buf_reset( xptr_t buf_xp );

/************************************************************************************
 * This function moves <nbytes> from a remote_buf_t identified by the <buf_xp>
 * argument to an user buffer identified by the <u_buf> argument.
 * It uses the hal_copy_to_uspace() function, and updates "sts" and "ptr".
 * The calling function is supposed to previously check the buffer status. 
 * It can be called by a thread runing in any cluster.
 ************************************************************************************
 * @ buf_xp    : [in] extended pointer pointer on remote buffer descriptor.
 * @ u_buf     : [in] pointer on destination user buffer in user space.
 * @ nbytes    : [in] number of bytes to move.
 * @ return 0 if success / return -1 if not enough bytes in buffer.
 ***********************************************************************************/
error_t remote_buf_get_to_user( xptr_t    buf_xp,
                                uint8_t * u_buf,
                                uint32_t  nbytes );

/************************************************************************************
 * This function moves <nbytes> from a remote_buf_t identified by the <buf_xp>
 * argument to a local kernel buffer identified by the <k_buf> argument.
 * It uses an hal_remote_memcpy() function, and updates the "sts" and "ptr" fields.
 * The calling function is supposed to previously check the buffer status. 
 * It can be called by a thread runing in any cluster.
 ************************************************************************************
 * @ buf_xp    : [in] extended pointer pointer on remote buffer descriptor.
 * @ k_buf     : [in] local pointer on local destination kernel buffer.
 * @ nbytes    : [in] number of bytes to move.
 * @ return 0 if success / return -1 if not enough bytes in buffer.
 ***********************************************************************************/
error_t remote_buf_get_to_kernel( xptr_t    buf_xp,
                                  uint8_t * k_buf,
                                  uint32_t  nbytes );

/************************************************************************************
 * This function moves <nbytes> from an user buffer identified by the <u_buf> 
 * argument to a remote_buf_t identified by the <buf_xp> argument.
 * It uses the hal_copy_from_uspace() function, and updates "sts" and "ptw"..
 * The calling function is supposed to previously check the buffer status. 
 * It can be called by a thread runing in any cluster.
 ************************************************************************************
 * @ buf_xp    : [in] extended pointer pointer on remote buffer descriptor.
 * @ u_buf     : [in] pointer on source user buffer in user space.
 * @ nbytes    : [in] number of bytes to move.
 * @ return 0 if success / return -1 if not enough bytes in buffer.
 ***********************************************************************************/
error_t remote_buf_put_from_user( xptr_t    buf_xp,
                                  uint8_t * u_buf,
                                  uint32_t  nbytes );

/************************************************************************************
 * This function moves <nbytes> from a local kernel buffer identified by the <k_buf> 
 * argument to a remote_buf_t identified by the <buf_xp> argument.
 * It uses an hal_remote_memcpy() function, and updates the "sts" and "ptw" fields.
 * The calling function is supposed to previously check the buffer status. 
 * It can be called by a thread runing in any cluster.
 ************************************************************************************
 * @ buf_xp    : [in] extended pointer pointer on remote buffer descriptor.
 * @ k_buf     : [in] local pointer on local source kernel buffer.
 * @ nbytes    : [in] number of bytes to move.
 * @ return 0 if success / return -1 if not enough bytes in buffer.
 ***********************************************************************************/
error_t remote_buf_put_from_kernel( xptr_t    buf_xp,
                                    uint8_t * k_buf,
                                    uint32_t  nbytes );

/************************************************************************************
 * This function returns the current number of bytes stored in buffer.
 ************************************************************************************
 * @ buf_xp    : [in] extended pointer pointer on remote buffer descriptor.
 * @ return current number of bytes in buffer.
 ***********************************************************************************/
uint32_t remote_buf_status( xptr_t  buf_xp );

#endif	/* _REMOTE_BUFFER_H_ */
