/*
 * soclib_dma.c - soclib Multi Channels DMA driver implementation
 *
 * Author     Alain Greiner (2017,2018,2019,2020)

 * Copyright (c) UPMC Sorbonne Universites
 *
 * This file is part of ALMOS-MKH.
 *
 * ALMOS-MKH is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2.0 of the License.
 *
 * ALMOS-MKH is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with ALMOS-MKH; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */

#include <hal_kernel_types.h>
#include <chdev.h>
#include <dev_dma.h>
#include <thread.h>
#include <soclib_dma.h>

///////////////////////////////////////
void soclib_dma_init( chdev_t * chdev )
{
    // get hardware device cluster and local pointer
    cxy_t      dma_cxy  = GET_CXY( chdev->base );
    uint32_t * dma_ptr  = (uint32_t *)GET_PTR( chdev->base );

    // set driver specific fields in chdev descriptor
    chdev->cmd = &soclib_dma_cmd;
    chdev->isr = &soclib_dma_isr;

    // disable interrupts
	hal_remote_s32( XPTR( dma_cxy , dma_ptr + DMA_IRQ_DISABLED ) , 1 );

} // soclib_dma_init()

//////////////////////////////////////////////////////////////////
void __attribute__ ((noinline)) soclib_dma_cmd( xptr_t thread_xp )
{
    bool_t     sync;
    xptr_t     dev_xp;       // extended pointer on DMA devive
    xptr_t     dst_xp;       // extended pointer on destination buffer
    xptr_t     src_xp;       // extended pointer on source buffer
    uint32_t   size;         // buffer size
    uint32_t   status;       // DMA status

    // get client thread cluster and local pointer
    cxy_t      client_cxy = GET_CXY( thread_xp );
    thread_t * client_ptr = (thread_t *)GET_PTR( thread_xp );

#if (DEBUG_HAL_IOC_RX || DEBUG_HAL_IOC_TX)
uint32_t    cycle        = (uint32_t)hal_get_cycles();
thread_t  * this         = CURRENT_THREAD;
process_t * process      = hal_remote_lpt( XPTR( th_cxy , &th_ptr->process ) );
pid_t       client_pid   = hal_remote_l32( XPTR( th_cxy , &process->pid ) );
trdid_t     client_trdid = hal_remote_l32( XPTR( th_cxy , &th_ptr->trdid ) );
#endif 

// TODO both the client and the server threads are allways local,
// we could replace all these remote accesses by local accesses !!!  [AG]

    // get command arguments and extended pointer on DMA device descriptor
    sync   =         hal_remote_l32( XPTR( client_cxy , &client_ptr->dma_cmd.sync   ) );
    dev_xp = (xptr_t)hal_remote_l64( XPTR( client_cxy , &client_ptr->dma_cmd.dev_xp ) );
    dst_xp = (xptr_t)hal_remote_l64( XPTR( client_cxy , &client_ptr->dma_cmd.dst_xp ) );
    src_xp = (xptr_t)hal_remote_l64( XPTR( client_cxy , &client_ptr->dma_cmd.src_xp ) );
    size   =         hal_remote_l32( XPTR( client_cxy , &client_ptr->dma_cmd.size   ) );

    // get DMA device cluster and local pointer
    cxy_t     dev_cxy = GET_CXY( dev_xp );
    chdev_t * dev_ptr = (chdev_t *)GET_PTR( dev_xp );

    // get pointers on DMA peripheral
    xptr_t     dma_xp  = hal_remote_l32( XPTR( dev_cxy , &dev_ptr->base ) );
    cxy_t      dma_cxy = GET_CXY( dma_xp );
    uint32_t * dma_ptr = GET_PTR( dma_xp );

    // get DMA channel index and channel base address
    uint32_t * base = dma_ptr + DMA_SPAN * dev_ptr->channel;

    // split dst and src buffers addresses in two 32 bits words
    uint32_t   dst_lsb = (uint32_t)(dst_xp);
    uint32_t   dst_msb = (uint32_t)(dst_xp>>32);
    uint32_t   src_lsb = (uint32_t)(src_xp);
    uint32_t   src_msb = (uint32_t)(src_xp>>32);

    // set SOCLIB_DMA registers and launch tranfer operation
    hal_remote_s32( XPTR( dma_cxy , base + DMA_SRC     ) , src_lsb ); 
    hal_remote_s32( XPTR( dma_cxy , base + DMA_SRC_EXT ) , src_msb ); 
    hal_remote_s32( XPTR( dma_cxy , base + DMA_DST     ) , dst_lsb );
    hal_remote_s32( XPTR( dma_cxy , base + DMA_DST_EXT ) , dst_msb );
    hal_remote_s32( XPTR( dma_cxy , base + DMA_LEN_STS ) , size    );

 #if DEBUG_HAL_DMA
if( DEBUG_HAL_DMA < cycle )
printk("\n[%s] thread[%x,%x] launched DMA for client thread[%x,%x] / cycle %d\n",
__FUNCTION__, this->process->pid, this->trdid, client_pid, client_trdid, cycle );
#endif

    // waiting policy  depends on the command type
    // - for an asynchronous command, this function is called by the server thread
    //   => block and deschedule after launching the transfer. 
    //   The operation status is reported in the command by the ISR, and the
    //   server thread is re-activated by the ISR.
    // - for a synchronous command, this function is called by the client thread
    //   => mask the DMA_IRQ and poll the DMA status register until transfer completion,
    //   and reports status in the command when the transfer is completed.

    if( sync )                            // client thread poll status until completion
    {
        while( 1 )
        {
            status = hal_remote_l32( XPTR( dma_cxy , base + DMA_LEN_STS ) );

            if( (status == DMA_SUCCESS) || (status == DMA_IDLE) )
            {
                // set operation status in command
                hal_remote_s32( XPTR( client_cxy , &client_ptr->dma_cmd.error ) , 0 );

#if DEBUG_HAL_DMA
cycle = (uint32_t)hal_get_cycles();
if( DEBUG_HAL_DMA < cycle )
printk("\n[%s] thread[%x,%x] exit after SYNC success / cycle %d\n",
__FUNCTION__, this->process->pid, this->trdid, cycle );
#endif
                // exit while
                break;
            }
            else if( (status == DMA_ERROR_READ) || (status == DMA_ERROR_WRITE) )
            {
                // set operation status in command
                hal_remote_s32( XPTR( client_cxy , &client_ptr->dma_cmd.error ) , 1 );

#if DEBUG_HAL_DMA
cycle = (uint32_t)hal_get_cycles();
if( DEBUG_HAL_DMA < cycle )
printk("\n[%s] thread[%x,%x] exit after SYNC failure / cycle %d\n",
__FUNCTION__, this->process->pid, this->trdid, cycle );
#endif
                // exit while
                break;
            }
        }
    }    
    else                                // server thread block and deschedule
    {
        // server thread blocks on ISR
        thread_block( XPTR( local_cxy , CURRENT_THREAD ) , THREAD_BLOCKED_ISR );

        // enable DMA interrupts
	    hal_remote_s32( XPTR( dma_cxy , dma_ptr + DMA_IRQ_DISABLED ) , 0 );

        // server thread deschedules
        sched_yield("blocked on ISR");

        // disable DMA interrupts
	    hal_remote_s32( XPTR( dma_cxy , dma_ptr + DMA_IRQ_DISABLED ) , 1 );

#if DEBUG_HAL_DMA
cycle = (uint32_t)hal_get_cycles();
if( DEBUG_HAL_DMA < cycle )
printk("\n[%s] thread[%x,%x] exit after ASYNC / client thread[%x,%x] / cycle %d\n",
__FUNCTION__ , this->process->pid, this->trdid, client_pid, client_trdid, cycle );
#endif

    }
} // soclib_dma_cmd()

/////////////////////////////////////////////////////////////////
void __attribute__ ((noinline)) soclib_dma_isr( chdev_t * chdev )
{
    // get extended pointer on server thread
    xptr_t server_xp = XPTR( local_cxy , &chdev->server );

   // get extended pointer on client thread
    xptr_t root      = XPTR( local_cxy , &chdev->wait_root );
    xptr_t client_xp = XLIST_FIRST( root , thread_t , wait_list );

    // get client thread cluster and local pointer
    cxy_t      client_cxy = GET_CXY( client_xp );
    thread_t * client_ptr = (thread_t *)GET_PTR( client_xp );

    // get SOCLIB_DMA peripheral cluster and local pointer
    cxy_t      dma_cxy  = GET_CXY( chdev->base );
    uint32_t * dma_ptr  = (uint32_t *)GET_PTR( chdev->base );

    // build DMA channel base address
    uint32_t * base = dma_ptr + (DMA_SPAN * chdev->channel);

    // get DMA status register 
	uint32_t status = hal_remote_l32( XPTR( dma_cxy , base + DMA_LEN_STS ) );   

    // acknowledge IRQ
    hal_remote_s32( XPTR( dma_cxy , base + DMA_RESET ) , 0 );

    // set operation status in command
	error_t  error = ( status != DMA_SUCCESS ); 
    hal_remote_s32( XPTR( client_cxy , &client_ptr->dma_cmd.error ) , error );

    // unblock server thread 
    thread_unblock( server_xp , THREAD_BLOCKED_ISR );

} // soclib_dma_isr()



