/*
 * fatfs.h - FATFS file system API definition.
 *
 * Author    Alain Greiner (2016,2017,2018,2019,2020)
 *
 * Copyright (c) UPMC Sorbonne Universites
 *
 * This file is part of ALMOS-MKH.
 *
 * ALMOS-MKH is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2.0 of the License.
 *
 * ALMOS-MKH is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with ALMOS-MKH; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */

#ifndef _FATFS_H_
#define _FATFS_H_

#include <hal_kernel_types.h>
#include <remote_rwlock.h>
#include <vfs.h>
#include <dev_ioc.h>


/******************************************************************************************
 * The FATFS File System implements a FAT32 read/write file system.
 *
 * The FATFS specific extensions to the generic VFS are the following:
 * 1) The vfs_ctx_t "extend" field contains a local pointer on the local 
 *    fatfs_ctx_t structure.
 *
 * 2) The vfs_inode_t "extend" contains, for each inode,
 *    the first FATFS cluster_id (after cast to intptr).
 *
 * 3) The vfs_dentry_t "extend" field contains a local pointer on the local
 *    FATFS directory entry (32 bytes) in the directory mapper.
 *
 * In the FAT32 File System, the File Allocation Table is is actually an array 
 * of uint32_t slots. Each slot in this array contains the index (called cluster_id)
 * of another slot in this array, to form one linked list for each file stored on
 * device in the FAT32 File System. This index in the FAT array is also the index of
 * the FATFS cluster on the device. One FATFS cluster is supposed to contain one PPM page. 
 * For a given file, the entry point in the FAT is the cluster_id of the FATFS cluster 
 * containing the first page of the file, but it can be any cluster_id already allocated
 * to the file. 
 *****************************************************************************************/
 
/*************** Partition Boot Sector Format **********************************/
//                                     offset |  length
#define BS_JMPBOOT                          0 ,  3
#define BS_OEMNAME                          3 ,  8
#define BPB_BYTSPERSEC                     11 ,  2
#define BPB_SECPERCLUS                     13 ,  1
#define BPB_RSVDSECCNT                     14 ,  2
#define BPB_NUMFATS                        16 ,  1
#define BPB_ROOTENTCNT                     17 ,  2
#define BPB_TOTSEC16                       19 ,  2
#define BPB_MEDIA                          21 ,  1
#define BPB_FATSZ16                        22 ,  2
#define BPB_SECPERTRK                      24 ,  2
#define BPB_NUMHEADS                       26 ,  2
#define BPB_HIDDSEC                        28 ,  4
#define BPB_TOTSEC32                       32 ,  4
#define BPB_PARTITION_TABLE               446 , 64 

// FAT 32
#define BPB_FAT32_FATSZ32                  36 ,  4
#define BPB_FAT32_EXTFLAGS                 40 ,  2
#define BPB_FAT32_FSVER                    42 ,  2
#define BPB_FAT32_ROOTCLUS                 44 ,  4
#define BPB_FAT32_FSINFO                   48 ,  2
#define BPB_FAT32_BKBOOTSEC                50 ,  2
#define BS_FAT32_DRVNUM                    64 ,  1
#define BS_FAT32_BOOTSIG                   66 ,  1
#define BS_FAT32_VOLID                     67 ,  4
#define BS_FAT32_VOLLAB                    71 , 11
#define BS_FAT32_FILSYSTYPE                82 ,  8

// Partitions
#define FIRST_PARTITION_ACTIVE            446 ,  8
#define FIRST_PARTITION_BEGIN_LBA         454 ,  4
#define FIRST_PARTITION_SIZE              458 ,  4 
#define SECOND_PARTITION_ACTIVE           462 ,  8
#define SECOND_PARTITION_BEGIN_LBA        470 ,  4
#define SECOND_PARTITION_SIZE             474 ,  4
#define THIRD_PARTITION_ACTIVE            478 ,  8
#define THIRD_PARTITION_BEGIN_LBA         486 ,  4
#define THIRD_PARTITION_SIZE              490 ,  4
#define FOURTH_PARTITION_ACTIVE           494 ,  8
#define FOURTH_PARTITION_BEGIN_LBA        502 ,  4
#define FOURTH_PARTITION_SIZE             506 ,  4    
/*******************************************************************************/

#define MBR_SIGNATURE_POSITION            510 , 2
#define MBR_SIGNATURE_VALUE               0xAA55  

/************** FAT_FS_INFO SECTOR  ********************************************/
#define FS_SIGNATURE_VALUE_1              0x52526141
#define FS_SIGNATURE_VALUE_2              0x72724161
#define FS_SIGNATURE_VALUE_3              0x000055AA  
#define FS_SIGNATURE_POSITION_1           0   , 4  
#define FS_SIGNATURE_POSITION_2           484 , 4
#define FS_SIGNATURE_POSITION_3           508 , 4  
#define FS_FREE_CLUSTERS                  488 , 4
#define FS_FREE_CLUSTER_HINT              492 , 4
/*******************************************************************************/

#define DIR_ENTRY_SIZE          32
                   
#define NAME_MAX_SIZE           31

/******* SFN Directory Entry Structure (32 bytes) ******************************/
//                            offset | length
#define DIR_NAME                   0 , 11   // dir_entry name
#define DIR_ATTR                  11 ,  1   // attributes
#define DIR_NTRES                 12 ,  1   // reserved for the OS        
#define DIR_CRT_TIMES_TENTH       13 ,  1 
#define DIR_FST_CLUS_HI           20 ,  2   // cluster index 16 MSB bits
#define DIR_WRT_TIME              22 ,  2   // time of last write
#define DIR_WRT_DATE              24 ,  2   // date of last write
#define DIR_FST_CLUS_LO           26 ,  2   // cluster index 16 LSB bit
#define DIR_FILE_SIZE             28 ,  4   // dir_entry size (up to 4 Gbytes)
/*******************************************************************************/

/******* LFN Directory Entry Structure  (32 bytes) *****************************/
//                            offset | length
#define LDIR_ORD                   0 ,  1   // Sequence number (from 0x01 to 0x0f)    
#define LDIR_NAME_1                1 , 10   // name broken into 3 parts 
#define LDIR_ATTR                 11 ,  1   // attributes (must be 0x0F) 
#define LDIR_TYPE                 12 ,  1   // directory type (must be 0x00)
#define LDIR_CHKSUM               13 ,  1   // checksum of name in short dir  
#define LDIR_NAME_2               14 , 12 
#define LDIR_RSVD                 26 ,  2   // artifact of previous fat (must be 0)
#define LDIR_NAME_3               28 ,  4   
/*******************************************************************************/

/***********************  DIR_ATTR values  (attributes) ************************/
#define ATTR_READ_ONLY            0x01
#define ATTR_HIDDEN               0x02
#define ATTR_SYSTEM               0x04
#define ATTR_VOLUME_ID            0x08
#define ATTR_DIRECTORY            0x10
#define ATTR_ARCHIVE              0x20
#define ATTR_LONG_NAME_MASK       0x0f      // READ_ONLY|HIDDEN|SYSTEM|VOLUME_ID
/*******************************************************************************/

/********************* DIR_ORD special values **********************************/
#define FREE_ENTRY                0xE5     // this entry is free in the directory
#define NO_MORE_ENTRY             0x00     // no more entry in the directory
/*******************************************************************************/

/******************** CLuster Index Special Values *****************************/
#define FREE_CLUSTER              0x00000000
#define RESERVED_CLUSTER          0x00000001
#define BAD_CLUSTER               0x0FFFFFF7
#define END_OF_CHAIN_CLUSTER_MIN  0x0ffffff8
#define END_OF_CHAIN_CLUSTER_MAX  0x0fffffff
/*******************************************************************************/

/****  Forward declarations  ****/

struct mapper_s;
struct page_s;
struct vfs_ctx_s;
struct vfs_inode_s;
struct vfs_dentry_s;

/*****************************************************************************************
 * This structure defines a FATFS specific context extension to the VFS context.
 * This fatfs context is replicated in all clusters.
 * It contains read-only informations such as the total number of sectors in FAT region,
 * the number of bytes per sector, the number of sectors per cluster, the lba of FAT,
 * the lba of data region, the cluster_id for the root directory, and an extended 
 * pointer on the FAT mapper.
 *
 * WARNING 1 : All access to the FAT are protected by a remote_rwlock. 
 * - it is taken in READ mode by the fatfs_get_cluster() function to scan the
 *   linked list associated to a given inode. 
 * - it is taken in WRITE mode by the fatfs_cluster_alloc() and fatfs_release_inode()
 *   functions to modify the FAT in both the FAT mapper and on IOC device.
 *
 * WARNING 2 : Most fields are constant values, but <free_cluster_hint>, <free_clusters>,
 * <lock>, and the buffer pointed by the <fs_info_xp> are shared variables, that can
 * modified by any thread running in any cluster. The <fs_info_buffer>  contains
 * a copy of the FS_INFO sector, and is only allocated in the FAT cluster. 
 * It is used to synchronously update the free clusters info on IOC device.
 *  => For all these variables, only the values stored in the FAT cluster must be used.
 ****************************************************************************************/

typedef struct fatfs_ctx_s
{
    /* read-only constants replicated in all clusters                                   */
    uint32_t            fat_sectors_count;     /*! number of sectors in FAT region      */
    uint32_t            bytes_per_sector;      /*! number of bytes per sector           */
    uint32_t            sectors_per_cluster;   /*! number of sectors per cluster        */
    uint32_t            fat_begin_lba;         /*! lba of FAT region                    */
    uint32_t            cluster_begin_lba;     /*! lba of data region                   */
    uint32_t            fs_info_lba;           /*! lba of FS_INFO sector                */
    uint32_t            root_dir_cluster;      /*! cluster index for  root directory    */
    struct mapper_s   * fat_mapper;            /*! local pointer on FAT mapper          */

    /* shared variables (only the copy in FAT cluster must be used)                     */
    uint32_t            free_cluster_hint;     /*! free_cluster_hint + 1 is first free  */
    uint32_t            free_clusters;         /*! number of free clusters              */
    remote_rwlock_t     lock;                  /*! exclusive access to FAT              */
    uint8_t           * fs_info_buffer;        /*! local pointer on FS_INFO buffer      */
}
fatfs_ctx_t;

//////////////////////////////////////////////////////////////////////////////////////////
//              FATFS specific extern functions  
//////////////////////////////////////////////////////////////////////////////////////////

/*****************************************************************************************
 * This debug function display the content of the FATFS context copy in cluster
 * identified by the <cxy> argument.
 * This function can be called by any thread running in any cluster.
 *****************************************************************************************
 * @ cxy       :  target cluster identifier.
 ****************************************************************************************/
void fatfs_display_ctx( cxy_t cxy );

/*****************************************************************************************
 * This debug function access the FAT mapper to display the current FAT state,
 * as defined by the <min_slot>, and <nb_slots> arguments.
 * It display as many lines (8 slots par line) as required to display <nb_slots>,
 * starting from the <min_slot>. The displayed slots can spread on several FAT mapper
 * pages. It loads the missing pages from IOC to mapper if required.
 * This function can be called by any thread running in any cluster.
 *****************************************************************************************
 * @ min_slot   : first FATFS cluster index.
 * @ nb_slots   : number of slots (one slot is 4 bytes.
 ****************************************************************************************/
void fatfs_display_fat( uint32_t  min_slot,
                        uint32_t  nb_slots );

/*****************************************************************************************
 * This function checks the current values of the "free_clusters" and "free_cluster_hint" 
 * variables in the FS_INFO sector on IOC, versus the values stored in the fatfs context.
 * As these values are synchronously updated on IOC device at each modification, 
 * it does nothing if the values are equal. It updates the FS_INFO sector on IOC device,
 * and displays a warning message on TXT0 if they are not equal.
 * This function can be called by any thread running in any cluster. 
 *****************************************************************************************
 * @ return 0 if success / return -1 if failure during IOC device access.
 ****************************************************************************************/
error_t fatfs_check_free_info( void );

//////////////////////////////////////////////////////////////////////////////////////////
// Generic API: These functions are called by the kernel VFS,
//              and must be implemented by all File Systems.
//////////////////////////////////////////////////////////////////////////////////////////

/*****************************************************************************************
 * This fuction allocates memory for a FATFS context descriptor in a cluster
 * identified by the <cxy> argument.
 *****************************************************************************************
 * @ cxy   : target cluster identifier.
 * @ return an extended pointer on the created context / return XPTR_NULL if failure.
 ****************************************************************************************/
xptr_t  fatfs_ctx_alloc( cxy_t  cxy );

/*****************************************************************************************
 * This fuction initialize a fatfs context identified by the <fatfs_ctx_xp> argument
 * from informations found in the IOC device boot record. This initialisation includes
 * allocation of the FS_INFO buffer and creation of the FAT mapper in the same cluster.
 *****************************************************************************************
 * @ fatfs_ctx_xp   : extended pointer on fatfs context.
 * @ return 0 if success / return -1 if failure.
 ****************************************************************************************/
error_t  fatfs_ctx_init( xptr_t  fatfs_ctx_xp );

/*****************************************************************************************
 * This function releases memory dynamically allocated for the FATFS context.
 *****************************************************************************************
 * @ vfs_ctx_xp   : extended pointer on FATFS context.
 ****************************************************************************************/
void fatfs_ctx_destroy( xptr_t  fatfs_ctx_xp );

/*****************************************************************************************
 * This function implements the generic vfs_fs_add_dentry() function for the FATFS.
 *****************************************************************************************
 * This function introduces in a directory mapper identified by the <parent_inode_xp>
 * argument a new directory entry identified by the <dentry_ptr> argument.
 * The dentry descriptor and the associated inode descriptor must have been previously
 * allocated, initialized, and registered in the Inode Tree. 
 * The dentry descriptor defines the "name" field.
 * The inode descriptor defines the "type", "size", and "cluster_id" fields.
 * The "extension field" in dentry descriptor is set : index in the FAT32 directory.
 * All modified pages in the directory mapper are synchronously updated on IOC device.
 * This function can be called by any thread running in any cluster.
 *
 * Implementation note : this function works in two steps:
 * - It scan the set of 32 bytes FATFS directry entries, using two embedded loops  
 *   to find the end of directory (NO_MORE_ENTRY marker).
 * - Then it writes 3, 4, or 5 directory entries (depending on the name length), using 
 *   a 5 steps FSM (one state per entry to be written), and updates on IOC device the 
 *   modified pages.
 *****************************************************************************************
 * @ parent_inode_xp : [in]  extended pointer on parent directory inode.
 * @ dentry_ptr      : [in]  local pointer on dentry (in parent directory cluster).
 * @ index           : [out] index of the new entry in the FAT32 directory.
 * @ return 0 if success / return -1 if failure.
 ****************************************************************************************/
error_t fatfs_add_dentry( xptr_t                parent_inode_xp,
                          struct vfs_dentry_s * dentry_ptr );

/*****************************************************************************************
 * This function implements the generic vfs_fs_remove_dentry() function for the FATFS.
 *****************************************************************************************
 * This function updates a directory identified by the <parent_inode_xp> argument
 * to remove a directory entry identified by the <dentry_ptr> argument.
 * All modified pages in directory mapper are synchronously updated on IOC device.
 * This function can be called by any thread running in any cluster.
 *
 * Implementation note: this function uses the dentry extension to directly access 
 * the NORMAL directory entry and invalidate all involved LFN entries. Then it
 * updates the modified pages on IOC device.
 *****************************************************************************************
 * @ parent_inode_xp   : [in] extended pointer on parent directory inode.
 * @ dentry_ptr        : [in] local pointer on dentry (in parent directory cluster).
 * @ return 0 if success / return -1 if failure.
 ****************************************************************************************/
error_t fatfs_remove_dentry( xptr_t                parent_inode_xp,
                             struct vfs_dentry_s * dentry_ptr );

/*****************************************************************************************
 * This function implements the generic vfs_fs_new_dentry_from_mapper() for the FATFS.
 *****************************************************************************************
 * It scan a parent directory mapper, identified by the <parent_inode_xp> argument 
 * to find a directory entry name defined by the <dentry_ptr> argument, and completes
 * the initialization of the dentry and the associated child_inode descriptors,
 * from informations found in the parent directory mapper :
 * - It set the "type", "size", and "extend" fields in the child inode descriptor.
 * - It set the " extend" field in the dentry descriptor.
 * The child inode descriptor, and the dentry descriptor must have been previously
 * allocated and introduced in the Inode Tree.
 * This function can be called by any thread running in any cluster.
 *****************************************************************************************
 * @ parent_inode_xp : extended pointer on parent inode (directory).
 * @ dentry_ptr      : local pointer on new dentry (in parent inode cluster).
 * @ return 0 if success / return -1 if failure.
 ****************************************************************************************/
error_t fatfs_new_dentry_from_mapper( xptr_t                parent_inode_xp,
                                      struct vfs_dentry_s * dentry_ptr );

/*****************************************************************************************
 * This function implements the generic vfs_fs_new_dentry_to_mapper() for the FATFS.
 *****************************************************************************************
 * This function  introduces a brand new dentry identified by the <dentry_ptr> argument
 * in the mapper of a directory identified by the <parent_inode_xp> argument.
 * It is called by the vfs_lookup() function.
 * The child inode descriptor, and the dentry descriptor must have been previously
 * allocated and introduced in the Inode Tree. The dentry descriptor contains the name.
 * 1. It allocates a new FATFS cluster_id,
 * 2. It registers the allocated cluster_id in the child inode extension,
 * 3. It add a new entry (32 bytes) in the directory mapper,
 * This function can be called by any thread running in any cluster.
 *****************************************************************************************
 * @ parent_inode_xp : [in]  extended pointer on parent inode (directory).
 * @ dentry_ptr      : [in]  local pointer on dentry (in parent inode cluster).
 * @ return 0 if success / return -1 if failure.
 ****************************************************************************************/
error_t fatfs_new_dentry_to_mapper( xptr_t                parent_inode_xp,
                                    struct vfs_dentry_s * dentry_ptr );

/*****************************************************************************************
 * This function implements the generic vfs_fs_update_dentry() function for the FATFS.
 *****************************************************************************************
 * It update the "size" of a directory entry identified by the <dentry_ptr> argument in
 * the mapper of a directory identified by the <parent_inode_xp> argument, from the
 * value registered in the inode descriptor.
 * It scan the directory mapper to find the entry such as name == dentry_ptr->name.
 * It set the "size" field in the directory mapper, and updates the modified directory
 * page on the IOC device.
 * This function can be called by any thread running in any cluster.
 *
 * TODO the current implementation uses the fatfs_scan_directory to access the 
 * FAT32 directory by name. We can access directly this directory entry if we use
 * the dentry "extend" field...
 *****************************************************************************************
 * @ parent_inode_xp : extended pointer on inode (directory).
 * @ dentry_ptr      : local pointer on dentry (in parent directory cluster).
 * @ return 0 if success / return -1 if child not found.
 ****************************************************************************************/
error_t fatfs_update_dentry( xptr_t                parent_inode_xp,
                             struct vfs_dentry_s * dentry_ptr );

/*****************************************************************************************
 * This function implements the generic vfs_fs_get_user_dir() function for the FATFS.
 *****************************************************************************************
 * It is called by the remote_dir_create() function to scan the mapper of a directory 
 * identified by the <inode> argument, and copy up to <max_dirent> valid dentries to a 
 * local dirent array, defined by the <array> argument. The <min_dentry> argument defines
 * the index of the first dentry to be copied to the target dirent array.
 * This function returns in the <entries> buffer the number of dentries actually written,
 * and signals in the <done> buffer when the last valid entry has been found.
 * If the <detailed> argument is true, a dentry/inode couple that does not exist in
 * the Inode Tree is dynamically created, and all dirent fields are documented in the
 * dirent array. Otherwise, only the dentry name is documented.
 *
 * WARNING : It must be called by a thread running in the cluster containing the
 * target directory inode.
 *****************************************************************************************
 * @ parent_inode_xp  : [in]  extended pointer on directory inode.
 * @ array      : [in]  local pointer on array of dirents.
 * @ max_dirent : [in]  max number of slots in dirent array.
 * @ min_dentry : [in]  index of first dentry to be copied into array.
 * @ detailed   : [in]  dynamic inode creation if true.
 * @ entries    : [out] number of dentries actually copied into array.
 * @ done       : [out] Boolean true when last entry found.
 * @ return 0 if success / return -1 if failure.
 ****************************************************************************************/
error_t fatfs_get_user_dir( struct vfs_inode_s * inode,
                            struct dirent      * array, 
                            uint32_t             max_dirent,
                            uint32_t             min_dentry,
                            bool_t               detailed,
                            uint32_t           * entries,
                            bool_t             * done );

/*****************************************************************************************
 * This function implements the generic vfs_fs_sync_inode() function for the FATFS.
 *****************************************************************************************
 * It updates the FATFS on the IOC device for a given inode identified by 
 * the <inode_xp> argument. It scan all pages registered in the associated mapper,
 * and copies from mapper to device each page marked as dirty.
 * WARNING : The target <inode> cannot be a directory, because all modifications in a 
 * directory are synchronously done on the IOC device by the two fatfs_add_dentry() 
 * and fatfs_remove_dentry() functions.
 * This function can be called by any thread running in any cluster.
 *****************************************************************************************
 * @ inode_xp   : extended pointer on inode.
 * @ return 0 if success / return -1 if failure. 
 ****************************************************************************************/
error_t fatfs_sync_inode( xptr_t  inode_xp );

/*****************************************************************************************
 * This function implements the generic vfs_fs_sync_fat() function for the FATFS.
 *****************************************************************************************
 * It updates the FAT on the IOC device for the FAT itself.
 * It scan all clusters registered in the FAT mapper, and copies from mapper to device 
 * each page marked as dirty.
 * This function can be called by any thread running in any cluster. 
 *
 * Implementation note : this function uses the grdxt_remote_get_first() function
 * to test only the pages actually registered in the FAT mapper.
 *****************************************************************************************
 * @ return 0 if success / return -1 if failure during IOC device access.
 ****************************************************************************************/
error_t fatfs_sync_fat( void );

/*****************************************************************************************
 * This function implements the generic vfs_fs_release_inode() function for the FATFS.
 *****************************************************************************************
 * This function is used to remove a given file or directory from the FATFS file system.
 * It releases all clusters allocated to a file/directory identified by the <inode_xp>
 * argument. All released clusters are marked FREE_CLUSTER in the FAT mapper. 
 * This function calls the recursive function fatfs_cluster_release() to release 
 * the clusters in reverse order of the linked list (from last to first). 
 * When the FAT mapper has been updated, it calls the fatfs_sync_fat() function to
 * synchronously update all modified pages in the FAT mapper to the IOC device.
 * Finally the FS-INFO sector on the IOC device is updated.
 * This function can be called by any thread running in any cluster.
 *****************************************************************************************
 * @ inode_xp   : extended pointer on inode.
 * @ return 0 if success / return EIO if failure during device access.
 ****************************************************************************************/
error_t fatfs_release_inode( xptr_t inode_xp );

/*****************************************************************************************
 * This function implements the generic vfs_fs_move_page() function for the FATFS.
 *****************************************************************************************
 * This function moves a page from/to the mapper to/from the FATFS file system on device.
 * The page must have been previously allocated and registered in the mapper.   
 * The pointer on the mapper and the page index in file are found in the page descriptor.
 * It is used for both a regular file/directory mapper, and the FAT mapper.
 * - For the FAT mapper, it read/write the FAT region on IOC device.
 * - For a regular file, it scan the FAT mapper to get the cluster_id on IOC device,
 *   and read/write this cluster.
 * This function can be called by any thread running in any cluster.
 *
 * WARNING : For the FAT mapper, the inode field in the mapper MUST be NULL, as this
 *           is used to indicate that the corresponding mapper is the FAT mapper.
 *****************************************************************************************
 * @ page_xp   : extended pointer on page descriptor.
 * @ cmd_type  : IOC_READ / IOC_WRITE / IOC_SYNC_READ / IOC_SYNC_WRITE
 * @ return 0 if success / return EIO if error during device access. 
 ****************************************************************************************/
error_t fatfs_move_page( xptr_t          page_xp,
                         ioc_cmd_type_t  cmd_type );






#endif	/* _FATFS_H_ */
