/*
 * ksocket.h - kernel socket descriptor and API definition.
 * 
 * Authors  Alain Greiner    (2016,2017,2018,2019,2020)
 *
 * Copyright (c) UPMC Sorbonne Universites
 *
 * This file is part of ALMOS-MKH
 *
 * ALMOS-MKH is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2.0 of the License.
 *
 * ALMOS-MKH is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with ALMOS-MKH; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */

#ifndef _KSOCKET_H_
#define _KSOCKET_H_

#include <kernel_config.h>
#include <hal_kernel_types.h>
#include <xlist.h>
#include <remote_buf.h>
#include <remote_busylock.h>

/***************************************************************************************** 
 * This structure defines a kernel socket descriptor, used for both UDP or TCP sockets.
 * A socket is a private resource used by a most two user threads : one TX client
 * thread to send packets, and one RX client thread, to receive packets. The TX client
 * thread and the RX client thread can be the same thread.
 * 
 * When the Network Interface Controller contains several channels, the set of all
 * existing sockets is split in as many subsets as the number of NIC channels, in order
 * to parallelize the transfers. The distribution key defining the channel index
 * is computed from the (remote_addr/remote_port) couple: by the NIC hardware for the
 * RX packets; by the software for the TX packets, using a dedicated NIC driver function.
 * All sockets that have the same key share the same channel, and each socket is
 * therefore linked to two chdevs : NIC_TX[key] & NIC_RX[key].
 * The socket allows the NIC-TX and NIC_RX server threads to access various buffers:
 * - the kernel "tx_buf" buffer contains the data to be send by the TX server thread.
 *   It is dynamically allocated, and used as retransmission buffer when required.
 * - the kernel "rx_buf" buffer contains the data received by the RX server thread.
 *   It is allocated in socket and handled as a single writer / single reader FIFO.
 * - the kernel "r2t" buffer allows the RX server thread to make direct requests
 *   to the associated TX server (mainly used to handle the TCP ACKs).
 * - the kernel "crq" buffer allows to store concurrent remote client connect requests
 *   to a local server socket. It is allocated in socket. 
 *
 * The synchronisation mechanism between the client threads and the server threads
 * is different for the TX and RX directions:
 *
 * 1) TX stream
 *
 * - The internal API between the TX client thread and the NIC_TX server thread defines 
 *   four command types, stored in the "tx_cmd" variable of the socket descriptor:
 *   . SOCKET_TX_CONNECT : TCP client request to start the 3 steps connection handshake.
 *   . SOCKET_TX_ACCEPT  : TCP server request to accept one pending connection request. 
 *   . SOCKET_TX_SEND    : local (UDP/TCP) request to send data to a remote (UDP/TCP).
 *   . SOCKET_TX_CLOSE   : local TCP socket request remote TCP socket to close connection.
 * - All commands are blocking for the TX client thread: to make a command, the TX client 
 *   registers the command type in the socket "tx_cmd",field, set the "tx_valid" field,
 *   reset the "tx_error" field, and registers itself in the "tx_client" field.
 *   Then, it unblocks the TX server thread from the BLOCKED_CLIENT condition, blocks itself 
 *   on the BLOCKED_IO condition, and deschedules. For a SEND, the "tx_buf" kernel buffer 
 *   is dynamicaly allocated by the client thread, that copies the payload from the user 
 *   buffer to this kernel buffer, that is used as retransmission buffer, when required.
 * - A command is valid for the TX server when the socket descriptor "tx_valid" is true.
 *   For a SEND command, the "tx_valid" is reset by the NIC_TX server when the last byte has 
 *   been sent, but the TX client thread is unblocked by the NIC_RX server thread only when
 *   the last byte has been acknowledged, or to report an error.
 *   For the CONNECT, ACCEPT and CLOSE commands, the "tx_valid" is reset by the NIC_TX server
 *   when the first segment of the handshake has been sent, but the TX client thread is 
 *   unblocked by the NIC_RX server thread only when the handshake is actually completed.
 *   The TX server thread is acting as a multiplexer. It scans the list of attached sockets,
 *   to sequencially handle the valid commands: one UDP packet or TCP segment per iteration.
 *   The TX server blocks and deschedules on the BLOCKED_CLIENT condition when there is
 *   no more valid TX command or R2T request registered in any socket. It is unblocked 
 *   from BLOCKED_CLIENT by a client thread registering a TX command, or by the RX server
 *   thread registering a R2T request. The TX server thread signals an error to the TX client
 *   thread using the "tx_error" field in socket descriptor.
 *   When "tx_valid" or "r2t_valid" are true, the TX server thread build and send an UDP
 *   packet or TCP segment. A single SEND command can require a large number of TCP
 *   segments to move a big data buffer. 
 *   This TX server thread blocks and deschedules on the BLOCKED_ISR condition when there 
 *   the NIC_RX queue is full . It is unblocked by the hardware NIC_TX_ISR. 
 * - In order to detect and report error for multiple simultaneous TX accesses to the same
 *   socket, the client thread makes a double check before posting a new TX command :
 *   the "tx_valid" field must be false, and the "tx_client" field must be XPTR_NULL.
 *   The "tx_valid" field is reset by the TX server thread, and the "tx_client"
 *   field is reset by the TX client thread itself, when it resumes after a TX command. 
 *   . For a SEND command on an UDP socket, the TX server thread reset "tx_valid" and
 *     unblocks the TX client thread as soon as the last data byte has been sent.
 *   . For a SEND command on a TCP socket, the TX server thread reset "tx_valid" when the 
 *     last data byte has been sent, but the TX client thread is unblocked by the TX server
 *     only when the last data byte has been acknowledged by the remote socket. 
 *   . For the CONNECT or ACCEPT commands, the "tx_valid" flag is reset and the TX client
 *     thread is unblocked by the RX server thread only when the command is completed, 
 *     and the local TCP socket is actually in the ESTAB state.
 *   . For a CLOSE command, the "tx_valid" flag is reset, and the TX client thread is
 *     unblocked by the RX server thread only when the remote socket is disconnected.
 *
 * 2) RX stream
 *
 * - The internal API between the RX client thread and the RX server thread defines two 
 *   command types stored in the rx_cmd variable of the socket descriptor:
 *   . SOCKET_RX_ACCEPT : TCP server request a connection request from CRQ queue.
 *   . SOCKET_RX_RECV   : local (UDP/TCP) socket expect data from a remote (UDP/TCP).
 *   For the RECV command the communication is done through the "rx_buf" buffer,
 *   attached to the socket, and handled as a single-writer / single reader-FIFO.
 *   For the ACCEPT command the communication is done through the CRQ buffer, attached
 *   to the socket, and handled as a single-writer / single reader-FIFO.
 *   These two commands are blocking for the RX client thread as long as the buffer is
 *   empty. The client thread set the socket "rx_valid" field, reset the "rx_error" field,
 *   registers itself in the "rx_client" field, and  blocks on the BLOCKED_IO condition.
 * - The RX server thread is acting as a demultiplexor: it handle one received TCP segment,
 *   or UDP packet per iteration in the loop on the NIC_RX queue, and moves the data to
 *   the relevant buffer of the socket matching the packet. It discard packets that don't
 *   match a registered socket. When a client thread is registered in the socket descriptor,
 *   the RX server thread reset the "rx_valid" field and unblocks the RX client thread from
 *   the BLOCKED_IO condition as soon as there is data available in the "rx_buf". 
 *   This RX server thread blocks and deschedules on the BLOCKED_ISR condition when there 
 *   is no more packets in the NIC_RX queue. It is unblocked by the hardware NIC_RX_ISR. 
 * - In order to detect and report error for multiple simultaneous RX accesses to the same
 *   socket, the RX client thread makes a double check before posting a new RX command :
 *   the "rx_valid" field must be false, and the "rx_client" field must be XPTR_NULL.
 *   The "rx_valid" field is reset by the RX server thread, and the "rx_client"
 *   field is reset by the RX client thread itself, when it resumes after an RX command. 
 *
 * 3) R2T queue
 *
 * To implement the TCP "3 steps handshake" protocol for connection or to send RST,
 * the RX server thread can directly request the associated TX server thread to send 
 * control packets in  the TX stream, using a dedicate R2T (RX to TX) FIFO stored in 
 * the socket descriptor. Each R2T request occupy one byte in this R2T queue.
 *
 * 4) CRQ queue
 *
 * The remote CONNECT requests received by a TCP socket (SYN segments) are stored in a
 * dedicated CRQ FIFO stored in the local socket descriptor. These requests are consumed
 * by the local client thread executing an ACCEPT.
 * Each CRQ request occupy sizeof(connect_request_t) bytes in this CRQ queue.
 * The connect_request_t structure containing the request arguments is defined below.
 *
 * Note : the socket domains and types are defined in the "shared_socket.h" file.
 ****************************************************************************************/

/***************************************************************************************** 
 * This enum defines the set of commands that can be registered in the socket
 * by the TX & RX client threads to be executed by the NIC_TX & NIC_TX server threads.
 ****************************************************************************************/
typedef enum socket_cmd_type_e
{
    CMD_TX_CONNECT      = 20,         /*! request a SYN segment     (TCP only)          */
    CMD_TX_ACCEPT       = 21,         /*! request a SYN-ACK segment (TCP only)          */
    CMD_TX_CLOSE        = 22,         /*! request a RST segment     (TCP only)          */
    CMD_TX_SEND         = 23,         /*! request to send data      (TCP or UDP)        */

    CMD_RX_ACCEPT       = 30,         /*! wait request from CRQ     (TCP only)          */ 
    CMD_RX_RECV         = 31,         /*! wait DATA from rx_buf     (TCP or UDP)        */
}
socket_cmd_type_t;

/***************************************************************************************** 
 * This enum defines the set of command status that can be returned by the NIC_RX and
 * NIC_TX server threads to the TX & RX client threads.
 * The success must be signaled by the null value / the various failure cases are
 * signaled by a non-null value.
 ****************************************************************************************/
typedef enum socket_cmd_sts_e
{
    CMD_STS_SUCCESS     =  0,
    CMD_STS_EOF         =  1,
    CMD_STS_RST         =  2,
    CMD_STS_BADACK      =  3,
    CMD_STS_BADSTATE    =  4,
    CMD_STS_BADCMD      =  5,
}
socket_cmd_sts_t;

/***************************************************************************************** 
 * This enum defines the set of tates for an UDP socket.
 ****************************************************************************************/
typedef enum udp_socket_state_e
{
    UDP_STATE_UNBOUND    = 0x00,
    UDP_STATE_BOUND      = 0x01,
    UDP_STATE_ESTAB      = 0x02,
}
udp_socket_state_t;

/***************************************************************************************** 
 * This enum defines the set of tates for an TCP socket.
 ****************************************************************************************/
typedef enum tcp_socket_state_e
{
    TCP_STATE_UNBOUND    = 0x10,
    TCP_STATE_BOUND      = 0x11,
    TCP_STATE_LISTEN     = 0x12,
    TCP_STATE_SYN_SENT   = 0x13,
    TCP_STATE_SYN_RCVD   = 0x14,
    TCP_STATE_ESTAB      = 0x15,
    TCP_STATE_FIN_WAIT1  = 0x16,
    TCP_STATE_FIN_WAIT2  = 0x17,
    TCP_STATE_CLOSING    = 0x18,
    TCP_STATE_TIME_WAIT  = 0x19,
    TCP_STATE_CLOSE_WAIT = 0x1A,
    TCP_STATE_LAST_ACK   = 0x1B,
    TCP_STATE_CLOSED     = 0x1C,
}
tcp_socket_state_t;

/***************************************************************************************** 
 * This structure defines one connection request, registered in the CRQ queue.
 ****************************************************************************************/
typedef struct connect_request_s
{
    uint32_t          addr;          /* requesting socket IP address                   */
    uint32_t          port;          /* requesting socket port number                  */
    uint32_t          iss;           /* requesting socket initial sequence number      */
    uint32_t          window;        /* requesting socket receive window               */
}         
connect_request_t;

/***************************************************************************************** 
 * This structure defines the socket descriptor.
 ****************************************************************************************/
typedef struct socket_s
{
    remote_queuelock_t lock;         /*! lock protecting socket state                  */
    pid_t              pid;          /*! owner process identifier                      */
    uint32_t           fdid;         /*! associated file descriptor index              */
    uint32_t           domain;       /*! domain : AF_LOCAL / AF_INET                   */
    uint32_t           type;         /*! type : SOCK_DGRAM / SOCK_STREAM               */
    uint32_t           state;        /*! socket state (see above)                      */
    uint32_t           local_addr;   /*! local  socket IP address                      */
    uint32_t           remote_addr;  /*! remote socket IP address                      */
    uint32_t           local_port;   /*! local  socket port number                     */
    uint32_t           remote_port;  /*! remote socket port number                     */
    uint32_t           nic_channel;  /*! derived from (remote_addr,remote_port)        */

    xlist_entry_t      tx_list;      /*! all sockets attached to same NIC_TX channel   */
    xptr_t             tx_client;    /*! extended pointer on current TX client thread  */
    bool_t             tx_valid;     /*! TX command valid                              */
    socket_cmd_type_t  tx_cmd;       /*! TX command (CONNECT / ACCEPT / SEND / CLOSE)  */
    uint32_t           tx_sts;       /*! signal a TX command success / failure         */
    uint8_t         *  tx_buf;       /*! pointer on TX data buffer in kernel space     */
    uint32_t           tx_len;       /*! number of data bytes for a SEND command       */
    uint32_t           tx_todo;      /*! number of bytes not yet sent                  */
    xlist_entry_t      tx_temp;      /*! temporary list of sockets (root in TX chdev)  */

    xlist_entry_t      rx_list;      /*! all sockets attached to same NIC_RX channel   */
    xptr_t             rx_client;    /*! extended pointer on current RX client thread  */
    bool_t             rx_valid;     /*! RX command valid                              */
    socket_cmd_type_t  rx_cmd;       /*! RX command ( ACCEPT / RECV )                  */
    uint32_t           rx_sts;       /*! signal a RX command success / failure         */
    remote_buf_t       rx_buf;       /*! embedded receive buffer descriptor            */

    remote_buf_t       r2tq;         /*! RX_to_TX requests queue descriptor            */
    remote_buf_t       crqq;         /*! connection requests queue descriptor          */

    /* the following fields defines the TCB (only used for a TCP connection)           */

    uint32_t           tx_nxt;       /*! next byte to send in TX_data stream           */
    uint32_t           tx_wnd;       /*! number of acceptable bytes in TX_data stream  */
    uint32_t           tx_una;       /*! first unack byte in TX_data stream            */
    uint32_t           rx_nxt;       /*! next expected byte in RX_data stream          */
    uint32_t           rx_wnd;       /*! number of acceptable bytes in RX_data stream  */
    uint32_t           rx_irs;       /*! initial sequence number in RX_data stream     */
}
socket_t;

/****************************************************************************************
 * This function returns a printable string for a socket domain.
 ****************************************************************************************
 * domain   :  AF_INET / AF_LOCAL 
 ***************************************************************************************/
char * socket_domain_str( uint32_t domain );

/****************************************************************************************
 * This function returns a printable string for a socket type.
 ****************************************************************************************
 * type   :  SOCK_DGRAM / SOCK_STREAM
 ***************************************************************************************/
char * socket_type_str( uint32_t type );

/****************************************************************************************
 * This function returns a printable string for an UDP or TCP socket state.
 ****************************************************************************************
 * state  :  UDP_STATE_*** / TCP_STATE*** 
 ***************************************************************************************/
char * socket_state_str( uint32_t state );

/****************************************************************************************
 * This function returns a printable string for a command type.
 ****************************************************************************************
 * type  :  command type 
 ***************************************************************************************/
char * socket_cmd_type_str( uint32_t type );

/****************************************************************************************
 * This function returns a printable string for a command status.
 ****************************************************************************************
 * sts   : command status.
 ***************************************************************************************/
char * socket_cmd_sts_str( uint32_t sts );



/****************************************************************************************
 *      Functions used by the NIC_TX and NIC_RX server threads.
 ***************************************************************************************/

/****************************************************************************************
 * This function is called by the dev_nic_rx_handle_tcp() function, executed by the
 * NIC_RX[channel] server thread, to register a R2T request defined by the <flags>
 * argument in the socket R2T queue, specified by the <queue_xp> argument.
 * This function unblocks the NIC_TX[channel] server thread, identified by the <channel>
 * argumentfrom the THREAD_BLOCKED_CLIENT condition.
 ****************************************************************************************
 * @ queue_xp   : [in] extended pointer on the R2T qeue descriptor.
 * @ flags      : [in] flags to be set in the TCP segment.
 * @ channel    : [in] NIC channel (both TX & RX).
 ***************************************************************************************/
void socket_put_r2t_request( xptr_t    queue_xp,
                             uint32_t  flags,
                             uint32_t  channel );
  
/****************************************************************************************
 * This function is called by the dev_nic_rx_handle_tcp() function to register
 * a client connection request, defined by the <remote_addr>, <remote_port>,
 * <remote_iss>, and <remote_window> arguments, * in the CRQ queue, specified 
 * by the <queue_xp> argument.
 ****************************************************************************************
 * @ queue_xp      : [in] extended pointer on the CRQ qeue descriptor.
 * @ remote_addr   : [in] remote socket IP address.
 * @ remote_port   : [in] remote socket port.
 * @ remote_iss    : [in] remote socket initial sequence number.
 * @ remote_window : [in] remote socket receive window
 * @ return 0 if success / return -1 if queue full.
 ***************************************************************************************/
error_t socket_put_crq_request( xptr_t    queue_xp,
                                uint32_t  remote_addr,
                                uint32_t  remote_port,
                                uint32_t  remote_iss,
                                uint32_t  remote_window );

/****************************************************************************************
 * This function is called by the socket_accept() function to extract a connection
 * request from a CRQ queue, specified by the <queue_xp> argument, to the buffers
 * defined by <remote_addr>, <remote_port>, <remote_iss>, and <remote_window>.
 *****************************************************************************************
 * @ queue_xp      : [in]  extended pointer on the CRQ qeue descriptor.
 * @ remote_addr   : [out] buffer for remote socket IP address.
 * @ remote_port   : [out] buffer for remote socket port.
 * @ remote_iss    : [out] buffer for remote socket initial sequence number.
 * @ remote_window : [out] buffer for remote socket receive window
 * @ return 0 if success / return -1 if queue empty.
 ***************************************************************************************/
error_t socket_get_crq_request( xptr_t     queue_xp,
                                uint32_t * remote_addr,
                                uint32_t * remote_port,
                                uint32_t * remote_iss,
                                uint32_t * remote_window );

/****************************************************************************************
 * This blocking function diplays the socket state (including the TCB).
 ****************************************************************************************
 * @ socket_xp     : [in] extended pointer on socket descriptor.
 $ @ string        : [in] name of calling function.
 ***************************************************************************************/
void socket_display( xptr_t         socket_xp,
                     const char   * func_str );



/****************************************************************************************
 *      Functions implementing the socket related system calls
 ***************************************************************************************/

/****************************************************************************************
 * This function implements the socket() syscall.
 * This function allocates and intializes in the calling thread cluster:
 * - a new socket descriptor, defined by the <domain> and <type> arguments,
 * - a new file descriptor, associated to this socket,
 * It registers the file descriptor in the reference process fd_array[],
 * set the socket state to UNBOUND, and returns the <fdid> value.
 ****************************************************************************************
 * @ domain  : [in] socket protocol family (AF_UNIX / AF_INET)
 * @ type    : [in] socket type (SOCK_DGRAM / SOCK_STREAM).
 * @ return a file descriptor <fdid> if success / return -1 if failure.
 ***************************************************************************************/
int socket_build( uint32_t   domain,
                  uint32_t   type );

/****************************************************************************************
 * This function implements the bind() syscall.
 * It assigns an IP address, defined by the <local_addr> argument, and a port number,
 * defined by the <local_port> argument to an unnamed local socket, identified by the
 * <fdid> argument, and set the socket state to BOUND. It applies to UDP or TCP sockets.
 * It does not require any service from the NIC_TX and NIC_RX server threads.
 * It can be called by a thread running in any cluster.
 ****************************************************************************************
 * @ fdid         : [in] file descriptor index identifying the socket.
 * @ local_addr   : [in] local IP address.
 * @ local_port   : [in] local port.
 * @ return 0 if success / return -1 if failure.
 ***************************************************************************************/
int socket_bind( uint32_t  fdid,
                 uint32_t  addr,
                 uint16_t  port );

/****************************************************************************************
 * This function implements the listen() syscall().
 * It is called by a (local) server process to specify the max size of the CRQ queue
 * for a socket identified by the <fdid> argument, that expect connection requests
 * from one or several (remote) client processes. The selected socket CRQ is supposed
 * to register all connections requests, whatever the client IP address and port values. 
 * 
 * This function applies only to a TCP socket, that must be in the BOUND state.
 * The socket is set to the LISTEN state.
 * It does not require any service from the NIC_TX and NIC_RX server threads.
 * It can be called by a thread running in any cluster.
 ****************************************************************************************
 * Implementation notes : 
 * The number N of channels available in the NIC contrôler can be larger than 1.
 * Depending on the remote client IP address and port, the  connection request can be
 * received by any NIC_RX[k] server thread. To find the relevant listening socket, each
 * NIC_RX[k] server thread must be able to scan the set of all listening sockets.
 * Therefore a list of listening sockets is implemented as a dedicated xlist, rooted in
 * the NIC_RX[0] chdev extension, and using the listening socket <rx_list> field,
 * because a listening socket is never used to move data.  
 ****************************************************************************************
 * @ fdid      : [in] file descriptor index identifying the local server socket.
 * @ crq_depth : [in] depth of CRQ queue of pending connection requests.
 * @ return 0 if success / return -1 if failure
 ***************************************************************************************/
int socket_listen( uint32_t fdid,
                   uint32_t crq_depth );

/****************************************************************************************
 * This blocking function implements the accept() syscall().
 * It applies only to TCP sockets in the LISTEN state.
 * It is executed by a server process, waiting for one (or several) client process(es)
 * requesting a connection on a listening socket identified by the <fdid> argument.
 * This socket must have been previouly created with socket(), bound to a local address
 * with bind(), and listening for connections after a listen(). It  blocks on the <IO>
 * condition if the CRQ is empty. Otherwise, it get a pending connection request from 
 * the listening socket CRQ queue, and creates a new socket with the same properties 
 * as the listening socket, allocating a new file descriptor for this new socket. 
 * It computes the nic_channel index [k] from <remote_addr> and <remote_port> values,
 * and initializes "remote_addr","remote_port", "nic_channel" in local socket.
 * It returns the new socket fdid as well as the remote IP address
 * and port, but only when the new socket is set to the ESTAB state. The new socket 
 * cannot accept connections, but the listening socket keeps open for new connections.  
 ****************************************************************************************
 * Implementation Note: 
 * This blocking function contains two blocking conditions because it requests services
 * to both the NIC_RX server thread, and he NIC_TX server thread.
 * It can be split in five steps:
 * 1) It makes several checkings on the listening socket domain, type, and state.
 * 2) If the socket CRQ queue is empty, the function makes an SOCKET_RX_ACCEPT command
 *    to the NIC_RX server thread, waiting registration of a connection request in the
 *    CRQ queue. Then it blocks on the <IO> condition and deschedules. It is unblocked 
 *    by the NIC_RX server thread receiving a valid TCP SYN segment.
 * 3) When it found a pending request, it creates a new socket with the same properties
 *    as the listening socket, and a new file descriptor for this socket. It initializes
 *    the new socket descriptor using the values in the registered connect_request_t
 *    structure, and set this new socket to the SYN_RECV state.
 * 4) Then it makes a SOCKET_TX_command to the NIC_TX thread, requesting a TCP SYN_ACK 
 *    segment to the remote socket. Then, it blocks on <IO> condition and dechedules.
 *    It is unblocked by the NIC_RX server thread when this SYN_ACK is acknowledged,
 *    and the new socket is set in ESTAB state (by the NIC_RX server).
 * 5) Finally, it returns the new socket fdid, and registers, in the <address> and
 *    <port> arguments, the remote client IP address & port. 
 ****************************************************************************************
 * @ fdid         : [in] file descriptor index identifying the listening socket.
 * @ address      : [out] server IP address.
 * @ port         : [out] server port address length in bytes.
 * @ return the new socket <fdid> if success / return -1 if failure
 ***************************************************************************************/
int socket_accept( uint32_t   fdid,
                   uint32_t * address,
                   uint16_t * port );

/****************************************************************************************
 * This blocking function implements the connect() syscall.
 * It is used by a client process to connect a local socket identified by 
 * the <fdid> argument, to a remote socket identified by the <remote_addr> and 
 * <remote_port> arguments. It can be used for both  UDP and TCP sockets. 
 * It computes the nic_channel index [k] from <remote_addr> and <remote_port> values,
 * and initializes "remote_addr","remote_port", "nic_channel" in local socket.
 * It registers the socket in the lists of sockets rooted in the NIC_RX[k] & NIC_TX[k]
 * chdevs. It can be called by a thread running in any cluster.
 * It returns only when the local socket is in the ESTAB state, or to report an error.
 ****************************************************************************************
 * Implementation Note:
 * - For a TCP socket, it updates the "remote_addr", "remote_port", "nic_channel" fields
 *   in the socket descriptor defined by the <fdid> argument, and register this socket,
 *   in the lists of sockets attached to the NIC_TX[k] and NIC_RX[k] chdevs.
 *   Then, it builds a TX_CONNECT command to the NIC_TX server thread to send a SYN to
 *   the remote socket, unblocks the NIC_TX server thread from the <CLIENT> condition,
 *   blocks itself on <IO> condition and deschedules. It is unblocked by the NIC_RX
 *   server thread when this thread receive the expected SYN-ACK, and the local socket
 *   has been set to the ESTAB state, or when an error is reported in "tx_error" field. 
 * - For an UDP socket, it simply updates "remote_addr", "remote_port", "nic_channel"
 *   in the socket descriptor defined by the <fdid> argument, and register this socket
 *   in the lists of sockets attached to the NIC_TX[k] and NIC_RX[k] chdevs.
 *   Then, it set the socket to the ESTAB state, or returns an error without blocking.
 ****************************************************************************************
 * @ fdid          : [in] file descriptor index identifying the socket.
 * @ remote_addr   : [in] remote IP address.
 * @ remote_port   : [in] remote port.
 * @ return 0 if success / return -1 if failure.
 ***************************************************************************************/
int socket_connect( uint32_t  fdid,
                    uint32_t  remote_addr,
                    uint16_t  remote_port );

/****************************************************************************************
 * This blocking function implements the send() syscall.
 * It is used to send data stored in the user buffer, identified the <u_buf> and <length>
 * arguments, to a connected (TCP or UDP) socket, identified by the <fdid> argument.
 * The work is actually done by the NIC_TX server thread, and the synchronisation
 * between the client and the server threads uses the "rx_valid" set/reset flip-flop:
 * The client thread registers itself in the socket descriptor, registers in the queue 
 * rooted in the NIC_TX[index] chdev, set "rx_valid", unblocks the server thread, and 
 * finally blocks on THREAD_BLOCKED_IO, and deschedules. 
 * When the TX server thread completes the command (all data has been sent for an UDP
 * socket, or acknowledged for a TCP socket), the server thread reset "rx_valid" and
 * unblocks the client thread.
 * This function can be called by a thread running in any cluster.
 * WARNING : This implementation does not support several concurent SEND/SENDTO commands
 * on the same socket, as only one TX thread can register in a given socket.
 ****************************************************************************************
 * @ fdid      : [in] file descriptor index identifying the socket.
 * @ u_buf     : [in] pointer on buffer containing packet in user space.
 * @ length    : [in] packet size in bytes.
 * @ return number of sent bytes if success / return -1 if failure.
 ***************************************************************************************/
int socket_send( uint32_t    fdid,
                 uint8_t   * u_buf,
                 uint32_t    length );

/****************************************************************************************
 * This blocking function implements the recv() syscall.
 * It is used to receive data that has been stored by the NIC_RX server thread in the
 * rx_buf of a connected (TCP or UDP) socket, identified by the <fdid> argument.
 * The synchronisation between the client and the server threads uses the "rx_valid"
 * set/reset flip-flop: If "rx_valid" is set, the client simply moves the available
 * data from the "rx_buf" to the user buffer identified by the <u_buf> and <length>
 * arguments, and reset the "rx_valid" flip_flop. If "rx_valid" is not set, the client
 * thread register itself in the socket descriptor, registers in the clients queue rooted
 * in the NIC_RX[index] chdev, and finally blocks on THREAD_BLOCKED_IO, and deschedules. 
 * The client thread is re-activated by the RX server, that set the "rx_valid" flip-flop 
 * as soon as data is available in the "rx_buf". The number of bytes actually transfered
 * can be less than the user buffer size.
 * This  function can be called by a thread running in any cluster.
 * WARNING : This implementation does not support several concurent RECV/RECVFROM 
 * commands on the same socket, as only one RX thread can register in a given socket.
 ****************************************************************************************
 * @ fdid      : [in] file descriptor index identifying the socket.
 * @ u_buf     : [in] pointer on buffer in user space.
 * @ length    : [in] buffer size in bytes.
 * @ return number of received bytes if success / return -1 if failure.
 ***************************************************************************************/
int socket_recv( uint32_t    fdid,
                 uint8_t   * u_buf,
                 uint32_t    length );

/****************************************************************************************
 * This blocking function implements the close() syscall for a socket.
 * - For a UDP socket, it simply calls the static socket_destroy() function to release 
 *   all structures associated to the local socket, including the file descriptor.
 * - For a TCP socket, it makes a CLOSE command to NIC_TX, and blocks on the <IO>
 *   condition. The close TCP hanshake is done by the NIC_TX and NIC_RX threads.
 *   It is unblocked when the socket is in CLOSED state, or when an error is reported.
 *   Finally, it calls the static socket_destroy() function to release all structures 
 *   associated to the local socket, including the file descriptor.
 ****************************************************************************************
 * @ file_xp     : [in] extended pointer on file descriptor.
 * @ fdid        : [in] file descriptor index identifying the socket.
 * @ return 0 if success / return -1 if failure.
 ***************************************************************************************/
int socket_close( xptr_t     file_xp,
                  uint32_t   fdid );


#endif	/* _KSOCKET_H_ */



