/*
 * kmem.c - kernel memory allocator implementation.
 *
 * Authors  Alain Greiner     (2016,2017,2018,2019,2020)
 *
 * Copyright (c) UPMC Sorbonne Universites
 *
 * This file is part of ALMOS-MKH.
 *
 * ALMOS-MKH is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2.0 of the License.
 *
 * ALMOS-MKH is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with ALMOS-MKH; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */

#include <kernel_config.h>
#include <hal_kernel_types.h>
#include <hal_special.h>
#include <printk.h>
#include <cluster.h>
#include <thread.h>
#include <memcpy.h>
#include <ppm.h>
#include <kcm.h>
#include <page.h>
#include <kmem.h>

///////////////////////////////////
void * kmem_alloc( uint32_t  order,
                   uint32_t  flags )
{

#if DEBUG_KMEM || DEBUG_KMEM_ERROR
thread_t * this  = CURRENT_THREAD;
uint32_t   cycle = (uint32_t)hal_get_cycles();
#endif

	if( order >= CONFIG_PPM_PAGE_ORDER )     // use PPM
	{
		// allocate memory from PPM
		page_t * page = (void *)ppm_alloc_pages( order - CONFIG_PPM_PAGE_ORDER );

		if( page == NULL )
		{

#if DEBUG_KMEM_ERROR
if (DEBUG_KMEM_ERROR < cycle)
printk("\n[ERROR] in %s : thread[%x,%x] failed for PPM / order %d / cluster %x / cycle %d\n",
__FUNCTION__ , this->process->pid , this->trdid , order , local_cxy , cycle );
#endif
			return NULL;
		}

		// reset page if requested
		if( flags & AF_ZERO ) page_zero( page );

        // get pointer on buffer from the page descriptor
        xptr_t page_xp = XPTR( local_cxy , page );
        void * ptr     = GET_PTR( ppm_page2base( page_xp ) );

#if DEBUG_KMEM
if( (DEBUG_KMEM < cycle) && (DEBUG_KMEM_CXY == local_cxy) && (DEBUG_KMEM_ORDER == order) )
printk("\n[%s] thread[%x,%x] from PPM / order %d / ppn %x / cxy %x / cycle %d\n",
__FUNCTION__, this->process->pid, this->trdid,
order, ppm_page2ppn(XPTR(local_cxy,ptr)), local_cxy, cycle );
#endif
        return ptr;
	}
	else                                     // use KCM 
	{
		// allocate memory from KCM
		void * ptr = kcm_alloc( order );

		if( ptr == NULL )
		{

#if DEBUG_KMEM_ERROR
if (DEBUG_KMEM_ERROR < cycle)
printk("\n[ERROR] in %s : thread[%x,%x] failed for KCM / order %d / cluster %x / cycle %d\n",
__FUNCTION__ , this->process->pid , this->trdid , order , local_cxy , cycle );
#endif
			return NULL;
		}

		// reset memory if requested
		if( flags & AF_ZERO ) memset( ptr , 0 , 1<<order );

#if DEBUG_KMEM
if( (DEBUG_KMEM < cycle) && (DEBUG_KMEM_CXY == local_cxy) && (DEBUG_KMEM_ORDER == order) )
printk("\n[%s] thread [%x,%x] from KCM / order %d / base %x / cxy %x / cycle %d\n",
__FUNCTION__, this->process->pid, this->trdid,
order, ptr, local_cxy, cycle ); 
#endif
        return ptr;
	}
}  // end kmem_alloc()

//////////////////////////////
void kmem_free( void    * ptr,
                uint32_t  order )
{
	if( order >= CONFIG_PPM_PAGE_ORDER )     // use PPM
	{
        page_t * page = GET_PTR( ppm_base2page( XPTR( local_cxy , ptr ) ) );

        ppm_free_pages( page );
    }
	else                                     // use KCM 
    {
        kcm_free( ptr , order );
	}
}  // end kmem_free()



////////////////////////////////////////
void * kmem_remote_alloc( cxy_t     cxy,
                          uint32_t  order,
                          uint32_t  flags )
{

#if DEBUG_KMEM || DEBUG_KMEM_ERROR
thread_t * this = CURRENT_THREAD;
uint32_t   cycle = (uint32_t)hal_get_cycles();
#endif

	if( order >= CONFIG_PPM_PAGE_ORDER )     // use PPM
	{
		// allocate memory from PPM
		xptr_t page_xp = ppm_remote_alloc_pages( cxy , order - CONFIG_PPM_PAGE_ORDER );

		if( page_xp == XPTR_NULL )
		{

#if DEBUG_KMEM_ERROR
if( DEBUG_KMEM_ERROR < cycle )
printk("\n[ERROR] in %s : thread[%x,%x] failed for PPM / order %d / cluster %x / cycle %d\n",
__FUNCTION__ , this->process->pid , this->trdid , order , cxy , cycle );
#endif
			return NULL;
		}

        // get extended pointer on remote buffer
        xptr_t base_xp = ppm_page2base( page_xp );

		// reset memory if requested
		if( flags & AF_ZERO ) hal_remote_memset( base_xp , 0 , 1<<order );

#if DEBUG_KMEM
if( (DEBUG_KMEM < cycle) && (DEBUG_KMEM_CXY == local_cxy) && (DEBUG_KMEM_ORDER == order) )
printk("\n[%s] thread[%x,%x] from PPM / order %d / ppn %x / cxy %x / cycle %d\n",
__FUNCTION__, this->process->pid, this->trdid,
order, ppm_page2ppn( page_xp ), cxy, cycle );
#endif
        return GET_PTR( base_xp );
	}
	else                                     // use KCM 
	{
		// allocate memory from KCM
		void * ptr = kcm_remote_alloc( cxy , order );

		if( ptr == NULL )
		{

#if DEBUG_KMEM_ERROR
if( DEBUG_KMEM_ERROR < cycle )
printk("\n[ERROR] in %s : thread[%x,%x] failed for KCM / order %d / cluster %x / cycle %d\n",
__FUNCTION__ , this->process->pid , this->trdid , order , cxy , cycle );
#endif
			return NULL;
		}

		// reset memory if requested
		if( flags & AF_ZERO )  hal_remote_memset( XPTR( cxy , ptr ) , 0 , 1<<order );

#if DEBUG_KMEM
if( (DEBUG_KMEM < cycle) && (DEBUG_KMEM_CXY == local_cxy) && (DEBUG_KMEM_ORDER == order) )
printk("\n[%s] thread [%x,%x] from KCM / order %d / base %x / cxy %x / cycle %d\n",
__FUNCTION__, this->process->pid, this->trdid,
order, ptr, cxy, cycle );
#endif
        return ptr;
	}
}  // kmem_remote_malloc()

/////////////////////////////////////
void kmem_remote_free( cxy_t     cxy,
                       void    * ptr,
                       uint32_t  order )
{
	if( order >= CONFIG_PPM_PAGE_ORDER )     // use PPM
	{
        page_t * page = GET_PTR( ppm_base2page( XPTR( cxy , ptr ) ) );

        ppm_remote_free_pages( cxy , page );
    }
	else                                     // use KCM 
    {
        kcm_remote_free( cxy , ptr , order );
	}
}  // end kmem_remote_free()



