/*
 * sys_get_cycle.c - get calling core cycles count.
 * 
 * Author    Alain Greiner (2016,2017,2018)
 *  
 * Copyright (c) UPMC Sorbonne Universites
 *
 * This file is part of ALMOS-MKH.
 *
 * ALMOS-MKH is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2.0 of the License.
 *
 * ALMOS-MKH is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with ALMOS-MKH; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */

#include <hal_kernel_types.h>
#include <hal_uspace.h>
#include <hal_vmm.h>
#include <hal_special.h>
#include <errno.h>
#include <core.h>
#include <thread.h>
#include <process.h>
#include <vmm.h>
#include <printk.h>

#include <syscalls.h>

//////////////////////////////////////
int sys_get_cycle ( uint64_t * cycle )
{
	error_t     error;
    vseg_t    * vseg;
	uint64_t    k_cycle;

    thread_t  * this    = CURRENT_THREAD;
    process_t * process = this->process;

#if (DEBUG_SYS_GET_CYCLE || CONFIG_INSTRUMENTATION_SYSCALLS)
uint64_t     tm_start = hal_get_cycles();
#endif

    // check buffer in user space
    error = vmm_get_vseg( process , (intptr_t)cycle , &vseg );

	if( error )
	{

#if DEBUG_SYSCALLS_ERROR
printk("\n[ERROR] in %s : user buffer unmapped %x / thread %x / process %x\n",
__FUNCTION__ , (intptr_t)cycle , this->trdid , process->pid );
#endif
        this->errno = EFAULT;
		return -1;
	}

    // call relevant core function
	k_cycle = hal_get_cycles();

    // copy to user space
	hal_copy_to_uspace( cycle,
                        XPTR( local_cxy , &k_cycle ),
                        sizeof(uint64_t) );

#if (DEBUG_SYS_GET_CYCLE || CONFIG_INSTRUMENTATION_SYSCALLS)
uint64_t     tm_end = hal_get_cycles();
#endif

#if DEBUG_SYS_GET_CYCLE
if( DEBUG_SYS_GET_CYCLE < tm_end )
printk("\n[%s] thread[%x,%x] exit / cycle %d\n",
__FUNCTION__ , process->pid, this->trdid, (uint32_t)tm_end );
#endif

#if CONFIG_INSTRUMENTATION_SYSCALLS
hal_atomic_add( &syscalls_cumul_cost[SYS_GET_CYCLE] , tm_end - tm_start );
hal_atomic_add( &syscalls_occurences[SYS_GET_CYCLE] , 1 );
#endif

	return 0; 

}  // end sys_get_cycle()
