/*
 * rpc.h - RPC (Remote Procedure Call) operations definition.
 *
 * Author  Alain Greiner (2016,2017,2018,2019,2020)
 *
 * Copyright (c) UPMC Sorbonne Universites
 * 
 * This file is part of ALMOS-MKH.
 *
 * ALMOS-MKH is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2.0 of the License.
 *
 * ALMOS-MKH is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with ALMOS-MKH; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */

#ifndef _RPC_H_
#define _RPC_H_

#include <kernel_config.h>
#include <hal_kernel_types.h>
#include <hal_atomic.h>
#include <bits.h>
#include <vseg.h>
#include <remote_fifo.h>

/**** Forward declarations ****/

struct process_s;
struct page_s;
struct vseg_s;
struct exec_info_s;
struct pthread_attr_s;
struct remote_sem_s;
struct user_dir_s;
struct fragment_s;
struct vfs_inode_s;
struct vfs_dentry_s;
struct vfs_file_s;
struct thread_s;
struct mapper_s;


/**********************************************************************************/
/**************  structures for Remote Procedure Calls ****************************/
/**********************************************************************************/

/***********************************************************************************
 * This enum defines all RPC indexes.
 * It must be consistent with the rpc_server[] arrays defined in in the rpc.c file.
 **********************************************************************************/

typedef enum
{
    RPC_UNDEFINED_0               = 0,   //
    RPC_UNDEFINED_1               = 1,   //
    RPC_UNDEFINED_2               = 2,   //      
    RPC_PROCESS_MAKE_FORK         = 3,
    RPC_USER_DIR_CREATE           = 4,
    RPC_USER_DIR_DESTROY          = 5,
    RPC_THREAD_USER_CREATE        = 6,
    RPC_THREAD_KERNEL_CREATE      = 7,
    RPC_UNDEFINED_8               = 8,
    RPC_PROCESS_SIGACTION         = 9,

    RPC_UNDEFINED_10              = 10,   //
    RPC_UNDEFINED_11              = 11,   //
    RPC_UNDEFINED_12              = 12,   // 
    RPC_UNDEFINED_13              = 13,   //
    RPC_UNDEFINED_14              = 14,   //
    RPC_VMM_RESIZE_VSEG           = 15,
    RPC_VMM_REMOVE_VSEG           = 16,
    RPC_VMM_CREATE_VSEG           = 17,
    RPC_VMM_SET_COW               = 18,
    RPC_UNDEFINED_19              = 19,   //

    RPC_MAX_INDEX                 = 20,
}
rpc_index_t;

/***********************************************************************************
 * This defines the prototype of the rpc_server functions,
 * defined by the rpc_server[] array in the rpc.c file. 
 **********************************************************************************/

typedef  void (rpc_server_t) ( xptr_t xp );

/***********************************************************************************
 *  This structure defines the RPC descriptor (100 bytes on a 32bits core)
 **********************************************************************************/

typedef struct rpc_desc_s
{
	rpc_index_t         index;       /*! index of requested RPC service      ( 4) */
	uint32_t          * rsp;         /*! local pointer ond responses counter ( 4) */
    struct thread_s   * thread;      /*! local pointer on client thread      ( 4) */
    uint32_t            lid;         /*! index of core running client thread ( 4) */ 
    bool_t              blocking;    /*! simple RPC mode when true           ( 4) */
    uint64_t            args[10];    /*! input/output arguments buffer       (80) */
} 
rpc_desc_t;

/**********************************************************************************/
/******* Generic functions supporting RPCs : client side **************************/
/**********************************************************************************/

/***********************************************************************************
 * This function is executed by the client thread in the client cluster. 
 * It puts one RPC descriptor defined by the <desc> argument in the remote fifo
 * defined by the <cxy> argument.  It sends an IPI to the server if fifo is empty.
 * The RPC descriptor must be allocated in the caller's stack, and initialised by
 * the caller. It exit with a Panic message if remote fifo is still full after 
 * (CONFIG_RPC_PUT_MAX_ITERATIONS) retries. 
 * - When the RPC <blocking> field is true, this function blocks and deschedule
 *   the client thread. It returns only when the server completes the RPC and
 *   unblocks the client thread.
 * - When the <blocking> field is false, this function returns as soon as the RPC
 *   has been registered in the FIFO, and the client thread must block itself when
 *   all RPCS have been registered in all target clusters.
 ***********************************************************************************
 * @ cxy   : server cluster identifier
 * @ desc  : local pointer on RPC descriptor in client cluster
 **********************************************************************************/
void rpc_send( cxy_t        cxy,    
               rpc_desc_t * desc );



/**********************************************************************************/
/******* Generic functions supporting RPCs : server side **************************/
/**********************************************************************************/

/***********************************************************************************
 * This function contains the infinite loop executed by a RPC server thread,
 * to handle pending RPCs registered in the RPC fifo attached to a given core.
 * In each iteration in this loop, it try to handle one RPC request:
 * - it tries to take the RPC FIFO ownership,
 * - it consumes one request when the FIFO is not empty,
 * - it releases the FIFO ownership,
 * - it execute the requested service,
 * - it unblock and send an IPI to the client thread,
 * - it suicides if the number of RPC threads for this core is to large,
 * - it block on IDLE and deschedule otherwise.  
 **********************************************************************************/
void rpc_server_func( void );

/***********************************************************************************
 * This function is executed in case of illegal RPC index.
 **********************************************************************************/
void __attribute__((noinline)) rpc_undefined( xptr_t xp __attribute__ ((unused)) );



/**********************************************************************************/
/******* Marshalling functions attached to the various RPCs ***********************/
/**********************************************************************************/

/***********************************************************************************
 * [0] undefined
 **********************************************************************************/

/***********************************************************************************
 * [1] undefined
 **********************************************************************************/

/***********************************************************************************
 * [2] undefined
 **********************************************************************************/

/***********************************************************************************
 * [3] The RPC_PROCESS_MAKE_FORK creates a "child" process descriptor, and the
 * associated "child" thread descriptor in a target remote cluster that can be 
 * any cluster.  The child process is initialized from informations found in the 
 * "parent" process descriptor (that must be the parent reference cluster),
 * and from the "parent" thread descriptor that can be in any cluster.
 ***********************************************************************************
 * @ cxy              : server cluster identifier.
 * @ ref_process_xp   : [in]  extended pointer on reference parent process.
 * @ parent_thread_xp : [in]  extended pointer on parent thread.
 * @ child_pid        : [out] child process identifier.
 * @ child_thread_ptr : [out] local pointer on child thread.
 * @ error            : [out]  error status (0 if success).
 **********************************************************************************/
void rpc_process_make_fork_client( cxy_t              cxy,
                                   xptr_t             ref_process_xp,
                                   xptr_t             parent_thread_xp,
                                   pid_t            * child_pid,
                                   struct thread_s ** child_thread_ptr,
                                   error_t          * error );

void rpc_process_make_fork_server( xptr_t xp );

/***********************************************************************************
 * [4] The RPC_USER_DIR_CREATE allows a client thread to create an user_dir_t
 * structure and the associated array of dirents in a remote cluster containing
 * the target directory <inode>. It creates an ANON vseg in the user reference 
 * process VMM identified by the <ref_xp>. This reference cluster cluster can be
 * different from both the client and server clusters.
 * It is called by the sys_opendir() function.
 ***********************************************************************************
 * @ cxy        : server cluster identifier.
 * @ inode      : [in]   local pointer on inode in server cluster.
 * @ ref_xp     : [in]   extended pointer on user reference process descriptor.
 * @ dir        : [out]  local pointer on created user_dir structure.
 **********************************************************************************/
void rpc_user_dir_create_client( cxy_t                 cxy,
                                 struct vfs_inode_s  * inode,
                                 xptr_t                ref_xp,
                                 struct user_dir_s  ** dir );

void rpc_user_dir_create_server( xptr_t xp );

/***********************************************************************************
 * [5] The RPC_USER_DIR_DESTROY allows a client thread to delete an user_dir_t
 * structure and the associated array of dirents in a remote cluster containing
 * the target directory inode. It is called by the sys_closedir() function.
 ***********************************************************************************
 * @ cxy        : server cluster identifier.
 * @ dir        : [in]  local pointer on created user_dir structure.
 * @ ref_xp     : [in]   extended pointer on user reference process descriptor.
 **********************************************************************************/
void rpc_user_dir_destroy_client( cxy_t               cxy,
                                  struct user_dir_s * dir,
                                  xptr_t              ref_xp );

void rpc_user_dir_destroy_server( xptr_t xp );

/***********************************************************************************
 * [6] The RPC_THREAD_USER_CREATE creates an user thread in the server cluster,
 * as specified by the arguments. It returns an extended pointer on the new
 * thread descriptor in server cluster, and an error code.
 * It is called by the sys_thread_create() system call.
 ***********************************************************************************
 * @ cxy       : server cluster identifier.
 * @ attr      : [in]  local pointer on pthread_attr_t in client cluster.
 * @ thread_xp : [out] buffer for thread extended pointer. 
 * @ error     : [out] error status (0 if success).
 **********************************************************************************/
void rpc_thread_user_create_client( cxy_t                   cxy,
                                    pid_t                   pid,
                                    void                  * start_func,
                                    void                  * start_arg,
                                    pthread_attr_t        * attr,
                                    xptr_t                * thread_xp,
                                    error_t               * error );

void rpc_thread_user_create_server( xptr_t xp );

/***********************************************************************************
 * [7] The RPC_THREAD_KERNEL_CREATE creates a kernel thread in the server cluster,
 * as specified by the type, func and args arguments. It returns the local pointer
 * on the thread descriptor in server cluster and an error code.
 * It is used by the dev_init() function to create the device server thread.
 ***********************************************************************************
 * @ cxy       : server cluster identifier.
 * @ type      : [in]  type of kernel thread.
 * @ func      : [in]  local pointer on thread function.
 * @ args      : [in]  local pointer on function arguments.
 * @ thread_xp : [out] pointer on buffer for thread extended pointer. 
 * @ error     : [out] error status (0 if success).
 **********************************************************************************/
void rpc_thread_kernel_create_client( cxy_t     cxy,
                                      uint32_t  type,
                                      void    * func,
                                      void    * args,
                                      xptr_t  * thread_xp,
                                      error_t * error );

void rpc_thread_kernel_create_server( xptr_t xp );

/***********************************************************************************
 * [8] undefined
 **********************************************************************************/

/***********************************************************************************
 * [9] The RPC_PROCESS_SIGACTION allows a client thread to request a remote cluster
 * to execute a given sigaction, defined by the <action_type> for a given process,
 * identified by the <pid> argument.  When this RPC is used in parallel mode,
 * the rpc_process_sigaction_client() function is not used. 
 ***********************************************************************************
 * @ cxy     : server cluster identifier.
 * @ pid     : [in] target process identifier.
 * @ action  : [in] sigaction index.
 **********************************************************************************/
void rpc_process_sigaction_client( cxy_t     cxy,
                                   pid_t     pid,
                                   uint32_t  action );
                             
void rpc_process_sigaction_server( xptr_t xp );

/*********************************************************************************** 
 * [10] undefined
 **********************************************************************************/

/*********************************************************************************** 
 * [11] undefined
 **********************************************************************************/

/*********************************************************************************** 
 * [12] undefined                                                       
 **********************************************************************************/

/*********************************************************************************** 
 * [13] undefined                                       
 **********************************************************************************/

/***********************************************************************************
 * [14] undefined
 **********************************************************************************/

/***********************************************************************************
 * [15] The RPC_VMM_RESIZE_VSEG allows a client thread to request a remote cluster
 * to resize a vseg identified by the <base> argument in a process descriptor
 * identified by the <pid> argument, as defined by the <new_base> and <new_size>
 * arguments. Both the VSL and the GPT are updated in the remote cluster. 
 ***********************************************************************************
 * @ cxy         : server cluster identifier.
 * @ pid         : [in] process identifier.
 * @ base        : [in] vseg base.
 * @ new_base    : [in] new vseg base.
 * @ new_size    : [in] new vseg size.
 **********************************************************************************/
void rpc_vmm_resize_vseg_client( cxy_t              cxy,
                                 pid_t              pid,
                                 intptr_t           base,
                                 intptr_t           new_base,
                                 intptr_t           new_size );
 
void rpc_vmm_resize_vseg_server( xptr_t xp );

/***********************************************************************************
 * [16] The RPC_VMM_REMOVE_VSEG allows a client thread  to request a remote cluster
 * to delete a vseg identified by the <vaddr> argument in a process descriptor
 * identified by the <pid> argument.
 * Both the VSL and the GPT are updated in the remote cluster.
 ***********************************************************************************
 * @ cxy       : server cluster identifier.
 * @ pid       : [in] process identifier.
 * @ base      : [in] vseg base.
 **********************************************************************************/
void rpc_vmm_remove_vseg_client( cxy_t      cxy,
                                 pid_t      pid,
                                 intptr_t   base );
 
void rpc_vmm_remove_vseg_server( xptr_t xp );

/***********************************************************************************
 * [17] The RPC_VMM_CREATE_VSEG allows a client thread to request the remote 
 * reference cluster of a given process to allocate and register in the reference
 * process VMM a new vseg descriptor.
 * On the server side, this RPC uses the vmm_create_vseg() function, and returns
 * to the client the local pointer on the created vseg descriptor.
 ***********************************************************************************
 * @ cxy         : server cluster identifier.
 * @ process     : [in]  local pointer on process descriptor in server.
 * @ type        : [in]  vseg type.
 * @ base        : [in]  base address (unused for dynamically allocated vsegs).
 * @ size        : [in]  number of bytes.
 * @ file_offset : [in]  offset in file (for CODE, DATA, FILE types).
 * @ file_size   : [in]  can be smaller than size for DATA type.
 * @ mapper_xp   : [in]  extended pointer on mapper (for CODE, DATA, FILE types).
 * @ vseg_cxy    : [in]  target cluster for mapping (if not data type).
 * @ vseg        : [out] local pointer on vseg descriptor / NULL if failure.
 **********************************************************************************/
void rpc_vmm_create_vseg_client( cxy_t              cxy,
                                 struct process_s * process,
                                 vseg_type_t        type,
                                 intptr_t           base,
                                 uint32_t           size,
                                 uint32_t           file_offset,
                                 uint32_t           file_size,
                                 xptr_t             mapper_xp,
                                 cxy_t              vseg_cxy,
                                 struct vseg_s   ** vseg );

void rpc_vmm_create_vseg_server( xptr_t xp );

/***********************************************************************************
 * [18] The RPC_VMM_SET_COW allows a client thread to request the remote reference
 * cluster to set the COW flag and reset the WRITABLE flag of all GPT entries for
 * the DATA, MMAP and REMOTE vsegs of process identified by the <process> argument.

 * of a remote scheduler, identified by the <lid> argument.
 ***********************************************************************************
 * @ cxy         : server cluster identifier.
 * @ process     : [in]  local pointer on reference process descriptor.
 **********************************************************************************/
void rpc_vmm_set_cow_client( cxy_t              cxy,
                             struct process_s * process );

void rpc_vmm_set_cow_server( xptr_t xp );

/***********************************************************************************
 * [19] undefined
 **********************************************************************************/


#endif
