/*
 * sys_thread_wakeup.c - unblock indicated thread from the SLEEP condition
 * 
 * Author    Alain Greiner     (2016,2017,2018,2019,2020)
 *
 * Copyright (c) UPMC Sorbonne Universites
 *
 * This file is part of ALMOS-MKH.
 *
 * ALMOS-MKH is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2.0 of the License.
 *
 * ALMOS-MKH is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with ALMOS-MKH; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */

#include <hal_kernel_types.h>
#include <thread.h>
#include <printk.h>
#include <process.h>
#include <errno.h>
#include <syscalls.h>

//////////////////////////////////////
int sys_thread_wakeup( trdid_t trdid )
{
	thread_t  * this    = CURRENT_THREAD;
    process_t * process = this->process;

#if DEBUG_SYS_THREAD_WAKEUP || DEBUG_SYSCALLS_ERROR || CONFIG_INTRUMENTATION_SYSCALLS
cycle_t      tm_start = hal_get_cycles();
#endif

#if DEBUG_SYS_THREAD_WAKEUP
if( DEBUG_SYS_THREAD_WAKEUP < (uint32_t)tm_start )
printk("\n[%s] thread[%x,%x] enter to activate thread %x / cycle %d\n",
__FUNCTION__, process->pid, this->trdid, trdid, (uint32_t)tm_start );
#endif

    // get target thread ltid and cxy
    ltid_t   target_ltid = LTID_FROM_TRDID( trdid );
    cxy_t    target_cxy  = CXY_FROM_TRDID( trdid );

    // check trdid argument
	if( (target_ltid >= CONFIG_THREADS_MAX_PER_CLUSTER) ||
        (cluster_is_active( target_cxy ) == false) )  
	{

#if DEBUG_SYSCALLS_ERROR
if( DEBUG_SYSCALLS_ERROR < (uint32_t)tm_start )
printk("\n[ERROR] in %s : thread[%x,%x] / illegal trdid argument %x\n",
__FUNCTION__, process->pid, this->trdid, trdid );
#endif
		this->errno = EINVAL;
		return -1;
	}

    // get extended pointer on target thread
    xptr_t thread_xp = thread_get_xptr( process->pid , trdid );

    if( thread_xp == XPTR_NULL )
    {

#if DEBUG_SYSCALLS_ERROR
if( DEBUG_SYSCALLS_ERROR < (uint32_t)tm_start )
printk("\n[ERROR] in %s : thread[%x,%x] / cannot find thread[%x,%x]\n",
__FUNCTION__, process->pid, this->trdid, process->pid, trdid );
#endif
        CURRENT_THREAD->errno = EINVAL;
        return -1;
    }

    // get target thread cluster and local pointer
    thread_t * tgt_ptr = GET_PTR( thread_xp );
    cxy_t      tgt_cxy = GET_CXY( thread_xp );

    // get state of the target thread alarm
    bool_t linked = hal_remote_l32( XPTR( tgt_cxy , &tgt_ptr->alarm.linked ) );

    // delete the alarm if active
    if( linked ) alarm_stop( thread_xp );

    // unblock target thread
    thread_unblock( thread_xp , THREAD_BLOCKED_SLEEP );

#if (DEBUG_SYS_THREAD_WAKEUP || CONFIG_INSTRUMENTATION_SYSCALLS)
uint64_t     tm_end = hal_get_cycles();
#endif


#if DEBUG_SYS_THREAD_WAKEUP
if( DEBUG_SYS_THREAD_WAKEUP < tm_end )
printk("\n[%s] thread[%x,%x] exit / thread[%x,%x] activated / cycle %d\n",
__FUNCTION__ , process->pid, this->trdid, process->pid, trdid, (uint32_t)tm_end );
#endif

#if CONFIG_INSTRUMENTATION_SYSCALLS
hal_atomic_add( &syscalls_cumul_cost[SYS_THREAD_WAKEUP] , tm_end - tm_start );
hal_atomic_add( &syscalls_occurences[SYS_THREAD_WAKEUP] , 1 );
#endif

    return 0;

}  // end sys_thread_wakeup()
